<?php

use Awardit\MagentoPsr\Psr3\Logger;
use Awardit\SimpleEvent\Aws\SqsEventListener;
use Awardit\SimpleEvent\{
    EventHandlerInterface,
    EventInterface,
    EventRegistry,
    Metadata,
};
use Awardit\SimpleEvent\Event\{
    ErpProduct,
    ProductPricingPrice,
};
use Aws\Credentials\Credentials;
use Aws\Sqs\SqsClient;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Worker implements EventHandlerInterface
{
    private Awardit_EventListener_Handler_ErpProduct $erpProductHandler;
    private Awardit_EventListener_Handler_ProductPricingPrice $productPricingPriceHandler;
    private LoggerInterface $logger;

    public function __construct()
    {
        $this->logger = new Logger('event-listener');
        $this->erpProductHandler = new Awardit_EventListener_Handler_ErpProduct($this->logger);
        $this->productPricingPriceHandler = new Awardit_EventListener_Handler_ProductPricingPrice($this->logger);
    }

    /**
     * Entry point method for listener.
     */
    public function run(): void
    {
        $credentials = new Credentials(
            Mage::getStoreConfig('awardit_aws/credentials/access_key_id') ?: '',
            Mage::getStoreConfig('awardit_aws/credentials/secret_access_key') ?: '',
            Mage::getStoreConfig('awardit_aws/credentials/session_token') ?: null,
        );
        $config = [
            'credentials'   => $credentials,
            'region'        => Mage::getStoreConfig('awardit_aws/core/region') ?: null,
        ];
        $queueUrl =  Mage::getStoreConfig('awardit_aws/sqs/queue_url');

        $registry = new EventRegistry([
            ErpProduct::class,
            ProductPricingPrice::class,
        ]);
        $sqsClient = new SqsClient($config);
        $listener = new SqsEventListener($this->logger, $sqsClient, $queueUrl, $registry, 10);
        $listener->run($this);
    }

    /**
     * Delegator for incoming events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(EventInterface $event, Metadata $meta): bool
    {
        try {
            switch (get_class($event)) {
                case ErpProduct::class:
                    return $this->erpProductHandler->handle($event);
                case ProductPricingPrice::class:
                    return $this->productPricingPriceHandler->handle($event);
                default:
                    // No-op, consume event
                    return true;
            }
        } catch (Awardit_EventListener_EventException $e) {
            $this->logger->error($e->getMessage(), [
                'exception' => $e,
            ]);
            return $e->isPermanent();
        } catch (Zend_Db_Statement_Exception $e) {
            $this->logger->error("Resetting magento due to: " . $e->getMessage(), [
                'exception' => $e,
            ]);

            // Reset Magento whenever we get a database failure
            Mage::reset();
            sleep(1);

            // Still want to indicate that we failed, to retry again
            throw $e;
        } finally {
            // Collect garbage
            gc_collect_cycles();
        }
    }
}
