<?php

use Awardit\SimpleEvent\Event\ProductPricingPrice;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Handler_ProductPricingPrice
{
    private LoggerInterface $logger;
    private Awardit_EventListener_Helper_Data $helper;

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
        $this->helper = new Awardit_EventListener_Helper_Data();
    }

    /**
     * Handler for ProductPricingPrice events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(ProductPricingPrice $productPrices): bool
    {
        $store = $this->getStore($productPrices->priceList);
        $product = $this->getProduct($productPrices->getEntityId(), $store);
        $currency = $store->getBaseCurrencyCode();

        $product->setWebsiteIds(array_merge($product->getWebsiteIds(), [$store->getWebsiteId()]));
        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingLog->loadLatest($product->getEntityId(), $store->getStoreId(), 'ProductPricingPrice');
        if ($pricingLog->getRevision() >= $productPrices->getRevision()) {
            $this->logger->debug("Ignore product '{$product->getEntityId()}', old revision");
            return true; // Ignore, revision already handled
        }

        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpLog->loadLatest($product->getEntityId(), null, 'ErpProduct');

        // Monetary prices
        foreach ($productPrices->prices as $price) {
            if ($currency != $price->currency) {
                continue; // Ignore
            }
            $this->handlePrice($price, $product);
        }
        // Point prices
        $pointTypes = array_keys(Mage::helper('points_core')->getTypeProviders());
        foreach ($productPrices->prices as $price) {
            if (!in_array($price->currency, $pointTypes)) {
                continue; // Ignore
            }
            $this->handlePoints($price, $product);
        }
        $product->setForcedEvaluation(true);
        $product->save();

        $newLog = Mage::getModel('awardit_eventlistener/event_log');
        $newLog->setPreviousLogId($pricingLog->getLogId());
        $newLog->setRevision($productPrices->getRevision());
        $newLog->setProductId($product->getEntityId());
        $newLog->setStoreId($store->getStoreId());
        $newLog->setEventType('ProductPricingPrice');
        /** @psalm-suppress UndefinedDocblockClass */
        $newLog->setEventData($productPrices->formatMessagePayload());
        $newLog->save();

        $this->logger->info("Updated product '{$product->getEntityId()}'");
        return true;
    }

    private function handlePrice(Price $price, Mage_Catalog_Model_Product $product): void
    {
        /** @psalm-suppress TypeDoesNotContainNull */
        if (is_null($price->price)) {
            return; // Magento don't accept null, observer sets not visible
        }
        $amount = (float)$price->price;
        switch ($price->type) {
            case 'sale':
                $product->setPrice($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            case 'recommended':
                $product->setMsrp($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            case 'invoice':
                $product->setInvoicePrice($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            case 'purchase':
                $product->setPurchasePrice($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            default:
                $this->logger->warning("Unsupported {$price->type} price for '{$product->getSku()}'");
                return;
        }
    }

    private function handlePoints(Price $price, Mage_Catalog_Model_Product $product): void
    {
        /** @psalm-suppress TypeDoesNotContainNull */
        if (is_null($price->price)) {
            return; // Magento don't accept null, observer sets not visible
        }

        $pointPrice = Mage::getModel('points_core/product_price');
        $pointPrice->loadByStoreProductTypeCustomerGroupId($product->getStore(), $product, $price->currency);

        if (empty($pointPrice->getId())) {
            $pointPrice->setStore($product->getStore());
            $pointPrice->setProduct($product);
            $pointPrice->setType($price->currency);
        }
        $amount = (int)$price->price;
        switch ($price->type) {
            case 'sale':
                $pointPrice->setPrice($amount);
                $this->logger->info("Updated {$price->type} points for '{$product->getSku()}'");
                break;
            case 'recommended':
            case 'invoice':
            case 'purchase':
                // @todo: Not default attribute
            default:
                $this->logger->warning("Unsupported {$price->type} points for '{$product->getSku()}'");
                break;
        }
        $pointPrice->save();
    }

    private function getProduct(string $code, Mage_Core_Model_Store $store): Mage_Catalog_Model_Product
    {
        return $this->helper->loadProduct($code, $store->getId());
    }

    private function getStore(string $code): Mage_Core_Model_Store
    {
        try {
            return Mage::app()->getStore($code);
        } catch (Mage_Core_Model_Store_Exception $e) {
            $this->logger->debug("Store '{$code}' do not exist, ignore",);
            throw new Awardit_EventListener_EventException(true);
        } catch (Throwable $t) {
            $this->logger->error("Store '{$code}' error");
            throw new Awardit_EventListener_EventException(false);
        }
    }
}
