<?php

declare(strict_types=1);

use Awardit\MagentoPsr\Psr3\Logger;
use Awardit\SimpleEvent\Event\ErpProduct;
use Awardit\SimpleEvent\Event\ProductPricingPrice;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Aws\Exception\CredentialsException;
use Crossroads\Magento\Test\Integration\MagentoManager;
use PHPUnit\Framework\TestCase;

class EventTest extends TestCase
{
    private static string $sku;
    private static int $store_id = 2;
    private static string $store_code = 'testing';

    public static function setUpBeforeClass(): void
    {
        // Need the same SKU in all tests.
        self::$sku = 'test-' . rand(1000, 9999);
        Mage::getConfig()->saveConfig('catalog/price/scope', '1');
    }

    public function setUp(): void
    {
        MagentoManager::reset();
        MagentoManager::init(self::$store_code);
        Mage::app()->setCurrentStore('0');
    }

    public function tearDown(): void
    {
        MagentoManager::logQueries();
    }

    public function testErpCreate(): void
    {
        $global = Mage::getModel('catalog/product');
        $global->setStoreId(0);
        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Create global product
        $erpEvent = $this->createErpProduct([
            'sku'   => self::$sku,
            'name'  => 'Test product 1',
        ]);
        $erpHandler->handle($erpEvent);

        // Load and check global product
        $global->load($global->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $global->getSku());
        $this->assertEquals(0, $global->getStoreId());
        $this->assertEquals('Test product 1', $global->getName());
        $this->assertEquals('Test product 1', $global->getDescription());
        $this->assertEquals('Test product 1', $global->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());
        $this->assertFalse($global->isAvailable());
        $this->assertFalse($global->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $global->getTypeId());
        $this->assertEquals(0, $global->getPrice());
        $this->assertEquals(0, $global->getMsrp());
        $this->assertNull($global->getWeight());
        $this->assertEquals('None', $global->getAttributeText('tax_class_id'));
        $this->assertNull($global->getAttributeText('manufacturer'));

        // Load and check global event log
        $erpLog->loadLatest($global->getEntityId(), null, 'ErpProduct');
        $this->assertEquals(1, $erpLog->getRevision());
        $this->assertEquals($erpEvent->formatMessagePayload(), $erpLog->getEventData());
    }

    public function testPriceCreate(): void
    {
        $local = Mage::getModel('catalog/product');
        $local->setStoreId(self::$store_id);
        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingHandler = new Awardit_EventListener_Handler_ProductPricingPrice(new Logger('test'));

        // Create local product
        $pricingEvent = $this->createProductPricingPrice([
            'sku'       => self::$sku,
            'priceList' => self::$store_code,
            'prices' => [
                [
                    'type'  => 'sale',
                    'price' => 123.45,
                ],
                [
                    'type'  => 'recommended',
                    'price' => 245.67,
                ],
            ],
            'revision'  => 1,
        ]);
        $pricingHandler->handle($pricingEvent);

        // Load and check local product
        $local->load($local->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1', $local->getName());
        $this->assertEquals('Test product 1', $local->getDescription());
        $this->assertEquals('Test product 1', $local->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $local->getVisibility());
        $this->assertFalse($local->isAvailable());
        $this->assertFalse($local->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $local->getTypeId());
        $this->assertNull($local->getWeight());
        $this->assertEquals('None', $local->getAttributeText('tax_class_id'));
        $this->assertNull($local->getAttributeText('manufacturer'));
        $this->assertEquals(123.45, $local->getPrice());
        $this->assertEquals(245.67, $local->getMsrp());

        // Load and check local event log
        $pricingLog->loadLatest($local->getEntityId(), self::$store_id, 'ProductPricingPrice');
        $this->assertEquals(1, $pricingLog->getRevision());
        $this->assertEquals($pricingEvent->formatMessagePayload(), $pricingLog->getEventData());
    }

    public function testErpUpdate(): void
    {
        $global = Mage::getModel('catalog/product');
        $global->setStoreId(0);
        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Update global product
        $erpEvent = $this->createErpProduct([
            'sku'               => self::$sku,
            'name'              => 'Test product 1 changed',
            'weight'            => 123.45,
            'sellable'          => true,
            'visible'           => true,
            'manufacturer'      => 'Test manufacturer',
            'taxClass'          => '2',
            'allowShortPick'    => true,
            'msrp'              => [
                [
                    'currency'  => 'SEK',
                    'price'     => '98.76',
                ],
                [
                    'currency'  => 'EUR',
                    'price'     => '9.87',
                ],
            ],
            'revision'          => 2,
        ]);
        $erpHandler->handle($erpEvent);

        // Load and check global product
        $global->load($global->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $global->getSku());
        $this->assertEquals(0, $global->getStoreId());
        $this->assertEquals('Test product 1 changed', $global->getName());
        $this->assertEquals('Test product 1 changed', $global->getDescription());
        $this->assertEquals('Test product 1 changed', $global->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());
        $this->assertFalse($global->isAvailable());
        $this->assertFalse($global->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $global->getTypeId());
        $this->assertEquals(0, $global->getPrice());
        $this->assertEquals(98.76, $global->getMsrp());
        $this->assertEquals(123.45, $global->getWeight());
        $this->assertEquals('Taxable Goods', $global->getAttributeText('tax_class_id'));
        $this->assertEquals('Test manufacturer', $global->getAttributeText('manufacturer'));

        // Load and check global event log
        $erpLog->loadLatest($global->getEntityId(), null, 'ErpProduct');
        $this->assertEquals(2, $erpLog->getRevision());
        $this->assertEquals($erpEvent->formatMessagePayload(), $erpLog->getEventData());
    }

    public function testEnableUpdate(): void
    {
        $local = Mage::getModel('catalog/product');
        $local->setStoreId(self::$store_id);
        $local->load($local->getIdBySku(self::$sku));
        $local->setEnriched(1);
        $local->save();

        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingHandler = new Awardit_EventListener_Handler_ProductPricingPrice(new Logger('test'));

        // Update local product
        $pricingEvent = $this->createProductPricingPrice([
            'sku'       => self::$sku,
            'priceList' => self::$store_code,
            'prices' => [
                [
                    'type'  => 'sale',
                    'price' => 234.67,
                ],
            ],
            'revision'  => 2,
        ]);
        $pricingHandler->handle($pricingEvent);

        // Load and check local product
        $local->load($local->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1 changed', $local->getName());
        $this->assertEquals('Test product 1 changed', $local->getDescription());
        $this->assertEquals('Test product 1 changed', $local->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_ENABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH, $local->getVisibility());
        $this->assertTrue($local->isAvailable());
        $this->assertTrue($local->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $local->getTypeId());
        $this->assertEquals(234.67, $local->getPrice());
        $this->assertEquals(245.67, $local->getMsrp());
        $this->assertEquals(123.45, $local->getWeight());
        $this->assertEquals('Test manufacturer', $local->getAttributeText('manufacturer'));

        // Load and check local event log
        $pricingLog->loadLatest($local->getEntityId(), self::$store_id, 'ProductPricingPrice');
        $this->assertEquals(2, $pricingLog->getRevision());
        $this->assertEquals($pricingEvent->formatMessagePayload(), $pricingLog->getEventData());
    }

    public function testErpArchive(): void
    {
        $global = Mage::getModel('catalog/product');
        $global->setStoreId(0);
        $local = Mage::getModel('catalog/product');
        $local->setStoreId(self::$store_id);
        $erpHandler = new Awardit_EventListener_Handler_ErpProduct(new Logger('test'));

        // Update global product
        $erpEvent = $this->createErpProduct([
            'sku'               => self::$sku,
            'name'              => 'Test product archived',
            'archived'          => true,
            'revision'          => 3,
        ]);
        $erpHandler->handle($erpEvent);

        // Load and check global product
        $global->load($global->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $global->getSku());
        $this->assertEquals(0, $global->getStoreId());
        $this->assertEquals('Test product archived', $global->getName());
        $this->assertEquals('Test product archived', $global->getDescription());
        $this->assertEquals('Test product archived', $global->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $global->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $global->getVisibility());
        $this->assertFalse($global->isAvailable());
        $this->assertFalse($global->isSalable());
        $this->assertEquals(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE, $global->getTypeId());
        $this->assertEquals(0, $global->getPrice());
        $this->assertEquals(98.76, $global->getMsrp());
        $this->assertNull($global->getWeight());
        $this->assertEquals('None', $global->getAttributeText('tax_class_id'));
        $this->assertNull($global->getAttributeText('manufacturer'));

        // Load and check local product
        $local->load($local->getIdBySku(self::$sku));
        $this->assertEquals(self::$sku, $local->getSku());
        $this->assertEquals(self::$store_id, $local->getStoreId());
        $this->assertEquals('Test product 1 changed', $local->getName());
        $this->assertEquals('Test product 1 changed', $local->getDescription());
        $this->assertEquals('Test product 1 changed', $local->getShortDescription());
        $this->assertEquals(Mage_Catalog_Model_Product_Status::STATUS_DISABLED, $local->getStatus());
        $this->assertEquals(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE, $local->getVisibility());
        $this->assertFalse($local->isAvailable());
        $this->assertFalse($local->isSalable());
    }

    public function testWorker(): void
    {
        // Just test that local code runs, i.e. expect SQS event read to fail
        $worker = new Awardit_EventListener_Worker();
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('Missing required client configuration options');
        $worker->run();
    }

    /**
     * @psalm-suppress InvalidArgument
     */
    private function createErpProduct(array $input): ErpProduct
    {
        $data = array_merge([
            'sku'               => 'GEN-PROD',
            'name'              => 'Generated product',
            'type'              => 'physical',
            'weight'            => null,
            'archived'          => false,
            'sellable'          => false,
            'visible'           => false,
            'allowShortPick'    => false,
            'manufacturer'      => null,
            'mpn'               => null,
            'taxClass'          => '0',
            'msrp'              => [],
            'purchasePrice'     => [],
            'revision'          => 1,
            'categories'        => [],
        ], $input);
        // TODO: Use rest-spread once we can use named parameters
        $instance = new ErpProduct($data);
        $instance->setRevision($data['revision']);
        return $instance;
    }

    /**
     * @psalm-suppress InvalidArgument
     */
    private function createProductPricingPrice(array $input): ProductPricingPrice
    {
        $data = array_merge([
            'sku'               => 'GEN-PROD',
            'priceList'         => 'GEN-STORE',
            'revision'          => 1,
            'archived'          => false,
            'prices'            => [],
            'campaign'          => null,
        ], $input);
        $data['prices'] = array_values(array_map(function (array $item) {
            return new Price([
                'type' => $item['type'],
                'currency' => 'SEK',
                'price' => (string)$item['price'],
                'lowestPrice30days' => (string)$item['price'],
                'customerGroup' => null,
            ]);
        }, $input['prices']));
        // TODO: Use rest-spread once we can use named parameters
        $instance = new ProductPricingPrice($data);
        return $instance;
    }
}
