<?php

use Awardit\SimpleEvent\Event\ProductPricingPrice;
use Awardit\SimpleEvent\Event\ProductPricingPrice\Price;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Handler_ProductPricingPrice
{
    private LoggerInterface $logger;

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Handler for ProductPricingPrice events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(ProductPricingPrice $productPrices): bool
    {
        $store = $this->getStore($productPrices->priceList);
        $product = $this->getProduct($productPrices->getEntityId(), $store);
        $currency = $store->getBaseCurrencyCode();

        $pricingLog = Mage::getModel('awardit_eventlistener/event_log');
        $pricingLog->loadLatest($product->getEntityId(), $store->getStoreId(), 'ProductPricingPrice');
        if ($pricingLog->getRevision() >= $productPrices->getRevision()) {
            $this->logger->debug("Ignore product '{$product->getEntityId()}', old revision");
            return true; // Ignore, revision already handled
        }

        $erpLog = Mage::getModel('awardit_eventlistener/event_log');
        $erpLog->loadLatest($product->getEntityId(), null, 'ErpProduct');

        $product->setStatus($this->getStatus($erpLog, $productPrices, $product));
        $product->setVisibility($this->getVisibility($erpLog, $product));

        // Monetary prices
        foreach ($productPrices->prices as $price) {
            if ($currency != $price->currency) {
                continue; // Ignore
            }
            $this->handlePrice($price, $product);
        }
        // Point prices
        $pointTypes = array_keys(Mage::helper('points_core')->getTypeProviders());
        foreach ($productPrices->prices as $price) {
            if (!in_array($price->currency, $pointTypes)) {
                continue; // Ignore
            }
            $this->handlePoints($price, $product);
        }

        $product->save();

        $newLog = Mage::getModel('awardit_eventlistener/event_log');
        $newLog->setPreviousLogId($pricingLog->getLogId());
        $newLog->setRevision($productPrices->getRevision());
        $newLog->setProductId($product->getEntityId());
        $newLog->setStoreId($store->getStoreId());
        $newLog->setEventType('ProductPricingPrice');
        /** @psalm-suppress UndefinedDocblockClass */
        $newLog->setEventData($productPrices->formatMessagePayload());
        $newLog->save();

        $this->logger->info("Updated product '{$product->getEntityId()}'");
        return true;
    }

    private function handlePrice(Price $price, Mage_Catalog_Model_Product $product): void
    {
        $amount = (float)$price->price;
        switch ($price->type) {
            case 'sale':
                $product->setPrice($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            case 'recommended':
                $product->setMsrp($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                return;
            case 'invoice':
            case 'purchase':
                // @todo: Not default attribute
            default:
                $this->logger->warning("Unsupported {$price->type} price for '{$product->getSku()}'");
                return;
        }
    }

    private function handlePoints(Price $price, Mage_Catalog_Model_Product $product): void
    {
        $amount = (int)$price->price;
        $pointPrice = Mage::getModel('points_core/product_price');
        $pointPrice->loadByStoreProductTypeCustomerGroupId($product->getStore(), $product, $price->currency);

        if (empty($pointPrice->getId())) {
            $pointPrice->setStore($product->getStore());
            $pointPrice->setProduct($product);
            $pointPrice->setType($price->currency);
        }
        switch ($price->type) {
            case 'sale':
                $pointPrice->setPrice($amount);
                $this->logger->info("Updated {$price->type} price for '{$product->getSku()}'");
                break;
            case 'recommended':
            case 'invoice':
            case 'purchase':
                // @todo: Not default attribute
            default:
                $this->logger->warning("Unsupported {$price->type} price for '{$product->getSku()}'");
                break;
        }
        $pointPrice->save();
    }

    private function getStatus(
        Awardit_EventListener_Model_Event_Log $erpLog,
        ProductPricingPrice $productPrices,
        Mage_Catalog_Model_Product $product
    ): int {
        if (empty($erpLog->getEventData()['visible'])) {
            return Mage_Catalog_Model_Product_Status::STATUS_DISABLED;
        }
        if ($productPrices->getIsDeleted()) {
            return Mage_Catalog_Model_Product_Status::STATUS_DISABLED;
        }
        if (!$product->getEnriched()) {
            return Mage_Catalog_Model_Product_Status::STATUS_DISABLED;
        }
        return Mage_Catalog_Model_Product_Status::STATUS_ENABLED;
    }

    private function getVisibility(
        Awardit_EventListener_Model_Event_Log $erpLog,
        Mage_Catalog_Model_Product $product
    ): int {
        if (!empty($erpLog->getEventData()['archived'])) {
            return Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE;
        }
        if ($product->getStatus() != Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
            return Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE;
        }
        return Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH;
    }

    private function getProduct(string $code, Mage_Core_Model_Store $store): Mage_Catalog_Model_Product
    {
        $product = Mage::getModel('catalog/product');
        $id = $product->getIdBySku($code);
        if (!$id) {
            $this->logger->warning("Product {$code} not available yet, wait");
            throw new Awardit_EventListener_EventException(false);
        }
        $product->setStoreId($store->getId());
        /** @psalm-suppress InvalidArgument */
        $product->load($id, ['enriched']);
        return $product;
    }

    private function getStore(string $code): Mage_Core_Model_Store
    {
        try {
            return Mage::app()->getStore($code);
        } catch (Mage_Core_Model_Store_Exception $e) {
            $this->logger->debug("Store '{$code}' do not exist, ignore",);
            throw new Awardit_EventListener_EventException(true);
        } catch (Throwable $t) {
            $this->logger->error("Store '{$code}' error");
            throw new Awardit_EventListener_EventException(false);
        }
    }
}
