<?php

use Awardit\SimpleEvent\Event\ErpProduct;
use Psr\Log\LoggerInterface;

class Awardit_EventListener_Handler_ErpProduct
{
    private LoggerInterface $logger;

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Handler for ErpProduct events.
     * Return true if event is fully handled, false to return it to queue.
     */
    public function handle(ErpProduct $erpProduct): bool
    {
        $currency = Mage::app()->getBaseCurrencyCode();
        $lastLog = Mage::getModel('awardit_eventlistener/event_log');
        $product = Mage::getModel('catalog/product');
        $product->setStoreId(0);
        $id = $product->getIdBySku($erpProduct->getEntityId());
        if ($id) {
            // Existing entry
            $product = $product->load($id);
            $lastLog->loadLatest($product->getEntityId(), null, 'ErpProduct');
            if ($lastLog->getRevision() >= $erpProduct->getRevision()) {
                $this->logger->debug("Ignore product '{$product->getEntityId()}', old revision");
                return true; // Ignore, revision already handled
            }
            $this->updateStoreProducts($id, $erpProduct->getIsDeleted(), $erpProduct->visible);
        } else {
            // New entry
            if ($erpProduct->getIsDeleted()) {
                $this->logger->debug("Ignore product '{$product->getEntityId()}', is deleted");
                return true; // Ignore, no need to create
            }
            $product->setSku($erpProduct->getEntityId());
            $product->setAttributeSetId($product->getDefaultAttributeSetId());
            $product->setTypeId($this->getType($erpProduct->type));
            $product->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
            $product->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE);
            $product->setCategoryIds([]);
            $product->setPrice(0.0);
            $product->setEnriched(0);
        }

        $product->setName($erpProduct->name);
        $product->setDescription($erpProduct->name);
        $product->setShortDescription($erpProduct->name);
        // @todo: No property that correspons to "sellable" input
        $product->setStockData([
            'use_config_manage_stock' => 0,
            'manage_stock' => 0,
            'backorders' => (int)$erpProduct->allowShortPick,
        ]);
        $product->setWeight($erpProduct->weight);
        // @todo: Are these id compatible with Visma tax codes?!
        $product->setTaxClassId($erpProduct->taxClass ?: 0);
        $product->setManufacturer($this->getOptionId('manufacturer', $erpProduct->manufacturer, true));
        foreach ($erpProduct->msrp as $price) {
            if ($currency != $price['currency']) {
                continue; // Ignore
            }
            $product->setMsrp((float)$price['price']);
        }
        $product->save();

        /* @todo: Unhandled
            mpn: null,
            purchasePrice: [],
        */

        $newLog = Mage::getModel('awardit_eventlistener/event_log');
        $newLog->setPreviousLogId($lastLog->getLogId());
        $newLog->setRevision($erpProduct->getRevision());
        $newLog->setProductId($product->getEntityId());
        $newLog->setEventType('ErpProduct');
        $newLog->setEventData($erpProduct->formatMessagePayload());
        $newLog->save();

        $this->logger->info("Updated product '{$product->getEntityId()}'");
        return true;
    }

    private function updateStoreProducts(int $id, bool $archived, bool $visible): void
    {
        foreach (Mage::app()->getStores(true) as $store) {
            $local = Mage::getModel('catalog/product');
            $local->setStoreId($store->getId());
            $local->load($id);
            if (!$local->getSku()) {
                continue; // Do not exist
            }
            if ($local->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
                continue; // No change needed
            }
            if ($local->getVisibility() == Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE) {
                continue; // No change needed
            }
            if ($archived) {
                // Changed to archived, disable it
                $local->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
            }
            if (!$visible) {
                // Changed to not visible, hide it
                $local->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE);
            }
            $local->save();
        }
    }

    private function getType(string $type): string
    {
        switch ($type) {
            case 'physical':
                return Mage_Catalog_Model_Product_Type::TYPE_SIMPLE;
            case 'virtual':
                return Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL;
            default:
                throw new RuntimeException("Product type '{$type}' not supported.");
        }
    }

    private function getOptionId(string $code, ?string $value, bool $create = false): ?int
    {
        return Mage::helper('integration')->getOptionId($code, $value, $create);
    }
}
