<?php

declare(strict_types=1);

use DDTrace\SpanData;
use DDTrace\Tag;
use GraphQL\Executor\ReferenceExecutor;
use GraphQL\Language\AST\FieldNode;
use GraphQL\Server\StandardServer;
use GraphQL\Type\Definition\FieldDefinition;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\ResolveInfo;
use GraphQL\Validator\DocumentValidator;

use function DDTrace\trace_method;

// Anything longer than 1ms will be kept:
// TODO: Configurable?
define("DD_GRAPHQL_MIN_RESOLVER_DURATION", 1000 * 1000 /* ns */);

if(extension_loaded("ddtrace")) {
    if(class_exists(MageQL_Core_Router_GraphQL::class)) {
        trace_method(
            MageQL_Core_Router_GraphQL::class,
            "graphQL",
            /**
             * @param array{0:Zend_Controller_Request_Http, 1:Zend_Controller_Response_Abstract, 2:string} $args
             */
            function(SpanData $span, array $args): void {
                [$_req, $_res, $schemaName] = $args;

                $span->name = "handleRequest";
                $span->resource = $schemaName;
                $span->service = "mageql";
            }
        );

        trace_method(
            MageQL_Core_Router_GraphQL::class,
            "initSession",
            function(SpanData $span): void {
                $span->name = "initSession";
                $span->service = "magento";
            }
        );
    }

    if(class_exists(ReferenceExecutor::class)) {
        trace_method(
            StandardServer::class,
            "executeRequest",
            function(SpanData $span): void {
                $span->name = "executeRequest";
                $span->service = "mageql";
            }
        );

        trace_method(
            DocumentValidator::class,
            "validate",
            function(SpanData $span): void {
                $span->name = "validateQuery";
                $span->service = "mageql";
            }
        );

        // GraphQL resolvers/mutations
        trace_method(
            ReferenceExecutor::class,
            "resolveFieldValueOrError",
            /**
             * @param array{0:FieldDefinition, 1:FieldNode, 2:callable, 3:mixed, 4:ResolveInfo} $args
             * @param mixed $retval
             */
            function(SpanData $span, array $args, $retval): ?bool {
                // We can't trace all of the graphql-calls, this will end up
                // hitting the 1K spans per request limit. Instead only keep
                // long-running ones without errors.
                if ($span->getDuration() < DD_GRAPHQL_MIN_RESOLVER_DURATION && ! $retval instanceof Throwable) {
                    // By returning false we will discard the span:
                    // https://github.com/DataDog/dd-trace-php/blob/1b7cd4529ea3253328ab828d35dc2899b65aa998/ext/php7/php7/engine_hooks.c#L409
                    return false;
                }

                [$fieldDef, $_fieldNode, $_resolver, $_root, $resolveInfo] = $args;

                $span->name = "resolveFieldValue";
                $span->resource = $resolveInfo->parentType->name . "." . $fieldDef->name;
                $span->service = "graphql";

                // Manually set error if we have one
                if($retval instanceof Throwable) {
                    $span->meta[Tag::ERROR_MSG] = $retval->getMessage();
                    $span->meta[Tag::ERROR_TYPE] = get_class($retval);
                    $span->meta[Tag::ERROR_STACK] = $retval->getTraceAsString();
                }

                return null;
            }
        );

        trace_method(
            ReferenceExecutor::class,
            "executeOperation",
            function(SpanData $span): void {
                $span->name = "executeOperation";
                $span->service = "mageql";
            }
        );

        trace_method(
            ReferenceExecutor::class,
            "buildResponse",
            function(SpanData $span): void {
                $span->name = "ReferenceExecutor::buildResponse";
                $span->service = "mageql";
            }
        );
    }

    trace_method(
        Mage_Core_Model_App::class,
        "init",
        function(SpanData $span): void {
            $span->name = "init";
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_App::class,
        "baseInit",
        function(SpanData $span): void {
            $span->name = "baseInit";
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_App::class,
        "_initModules",
        function(SpanData $span): void {
            $span->name = "initModules";
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_App::class,
        "loadAreaPart",
        /**
         * @param array{0:string, 1:string} $args
         */
        function(SpanData $span, array $args): void {
            [$area, $part] = $args;

            $span->name = "loadAreaPart";
            $span->resource = sprintf("%s (%s)", $area, $part);
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_App::class,
        "_initCurrentStore",
        function(SpanData $span): void {
            $span->name = "initCurrentStore";
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_Config::class,
        "saveCache",
        function(SpanData $span): void {
            $span->name = "Config.saveCache";
            $span->service = "magento";
        }
    );

    trace_method(
        Varien_Simplexml_Config::class,
        "loadCache",
        function(SpanData $span): void {
            $span->name = "Varien_Simplexml_Config.loadCache";
            $span->resource = $this->getCacheId();
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Core_Model_Layout::class,
        "generateXml",
        function(SpanData $span): void {
            $span->name = "Layout.generateXml";
            $span->service = "magento";
        }
    );

    trace_method(
        Mage_Index_Model_Process::class,
        "processEvent",
        /**
         * @param array{0:Mage_Index_Model_Event} $args
         */
        function(SpanData $span, array $args): void {
            [$event] = $args;
            /**
             * @var Mage_Index_Model_Process
             */
            $process = $this;

            $span->name = "processEvent";
            $span->resource = sprintf("%s (%s)", $process->getIndexerCode(), $event->getEntity());
            $span->service = "magento-index";

            $span->meta = [
                "event_type" => $event->getType(),
                "indexer_code" => $process->getIndexerCode(),
                "entity_code" => $event->getEntity()
            ];
        }
    );

    trace_method(
        \Mage_Index_Model_Process::class,
        "reindexAll",
        function(SpanData $span): void {
            /**
             * @var Mage_Index_Model_Process
             */
            $process = $this;
            $span->name = "reindexAll";
            $span->resource = $process->getIndexerCode();
            $span->service = "magento-index";
        }
    );

    trace_method(
        Mage_Index_Model_Process::class,
        "lockAndBlock",
        function(SpanData $span): void {
            /**
             * @var Mage_Index_Model_Process
             */
            $process = $this;
            $span->name = "lockAndBlock";
            $span->resource = $process->getIndexerCode();
            $span->service = "magento-index";
        }
    );

    // Images
    trace_method(
        Mage_Catalog_Model_Product_Image::class,
        "resize",
        function(SpanData $span): void {
            $span->name = "resize";
            $span->service = "magento-image";
        }
    );

    if(extension_loaded("redis")) {
        // Skip the getlasterror since it is not something which communicates with
        // the server and it always completes instantly:
        trace_method(
            Redis::class,
            "getLastError",
            function(): bool {
                // Discard
                return false;
            }
        );
    }
}
