<?php

/*
 * This Magento instance uses PointsCore.
 * That's why we extend Awardit_Integration_Model_Cli_PointsCoreProductImport.
 *
 */

class Awardit_Integration_Model_Cli_PointsCoreAwarditB2BProductImport extends Awardit_Integration_Model_Cli_PointsCoreProductImport {

    protected $_defaultHelper = "integration/LocalData";

    // No, we are not supposed to update price in this Magento instance
    protected $_updatePrice = false;

    // TrInf2 is used to set alternate product name for invoices
    private $extraVismaProductDataFields = [ "Prod.TrInf2" => "awardit_invoice_name" ];

    // R11 is used for purchase price in Awardit shops
    private $extraVismaProductPriceFields = [ "ISNULL(TRY_PARSE(R11 AS INT),0)" => "awardit_flag_1" ];

    public function getExtraVismaProductDataFields()
    {
        return array_merge($this->extraVismaProductDataFields, parent::getExtraVismaProductDataFields());
    }

    public function getExtraVismaProductPriceFields()
    {
        return array_merge($this->extraVismaProductPriceFields, parent::getExtraVismaProductPriceFields());
    }

    public function getExtraVismaProductPriceFieldsAsString()
    {
        $extraFields = [];

        foreach ($this->getExtraVismaProductPriceFields() as $field => $name) {
            $extraFields[] = "{$field} AS {$name}";
        }

        return implode(",", $extraFields) . (!empty($extraFields) ? "," : "");

    }

    public function CLI_tagSingleProduct($param)
    {
        $wantedParams = [ "!sku" => "string" ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        // Make sure sku exists in this Magento instance (use R11 for that)
        $sqlQuery = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND PrDcMat.R11 = 1 AND Prod.ProdNo = :prodNo";
        $params = [ "prodNo" => $extractedParams["sku"] ];
        $prodNo = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchOne($sqlQuery, $params);
        if ($prodNo != $extractedParams["sku"]) {
            Mage::helper($this->_defaultHelper)->log("Product [{$extractedParams["sku"]}] does not belong in this Magento instance!", LOG_ERR);
            return;
        }

        Mage::helper($this->_defaultHelper)->log("Tag single product for import: {$prodNo}");
        $this->tagProductsForImport([$prodNo]);
        Mage::helper($this->_defaultHelper)->log("Done!");

    }

    public function importSingleProduct(&$integrationProduct, $iteration = 0, $maxQty = 0)
    {
        // This is supposed to skip directly to "grand parent" class
        return Awardit_Integration_Model_Cli_ProductImport::importSingleProduct($integrationProduct, $iteration, $maxQty);
    }

    public function getAllProducts()
    {
        // Create filter to only include products with prices in this Magento instance
        $skus = [];
        Mage::helper($this->_defaultHelper)->log("Scanning for all products in this instance");

        $sqlQuery = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND PrDcMat.R11 = 1";
        $products = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery);

        foreach ($products as $prodRow) {
            $skus[] = $prodRow["ProdNo"];
        }
        return $skus;
    }

    public function getAllUpdatedProducts()
    {
        // Default time if we don't get one from the database
        $lastUpdateTimestamp = strtotime("-1 hour");
        $updatedProducts = [];

        $sqlQuery = "SELECT MAX(`created_at`) as last_update FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS;
        $timestamp = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery);
        if (!empty($timestamp)) {
            $lastUpdateTimestamp = strtotime($timestamp);
        }
        $lastUpdateDate = date("Ymd", $lastUpdateTimestamp);
        $lastUpdateTime = date("Hi", $lastUpdateTimestamp) - 0;

        $logDate = date("Y-m-d H:i:s", $lastUpdateTimestamp);
        Mage::helper($this->_defaultHelper)->log("Scanning for new and/or updated products since {$logDate}");

        // get latest products based on product-table
        $sqlQuery1 = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND (Prod.ChDt > {$lastUpdateDate} OR (Prod.ChDt = {$lastUpdateDate} AND Prod.ChTm > {$lastUpdateTime})) AND PrDcMat.R11 = 1";
        $productData = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery1);

        // get latest products based on price-table
        $sqlQuery2 = "SELECT DISTINCT(p1.ProdNo) AS ProdNo FROM PrDcMat p1 JOIN Prod p2 ON p2.ProdNo = p1.ProdNo JOIN PrDcMat p3 ON p3.ProdNo = p2.ProdNo WHERE p2.Gr7 = 1 AND (p1.ChDt > {$lastUpdateDate} OR (p1.ChDt = {$lastUpdateDate} AND p1.ChTm > {$lastUpdateTime})) AND p3.R11 = 1";
        $productPrice = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery2);

        // get latest products based on stock update
        $sqlQuery3 = "SELECT DISTINCT(p1.ProdNo) AS ProdNo FROM StcBal p1 JOIN Prod p2 ON p2.ProdNo = p1.ProdNo JOIN PrDcMat p3 ON p3.ProdNo = p2.ProdNo WHERE p2.Gr7 = 1 AND p2.Gr2 != 4 AND (p1.ChDt > {$lastUpdateDate} OR (p1.ChDt = {$lastUpdateDate} AND p1.ChTm > {$lastUpdateTime})) AND p3.R11 = 1";
        $productStock = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery3);

        // join all above products
        foreach ($productData as $row) {
            $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
        }

        foreach ($productPrice as $row) {
            if (empty($updatedProducts[$row["ProdNo"]])) {
                $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
            }
        }

        foreach ($productStock as $row) {
            if (empty($updatedProducts[$row["ProdNo"]])) {
                $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
            }
        }

        return $updatedProducts;
    }

    public function doLocalPreProcessingOfData(&$integrationProduct)
    {
        // Keep list of processed currencies
        $processedCurrencies = [];

        // Check all price rows
        foreach (array_keys($integrationProduct["price_data"]) as $index) {

            // We are looking for price rows with purchase price, currency id and awardit flag
            if (!empty($integrationProduct["price_data"][$index]["purchase_price"]) && !empty($integrationProduct["price_data"][$index]["purchase_currency_id"]) && !empty($integrationProduct["price_data"][$index]["awardit_flag_1"])) {
                $currencyId = intval($integrationProduct["price_data"][$index]["purchase_currency_id"]);

                // Only process each currency once
                if (!in_array($currencyId, $processedCurrencies)) {

                    $processedCurrencies[] = $currencyId;

                    // Iterate over all shops for currency
                    $shopIds = Mage::helper($this->_defaultHelper)->getVismaShopIdsByCurrencyId($currencyId);
                    foreach ($shopIds as $shopId) {

                        // If price row for store already exists, use that row instead of adding new one
                        if (Mage::helper($this->_defaultHelper)->havePriceRowForCurrencyShopIdGroupId($integrationProduct["price_data"], $currencyId, $shopId, 0)) {
                            continue;
                        }

                        // Insert "fake" price row for each Visma shop id matching currency, using `purchase price` as `price`
                        $priceRow = [
                            "purchase_price" => 0,
                            "purchase_currency_id" => 0,
                            "tax_id" => 0,
                            "suggested_price" => 0,
                            "currency_id" => $currencyId,
                            "shop_id" => $shopId,
                            "price" => $integrationProduct["price_data"][$index]["purchase_price"],
                            "customer_group_id" => 0,
                            "from_date" => 0,
                            "to_date" => 0,
                            "updated_by" => "",
                            "updated_at_date" => 0,
                            "updated_at_time" => 0,
                            "awardit_flag_1" => 0,
                            "price_index" => "0-{$shopId}-{$currencyId}"
                        ];
                        foreach ($this->getExtraVismaProductPriceFields() as $extraField) {
                            $priceRow[$extraField] = 0;
                        }
                        $integrationProduct["price_data"][] = $priceRow;
                    }
                } else {

                    // Found duplicate, tell someone!
                    $subject = "Product [{$integrationProduct["sku"]}] have multiple rows with purchase price for Magento10 (Awardit)!";
                    Mage::helper("integration")->sendAdminEmail(
                        $subject,
                        "Look for price rows where:\n"
                            . "R11 = {$integrationProduct["price_data"][$index]["awardit_flag_1"]}\n"
                            . "Inköpspris = {$integrationProduct["price_data"][$index]["purchase_price"]}\n"
                            . "Inköpsvaluta = {$integrationProduct["price_data"][$index]["purchase_currency_id"]}\n\n"
                            . "Debugdata:\n"
                            . "index = {$index}\n"
                            . "price_data = " . print_r($integrationProduct["price_data"][$index], true)
                        ,
                        ["it","products"]
                    );
                }

                // Clean up after purchase price is extracted - no need for those price rows after this
                unset($integrationProduct["price_data"][$index]);
            }
        }

        // Make sure awardit_invoice_name has correct character encoding
        if (!empty($integrationProduct["extra_data"]["awardit_invoice_name"]) && !Mage::helper($this->_defaultHelper)->isUTF8($integrationProduct["extra_data"]["awardit_invoice_name"])) {
            $integrationProduct["extra_data"]["awardit_invoice_name"] = Mage::helper($this->_defaultHelper)->convertString($integrationProduct["extra_data"]["awardit_invoice_name"]);
        }

        parent::doLocalPreProcessingOfData($integrationProduct);
    }

    public function doLocalPostProcessingOfData($integrationProduct, $product)
    {
        $productId = Mage::getModel("catalog/product")->getIdBySku($integrationProduct["sku"]);
        if (empty($productId)) {
            return;
        }

        // Find and save purchase price in upsert table
        $upsertPriceInserts = [];
        foreach (array_keys($integrationProduct["price_data"]) as $index) {

            // We are looking for price rows with price index
            if (!empty($integrationProduct["price_data"][$index]["price_index"])) {

                // Price index consists of: "<customer_group_id>-<shop_id>-<currency_id>"
                $parts = explode("-", $integrationProduct["price_data"][$index]["price_index"]);
                if (count($parts) == 3) {
                    $storeIds = Mage::helper($this->_defaultHelper)->getMagentoStoreIdsByVismaShopAndCurrency($parts[1], $parts[2]);
                    foreach ($storeIds as $storeId) {
                        $upsertPriceInserts[] = "({$productId}, {$storeId}, {$integrationProduct["price_data"][$index]["price"]})";
                    }
                }
            }
        }

        if (!empty($upsertPriceInserts)) {
            $sqlQuery = "INSERT INTO awardit_upsert_product (product_id, store_id, purchase_price) VALUES ";
            $sqlQuery .= implode(",", $upsertPriceInserts);
            $sqlQuery .= " ON DUPLICATE KEY UPDATE purchase_price = VALUES(purchase_price)";

            $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
            $stmt->execute();

        }

        parent::doLocalPostProcessingOfData($integrationProduct, $product);
    }

    public function setDefaultNewProductExtraData(&$integrationProduct, &$product, &$productData)
    {
        $product->setAwarditAwaitingPrice(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $product->setAwarditIsFeatured(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
        $product->setAwarditPartnerOwned(Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO);
    }

    public function setLocalProductExtraData($websiteId, $storeId, $integrationProduct, $globalProduct, $localProduct)
    {
        $isUpdated = parent::setLocalProductExtraData($websiteId, $storeId, $integrationProduct, $globalProduct, $localProduct);

        if (!empty($integrationProduct["extra_data"]["awardit_invoice_name"]) && $localProduct->getAwarditInvoiceName() != $integrationProduct["extra_data"]["awardit_invoice_name"]) {
            $localProduct->setData("awardit_invoice_name", $integrationProduct["extra_data"]["awardit_invoice_name"])->getResource()->saveAttribute($localProduct, "awardit_invoice_name");
            $isUpdated = true;
        }

        return $isUpdated;

    }

    public function getStoreIdsForCurrency($currencyId)
    {
        $storeIds = [];
        foreach(Mage::Helper("integration")->getShopIdConversionTable() as $shop) {
            if ($shop["currency_id"] == $currencyId) {
                $storeIds[] = $shop["store_id"];
            }
        }
        return $storeIds;
    }
}
