<?php

class Awardit_Integration_Model_Producthandler extends Mage_Core_Model_Abstract {

    const IMAGE_CONTENT_TYPES = [ "image/jpeg", "image/jpg", "image/gif", "image/png"];
    const MEDIA_URL_FOR_CATEGORY_IMAGES = "/media/catalog/category/";

    protected $_saveStatus = true;
    protected $_blacklistedAttributes = [];
    protected $_productAttributeTables = [
        'catalog_product_entity_datetime',
        'catalog_product_entity_decimal',
        'catalog_product_entity_int',
        'catalog_product_entity_text',
        'catalog_product_entity_varchar',
    ];

    /**
     * @return array
     */
    public function productHandler()
    {
        if (empty($_POST["data"])) {
            return [
                "status" => 1,
                "message" => "Missing input!\n"
            ];
        }

        $input = json_decode($_POST["data"], true);
        if (!is_array($input)) {
            return [
                "status" => 1,
                "message" => "Unable to decode input!\n"
            ];
        }

        if (empty($input["action"])) {
            return [
                "status" => 1,
                "message" => "Missing action!\n"
            ];
        }

        try {
            switch ($input["action"]) {
                case "getStores":
                case "getList":
                    $sqlQuery = "SELECT T3.name, T1.website_id, T2.group_id, T3.store_id FROM core_website T1 JOIN core_store_group T2 ON T2.website_id = T1.website_id JOIN core_store T3 ON T3.group_id = T2.group_id WHERE T1.website_id > 0 AND T3.name NOT LIKE '%old%' AND T3.name NOT LIKE '%inactive%' ORDER BY T3.name";
                    return [
                        "status" => 1,
                        "instance" => $input["instance"],
                        "direction" => $input["direction"],
                        "data" => Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery)
                    ];

                case "getData":
                    return $this->handleGetData($input, false);

                case "getDataV2":
                    return $this->handleGetData($input, true);

                case "getCategories":
                    return $this->getCategoriesForStore($input);

                case "getAttributes":
                    return $this->getAttributes();

                case "getAttributeData":
                    return $this->getAttributeData($input);

                case "getDefaultAttributeData":
                    return $this->getDefaultAttributeData($input);

                case "setAttributeData":
                    return $this->setAttributeData($input);

                case "setDefaultAttributeData":
                    return $this->setDefaultAttributeData($input);

                case "setProductsInCategory":
                    return $this->setProductsInCategory($input);

                case "setData":
                    return $this->setProductData($input, false);

                case "setDataV2":
                    return $this->setProductData($input, true);

                case "updateProductData":
                    return $this->updateProductData($input);

                case "purgeProducts":
                    // purgeProducts($doIndex = false, $doEcho = false, $doLog = false)
                    return $this->purgeProducts(false, false, true);

                case "resetProduct":
                    return $this->resetProduct($input);

                case "updateItem": // WIP
                    return [
                        "status" => 1,
                        "message" => "You have reached updateItem.\n"
                    ];

                case "getQuotesForCustomer":
                    return $this->getQuotesForCustomer($input);

                case "getCategoryTree":
                    return $this->getCategoryTree($input);

                case "getCategoryData":
                    return $this->getCategoryData($input);

                case "setCategoryData":
                    return $this->setCategoryData($input);

                case "getSpecialData":
                    return $this->getSpecialData($input);

                case "setSpecialData":
                    return $this->setSpecialData($input);

                default:
                    return [
                        "status" => 1,
                        "message" => "Unknown action!\n"
                    ];
            }

        } catch (Exception $ex) {
            return [
                "status" => 1,
                "message" => $ex->getMessage() . "\n",
                "trace" => $ex->getTraceAsString()
            ];
        }

    }

    /**
     * @return array
     */
    public function scanAttributes()
    {
        $data = [
            "data" => "",
            "status" => 0
        ];

        try {
            // Fetch all attribute sets from this instance
            $sqlQuery1 = "SELECT * FROM eav_attribute_set WHERE entity_type_id = :typeId";
            $result = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery1, ["typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId() ]);

            if (!empty($result)) {

                $instanceId = Mage::helper("integration")->getInstanceId();
                $attributeQty = count($result);
                $data["message"] = "Current attribute qty: {$attributeQty}\n";

                // Prepare translation table for update
                $sqlQuery2 = "UPDATE attribute_set_translations SET sync_flag = 1 WHERE instance_id = {$instanceId}";
                $affectedRows2 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery2);
                $data["message"] .= "Previous attribute qty: {$affectedRows2}\n";

                // Create string with data to insert
                $sqlQuery3 = "INSERT INTO attribute_set_translations (attribute_name, instance_id, attribute_id, sync_flag) VALUES\n";
                $values = [];
                foreach ($result as $row) {
                    $safeName = addslashes($row["attribute_set_name"]);
                    $values[] = "('{$safeName}', {$instanceId}, {$row["attribute_set_id"]}, NULL)";
                }
                $sqlQuery3 .= implode(",\n", $values);
                $sqlQuery3 .= "\nON DUPLICATE KEY UPDATE attribute_id = VALUES(attribute_id), sync_flag = NULL";
                $affectedRows3 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery3);
                $data["message"] .= "Uppdated attribute rows: {$affectedRows3}\n";

                // Check to see if there was any update done.
                // If nothing was updated, there must be some kind of problem
                if ($affectedRows3 > 0) {

                    // Remove old attribute sets previously set in preparation
                    $sqlQuery4 = "DELETE FROM attribute_set_translations WHERE instance_id = {$instanceId} AND sync_flag = 1";
                    $affectedRows4 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery4);
                    $data["message"] .= "Removed old attributes qty: {$affectedRows4}\n";
                } else {
                    $data["status"] = 1;
                    $data["message"] .= "No updated attribute rows, did something go wrong?\n";
                }
            } else {
                $data["status"] = 1;
                $data["message"] = "No attributes were found?!\n";
            }

        } catch (Exception $exception) {
            $data["status"] = 1;
            $data["message"] = $exception->getMessage() . "\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @param bool $isV2
     * @return array
     */
    public function handleGetData($input, $isV2 = false)
    {
        if (empty($input["sku"])) {
            return [
                "status" => 1,
                "message" => "Missing sku!\n",
            ];
        }

        $sku = $input["sku"];
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (empty($productId)) {
            return [
                "status" => 1,
                "message" => "Product [{$sku}] does not exist at source\n",
            ];
        }

        if ($isV2) {
            $product = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);
            $data = $this->getDataFromProduct($product, true, true, $isV2);

            // If we only want default data, return it now
            if (empty($input["store_id"])) {
                return $data;
            }

        }

        if (empty($input["store_id"])) {
            return [
                "status" => 1,
                "message" => "Missing store id!\n",
            ];
        }

        $storeId = intval($input["store_id"]);
        if (empty($storeId)) {
            return [
                "status" => 1,
                "message" => "Unknown store id ({$input["store_id"]})\n",
            ];
        }

        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
        if (!$isV2) {
            return $this->getDataFromProduct($localProduct, true, true, $isV2);
        }

        $data["attributes_store"] = $this->getAttributeDataFromProduct($localProduct, $isV2);
        $data["store_id"] = $storeId;

        return $data;
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return array
     */
    public function getDataFromProduct($product, $includeGallery = false, $includeConfig = false, $isV2 = false)
    {
        $storeId = intval($product->getStoreId());
        $data = [
            "status" => 1,
            "sku" => $product->getSku(),
            "store_id" => $storeId,
            "store_ids" => $product->getStoreIds(),
            "attribute_set_id" => $product->getAttributeSetId(),
            "type_id" => $product->getData("type_id"),
            "source_instance_id" => Mage::helper("integration")->getInstanceId(),
            "source_currency_id" => intval(Mage::helper("integration")->getVismaCurrencyId( Mage::getStoreConfig("currency/options/base", $storeId) )),
            "source_country_id" => intval(Mage::helper("integration")->getVismaCountryId( Mage::getStoreConfig("general/country/default", $storeId) )),
            "source_tax_class_id" => intval($product->getTaxClassId()),
            "attributes" => $this->getAttributeDataFromProduct($product, $isV2),
            "related" => $this->getRelatedProductsFromProduct($product),
            "gallery" => $includeGallery ? $this->getGalleryDataFromProduct($product) : [],
        ];

        if ($includeConfig) {
            $configData = $this->getConfigDataFromProduct($product);
            if (!empty($configData)) {
                $data = array_merge($data, $configData);
            }
        }

        return $data;
    }

    /**
     * @param array $input
     * @return int
     */
    public function getTaxClassId($input)
    {
        $sqlQuery1 = "
            SELECT t.tax_percent
            FROM tax_translations t
            WHERE t.magento_instance = {$input["source_instance_id"]} AND t.magento_tax_id = {$input["source_tax_class_id"]} AND t.currency_id IN (0, {$input["source_currency_id"]}) AND t.country_id IN (0, {$input["source_country_id"]})
            ORDER BY t.currency_id DESC, t.country_id DESC
        ";

        $taxPercent = sprintf("%02.2f\n", floatval(Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery1)));
        $storeId = intval($input["store_id"]);
        $instanceId = Mage::helper("integration")->getInstanceId();
        $currencyId = intval(Mage::helper("integration")->getVismaCurrencyId( Mage::getStoreConfig("currency/options/base", $storeId) ));
        $countryId = intval(Mage::helper("integration")->getVismaCountryId( Mage::getStoreConfig("general/country/default", $storeId) ));

        $sqlQuery2 = "
            SELECT t.magento_tax_id
            FROM tax_translations t
            WHERE t.magento_instance = {$instanceId} AND t.tax_percent = {$taxPercent} AND t.currency_id IN (0, {$currencyId}) AND t.country_id IN (0, {$countryId})
            ORDER BY t.currency_id DESC, t.country_id DESC
        ";
        return intval(Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery2));
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return array
     */
    public function getAttributeDataFromProduct($product, $isV2 = false)
    {
        $data = [];
        $attributesToExport = Mage::helper("integration")->getAttributesForAttributeSet($product->getAttributeSetId());
        foreach ($attributesToExport as $attributeCode => $attributeData) {
            if ($attributeData["frontend_input"] == "select") {
                $data[$attributeCode] = $product->getAttributeText($attributeCode);
            } else {
                $data[$attributeCode] = $product->getData($attributeCode);
            }
        }

        if ($isV2) {
            $data["source_price"] = $product->getPrice();
        }

        return $data;
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return array
     */
    public function getRelatedProductsFromProduct($product)
    {
        $data = [];
        $relatedCollection = $product->getRelatedProductCollection();
        foreach ($relatedCollection as $item) {
            $data[] = [
                "id" => $item->getId(),
                "sku" => $item->getSku(),
                "position" => $item->getPosition()
            ];
        }
        return $data;
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return array
     */
    public function getGalleryDataFromProduct($product)
    {
        $galleryData = $product->getData("media_gallery");
        if (empty($galleryData["images"])) {
            return [];
        }

        $data = $galleryData["images"];
        foreach (array_keys($data) as $index) {
            $data[$index]["url"] = Mage::getBaseUrl(Mage_Core_Model_Store::URL_TYPE_MEDIA, true) . "catalog/product" . $data[$index]["file"];
            $data[$index]["tags"] = [];

            if ($product->getImage() == $data[$index]["file"]) {
                $data[$index]["tags"][] = "image";
            }
            if ($product->getSmallImage() == $data[$index]["file"]) {
                $data[$index]["tags"][] = "small_image";
            }
            if ($product->getThumbnail() == $data[$index]["file"]) {
                $data[$index]["tags"][] = "thumbnail";
            }
        }
        return $data;
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return array
     */
    public function getConfigDataFromProduct($product)
    {
        if ($product->getData("type_id") !== Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE) {
            return [];
        }

        $data = [];
        $productAttributesOptions = $product->getTypeInstance(true)->getConfigurableOptions($product);
        if (!empty($productAttributesOptions) && count($productAttributesOptions) > 0) {
            $data["configurable"] = [];
            foreach ($productAttributesOptions as $productAttributeOption) {
                $data["configurable"][] = $productAttributeOption;
            }
            $data["configurable_attributes"] = $product->getTypeInstance()->getConfigurableAttributesAsArray($product);
        }

        return $data;
    }

    /**
     * @param array $input
     * @param bool $isV2
     * @return array
     */
    public function setProductData($input, $isV2 = false)
    {
        $testMode = false;
        $isUpdated = false;
        $isNew = false;

        $setConfigurables = empty($input["set_configurables"]) ? false : true;
        $setAttributes = empty($input["set_attributes"]) ? false : true;
        $forceCreate = empty($input["force_create"]) ? false : true;
        $setImages = empty($input["set_images"]) ? false : true;
        $setName = empty($input["set_name"]) ? false : true;
        $attributeSetWasChanged = false;

        if (empty($input["store_id"]) && !$forceCreate) {
            return [
                "status" => 1,
                "message" => "Missing store id!\n",
            ];
        }
        $storeId = intval($input["store_id"]);

        if (empty($storeId) && !$forceCreate) {
            return [
                "status" => 1,
                "message" => "Unknown value for store id ({$input["store_id"]})!\n",
            ];
        }

        if (empty($input["sku"])) {
            return [
                "status" => 1,
                "message" => "Missing sku!\n",
            ];
        }
        $sku = $input["sku"];
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);

        if (empty($productId)) {

            // If product does not exist, check to se if we need to create one.
            if ($setConfigurables && !empty($input["configurable"])) {

                // If it is a configurable product, create it
                $status = $this->createEmptyConfigurableProduct($input, $testMode);
                if (is_array($status)) {
                    return $status;
                }
                $productId = $status;


            } elseif ($forceCreate) {

                // If we are supposed to create new product, create it
                $status = $this->createProduct($input, $testMode);
                if (is_array($status)) {
                    return $status;
                }
                $productId = $status;

            } else {

                // Product does not exist and we are not supposed to create one
                return [
                    "status" => 1,
                    "message" => "Product with sku \"{$sku}\" does not exist at destination!\n",
                ];
            }

            if (empty($productId)) {
                return [
                    "status" => 1,
                    "message" => "Unable to create product with sku \"{$sku}\" at destination!\n",
                ];
            }

            $storeIds = [$storeId];
            $isNew = true;
        }

        // Product does exist or was just created, load it in global scope first
        $product = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);
        $storeIds = $product->getStoreIds();

        if (empty($productId)) {
            return [
                "status" => 1,
                "message" => "Product with sku \"{$sku}\" does not exist at destination!\n",
            ];
        }

        // begin: Handle related products
        if (!empty($input["related"])) {
            $relatedLinkData = [];
            foreach ($input["related"] as $relatedItem) {
                $relatedItemId = $product->getIdBySku($relatedItem["sku"]);
                if (!empty($relatedItemId)) {
                    $relatedLinkData[$relatedItemId] = [
                        "position" => $relatedItem["position"]
                    ];
                }
            }
            if (!empty($relatedLinkData)) {
                $product->setRelatedLinkData($relatedLinkData);
                $isUpdated = true;
            }
        }
        // end: Handle related products

        // begin: Handle store relation
        if (!empty($storeId) && !in_array($storeId, $storeIds)) {

            // $isUpdated might be true if related products was set.
            if ($isUpdated) {
                $status = $this->saveProduct($product, $testMode);
                if (is_array($status)) {
                    return $status;
                }
            }
            return [
                "status" => 1,
                "message" => "Product with sku \"{$sku}\" does not belong to destination store #{$storeId}!\n"
            ];
        }
        // end: Handle store relation

        // begin: Handle attribute set and status
        if (empty($input["attributes"])) {
            return [
                "status" => 1,
                "message" => "No attributes to copy for product with sku \"{$sku}\"!\n",
            ];
        }
        $input["local_attribute_set_id"] = $product->getAttributeSetId();

        if ($setAttributes && !$isNew && !empty($input["source_instance_id"]) && !empty($input["attribute_set_id"])) {
            $newAttributeSetId = Mage::helper("integration")->translateAttributeSetId($input["source_instance_id"], $input["attribute_set_id"]);

            if ($newAttributeSetId != $product->getAttributeSetId()) {

                if (!$testMode) {
                    $product->setAttributeSetId($newAttributeSetId);
                    $input["local_attribute_set_id"] = $product->getAttributeSetId();
                    $attributeSetWasChanged = true; // Need to know when to delete old attributes
                    $isUpdated = true; // Need to save product when changing attribute set
                } else {
                    $input["local_attribute_set_id"] = $newAttributeSetId;
                }
            }

            if ($this->_saveStatus) {
                $currentStatus = $product->getStatus();
                $newStatus = Mage::helper("integration")->getSystemAttributeOptionValue("status", $input["attributes"]["status"]);
                if ($newStatus !== false && $newStatus != $currentStatus) {
                    $product->setStatus($newStatus);
                    $isUpdated = true;
                }
            }
        }
        // end: Handle attribute set

        // begin: Handle product type (simple, virtual, etc.)
        if (!$isNew) {
            if ($setConfigurables && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE) {
                $isUpdated = $this->setConfigurableDataOnProduct($input, $product) ? true : $isUpdated;
            } elseif ($setAttributes && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL && $input["type_id"] === Mage_Catalog_Model_Product_Type::TYPE_SIMPLE) {
                $product->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE);
                $isUpdated = true;
            } elseif ($setAttributes && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_SIMPLE && $input["type_id"] === Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL) {
                $product->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL);
                $isUpdated = true;
            }
        }
        // end: Handle product type (simple, virtual, etc.)

        // begin: Handle image gallery
        if ($setImages && !empty($input["gallery"])) {

            // Only add images if product doesn't have one
            $galleryData = $product->getData("media_gallery");
            if (empty($galleryData["images"])) {
                $isUpdated = $this->addImageToProduct($input["gallery"], $product) ? true : $isUpdated;
            } else {
                // Product have images, skipping adding new ones
            }
        }
        // end: Handle image gallery

        // If we force created a new product, then set default attribute values first
        if ($forceCreate && $isNew) {

            $status = $this->setProductAttributes($input, $product, $testMode, $isV2);
            if (is_array($status)) {
                return $status;
            }
            $isUpdated = $status ? true : $isUpdated;
        }

        // Save default data
        if ($isUpdated) {
            $status = $this->saveProduct($product, $testMode);
            if (is_array($status)) {
                return $status;
            }
        }

        // If product was force created and we have no store specific data, we are done!
        if ($forceCreate && $isNew && empty($input["attributes_store"])) {
            return [
                "status" => 1,
                "sku" => $sku,
                "store_id" => $storeId,
                "instance" => $input["instance"],
                "debug" => "Only default data was used for product."
            ];
        }

        // If getDataV2 was used, swap out default attribute values for store specific ones
        if ($isV2) {
            if (empty($input["attributes_store"])) {
                return [
                    "status" => 1,
                    "message" => "Missing store specific attributes!\n"
                ];
            }
            $input["attributes"] = $input["attributes_store"];
        }

        // Reload product in local context
        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);

        if ($setAttributes || $setName) {
            if ($setAttributes) {

                // Set attributes
                $status = $this->setProductAttributes($input, $localProduct, $testMode, $isV2);
                if (is_array($status)) {
                    return $status;
                }
                $isUpdated = $status ? true : $isUpdated;

            } elseif ($setName) {

                // Set name
                if ($localProduct->getName() != $input["attributes"]["name"]) {
                    $localProduct->setName($input["attributes"]["name"]);
                    $isUpdated = true;
                }
            }

            if ($isUpdated) {
                $status = $this->saveProduct($product, $testMode);
                if (is_array($status)) {
                    return $status;
                }
            }

            if ($attributeSetWasChanged && !$testMode) {

                // Delete data from attributes no longer in new attribute set
                $resource = Mage::getSingleton("core/resource");
                $write = $resource->getConnection("core_write");

                $targetAttributes = Mage::getResourceModel("catalog/product_attribute_collection")
                    ->setAttributeSetFilter($input["local_attribute_set_id"])
                    ->getColumnValues("attribute_id");

                $condition = [
                    $write->quoteInto("entity_id = ?", $productId),
                    $write->quoteInto("attribute_id NOT IN (?)", $targetAttributes),
                ];

                try {
                    foreach ($this->_productAttributeTables as $table) {
                        $write->delete($resource->getTableName($table), $condition);
                    }
                } catch (Exception $ex) {
                    Mage::helper("integration")->logException($ex);
                    return [
                        "status" => 1,
                        "message" => "Exception while deleting old attributes!\nMessage: {$ex->getMessage()}",
                        "trace" => $ex->getTraceAsString()
                    ];
                }
            }
        }

        return [
            "status" => 1,
            "sku" => $sku,
            "store_id" => $storeId,
            "instance" => $input["instance"]
        ];

    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @param bool $testMode
     * @return bool|array
     */
    public function saveProduct($product, $testMode)
    {
        if (!$testMode) {
            try {
                $product->save();
            } catch (Exception $ex) {
                Mage::helper("integration")->logException($ex);
                return [
                    "status" => 1,
                    "message" => "Exception while saving product!\nMessage: {$ex->getMessage()}",
                    "trace" => $ex->getTraceAsString()
                ];
            }
        }

        return true;
    }

    /**
     * @param array $input
     * @return array
     */
    public function updateProductData($input)
    {
        $isUpdated = false;
        $data = [
            "status" => 1
        ];

        if (empty($input["store_id"]) || !is_numeric($input["store_id"])) {
            $data["message"] = "Unknown store #{$input["store_id"]}\n";
            return $data;
        }
        $storeId = intval($input["store_id"]);

        if (empty($input["sku"])) {
            $data["message"] = "Missing sku.\n";
            return $data;
        }
        $sku = $input["sku"];
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (empty($productId)) {
            $data["message"] = "Product #{$sku} does not exist at destination\n";
            return $data;
        }

        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
        foreach ($input["attributes"] as $attributeCode => $attributeValue) {
            if ($localProduct->getData($attributeCode) != $attributeValue) {
                $localProduct->setData($attributeCode, $attributeValue);
                $isUpdated = true;
            }
        }

        if ($isUpdated) {
            try {
                $localProduct->save();
                $data["message"] = "updated\n";
            } catch (Exception $ex) {
                Mage::helper("integration")->logException($ex);
                $data["status"] = 0;
                $data["message"] = "error!\n";
                $data["message"] .= "Exception while saving local product:\n";
                $data["message"] .= $ex->getMessage();
            }
        } else {
            $data["message"] = "no update needed\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @param bool $testMode
     * @return bool|array
     */
    public function setProductAttributes($input, &$product, $testMode = false, $isV2 = false)
    {
        $isUpdated = false;
        $attributesToImport = Mage::helper("integration")->getAttributesForAttributeSet($input["local_attribute_set_id"]);

        foreach ($input["attributes"] as $attributeCode => $attributeData) {

            // Skip "status" in local context. Only set this (elsware) for new configurable products.
            if ($attributeCode == "status") {
                continue;
            }

            if ($attributeCode == "source_price" && $isV2) {
                if ($product->getPrice() != $attributeData) {
                    $product->setData("price", $attributeData)->getResource()->saveAttribute($product, "price");
                    //$product->setPrice($attributeData);
                    $isUpdated = true;
                    continue;
                }
            }

            // Skip blacklisted attributes, if specified
            if (in_array($attributeCode, $this->_blacklistedAttributes)) {
                continue;
            }

            if (!array_key_exists($attributeCode, $attributesToImport)) {
                continue;
            }

            switch ($attributesToImport[$attributeCode]["frontend_input"]) {

                case "select":

                    if ($attributeData == "" || $attributeData == null) {
                        // Empty option value? Skip this attribute
                        continue 2;
                    }

                    try {
                        $options = Mage::getModel("eav/entity_attribute")->loadByCode("catalog_product", $attributeCode)->getSource()->getAllOptions();
                    } catch (Exception $ex) {
                        // Unable to load model or source, skip attribute
                        continue 2;
                    }

                    if (empty($options)) {
                        // No options? Skip this attribute
                        continue 2;
                    }

                    $foundOption = false;
                    foreach ($options as $option) {

                        // Find option
                        if ($option["label"] != $attributeData) {
                            continue 3;
                        }

                        // Only set data if it needs to be set
                        if ($product->getData($attributeCode) != $option["value"] && !$testMode) {
                            $product->setData($attributeCode, $option["value"])->getResource()->saveAttribute($product, $attributeCode);
                            $isUpdated = true;
                        }

                        // We found option, no need to look further
                        $foundOption = true;
                        break;
                    }

                    if (!$foundOption) {
                        return [
                            "status" => 1,
                            "message" => "Did not find attribute option \"{$attributeData}\" for \"{$attributeCode}\"\n"
                        ];
                    }
                    break;

                default:

                    // Only set data if it needs to be set
                    if ($product->getData($attributeCode) != $attributeData && !$testMode) {
                        $product->setData($attributeCode, $attributeData)->getResource()->saveAttribute($product, $attributeCode);
                        $isUpdated = true;
                    }
                    break;
            }
        }

        return $isUpdated;
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @return bool
     */
    public function addImageToProduct($input, &$product)
    {
        $isUpdated = false;

        try {
            $filePath = Mage::getBaseDir("media") . DS . "import";
            if (!is_dir($filePath) && !mkdir($filePath, 0777, true)) {
                Mage::log("Producthandler: Unable to create directory {$filePath}", Zend_Log::ERR, "producthandler");
                return $isUpdated;
            }

            foreach ($input as $imageData) {
                $filenameParts = explode("/", $imageData["file"]);
                $filename = $filenameParts[count($filenameParts) - 1];
                $filePathAndName = $filePath . DS . $filename;

                $rawData = file_get_contents($imageData["url"]);
                if ($rawData !== false) {
                    $bytes = file_put_contents($filePathAndName, $rawData);
                    if ($bytes !== false) {

                        $product->addImageToMediaGallery($filePathAndName, $imageData["tags"], true);
                        $gallery = $product->getData("media_gallery");
                        $lastImage = array_pop($gallery["images"]);

                        $lastImage["label"] = $imageData["label"];
                        $lastImage["position"] = $imageData["position"];
                        $lastImage["disabled"] = $imageData["disabled"];
                        $lastImage["label_default"] = $imageData["label_default"];
                        $lastImage["position_default"] = $imageData["position_default"];
                        $lastImage["disabled_default"] = $imageData["disabled_default"];

                        array_push($gallery["images"], $lastImage);
                        $product->setData("media_gallery", $gallery);
                        $isUpdated = true;
                    } else {
                        Mage::log("Producthandler: Unable to create file {$filePathAndName}", Zend_Log::DEBUG, "producthandler");
                    }
                } else {
                    Mage::log("Producthandler: Unable to read data from {$imageData["url"]}", Zend_Log::DEBUG, "producthandler");
                }
                unset($rawData);
            }
        } catch (Exception $exception) {
            Mage::helper("integration")->logException($exception, "Exception while adding image(s)!");
        }

        return $isUpdated;
    }

    /**
     * @param array $input
     * @param bool $testMode
     * @return int|array
     */
    public function createEmptyConfigurableProduct($input, $testMode)
    {
        $product = Mage::getModel("catalog/product");

        $newAttributeSetId = Mage::helper("integration")->translateAttributeSetId($input["source_instance_id"], $input["attribute_set_id"]);
        $websiteId = Mage::helper("integration")->getMagentoWebsiteIdByStoreId($input["store_id"]);

        $product
            ->setAttributeSetId($newAttributeSetId)
            ->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE)
            ->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH)
            ->setWebsiteIds([ $websiteId ])
            ->setStoreIds([ $input["store_id"] ])
            ->setCreatedAt(strtotime("now"))
            ->setSku($input["sku"])
            ->setPrice(0)
            ->setName($input["attributes"]["name"])
            ->setDescription($input["attributes"]["description"])
            ->setShortDescription($input["attributes"]["short_description"])
            ->setTaxClassId(0) // TODO: set correct VAT, use 0% as default for now.
        ;

        $configurableAttributesData = [];
        foreach ($input["configurable_attributes"] as $row) {
            $configurableAttributesData[] = [
                "label" => $row["label"],
                "values" => [],
                "attribute_id" => Mage::getSingleton("eav/config")->getAttribute("catalog_product", $row["attribute_code"])->getId(),
                "attribute_code" => $row["attribute_code"],
                "frontend_label" => $row["frontend_label"],
                "store_label" => $row["store_label"],
            ];
        }
        $product->setCanSaveConfigurableAttributes(true);
        $product->setConfigurableAttributesData($configurableAttributesData);
        $this->setConfigurableDataOnProduct($input, $product, true);

        $status = $this->saveProduct($product, $testMode);
        if (is_array($status)) {
            return $status;
        }

        return $product->getId();
    }

    /**
     * @param array $input
     * @param bool $testMode
     * @return int|array
     */
    public function createProduct($input, $testMode)
    {
        $product = Mage::getModel("catalog/product");
        $newAttributeSetId = Mage::helper("integration")->translateAttributeSetId($input["source_instance_id"], $input["attribute_set_id"]);
        $websiteId = Mage::helper("integration")->getMagentoWebsiteIdByStoreId($input["store_id"]);
        $newTaxClassId = $this->getTaxClassId($input);

        $product
            ->setAttributeSetId($newAttributeSetId)
            ->setTypeId($input["type_id"])
            ->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH)
            ->setWebsiteIds([ $websiteId ])
            ->setStoreIds([ $input["store_id"] ])
            ->setCreatedAt(strtotime("now"))
            ->setSku($input["sku"])
            ->setPrice(0.0)
            ->setTaxClassId($newTaxClassId)
            ->setWeight($input["attributes"]["weight"])
            ->setName($input["attributes"]["name"])
            ->setDescription($input["attributes"]["description"])
            ->setShortDescription($input["attributes"]["short_description"])
            ->setCategoryIds([])
            ->setStockData( ["use_config_manage_stock" => 0, "manage_stock" => 0 ])
            ->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED)
        ;

        $status = $this->saveProduct($product, $testMode);
        if (is_array($status)) {
            return $status;
        }

        return $product->getId();

    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @param bool $isNew
     * @return bool
     */
    public function setConfigurableDataOnProduct($input, &$product, $isNew = false)
    {
        $data = [
            "used_product_attribute_ids" => [],
            "attributes" => [],
            "status" => Mage_Catalog_Model_Product_Status::STATUS_DISABLED
        ];

        $optionValue = Mage::helper("integration")->getSystemAttributeOptionValue("status", $input["attributes"]["status"]);
        if ($optionValue !== false) {
            $data["status"] = $optionValue;
        }

        try {
            if (!empty($input["configurable"])) {

                foreach ($input["configurable"] as $attributes) {

                    $attributeCode = null;
                    $attributeId = null;

                    foreach ($attributes as $options) {
                        $sku = $options["sku"];

                        if (empty($attributeCode)) {
                            $attributeCode = $options["attribute_code"];
                            $attributeId = Mage::getResourceModel("eav/entity_attribute")->getIdByCode("catalog_product", $attributeCode);

                            if (!empty($attributeId)) {
                                $data["attributes"][$attributeId] = [
                                    "attribute_code" => $attributeCode,
                                    "attribute_id" => $attributeId,
                                    "options" => []
                                ];
                                $data["used_product_attribute_ids"][] = $attributeId;
                            }
                        }

                        if (!empty($attributeId) && !empty($attributeCode)) {
                            $optionLabel = $options["option_title"];
                            $optionId = Mage::helper("integration")->getOptionId($attributeCode, $optionLabel, false);

                            if (!empty($optionId)) {
                                $data["attributes"][$attributeId]["options"][$optionId] = [
                                    "sku" => $sku,
                                    "option_id" => $optionId,
                                    "option_label" => $optionLabel,
                                    "pricing_is_percent" => $options["pricing_is_percent"],
                                    "pricing_value" => $options["pricing_value"]
                                ];
                            }
                        }
                    }
                }
            }

            if (!empty($data["used_product_attribute_ids"]) && !empty($data["attributes"])) {
                if ($isNew) {
                    $product->getTypeInstance()->setUsedProductAttributeIds($data["used_product_attribute_ids"]);
                }
                $configurableProductsData = [];
                foreach ($data["attributes"] as $attributeId => $attributeData) {
                    foreach ($attributeData["options"] as $optionId => $optionData) {
                        $productId = Mage::getModel("catalog/product")->getIdBySku($optionData["sku"]);
                        $configurableProductsData[$productId] = [
                            0 => [
                                "label" => $optionData["option_label"],
                                "attribute_id" => $attributeId,
                                "value_index" => $optionData["option_id"],
                                "is_percent" => $optionData["pricing_is_percent"],
                                "pricing_value" => $optionData["pricing_value"]
                            ]
                        ];
                    }
                }

                $product->setConfigurableProductsData($configurableProductsData);
                if ($this->_saveStatus) {
                    $product->setStatus($data["status"]); // Set status in global scope
                }

                return true;
            }
        } catch (Exception $exception) {
            Mage::helper("integration")->logException($exception, "Exception while setting configurable data on product!");
        }
        return false;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getCategoriesForStore($input)
    {
        $storeId = null;
        $data = $input;
        $data["data"] = [];

        if (!empty($input["store"])) {
            $storeId = intval($input["store"]);
        }

        if (!empty($storeId)) {
            $sqlQuery1 = "
                SELECT
                    g.root_category_id
                FROM core_store s
                JOIN core_store_group g ON g.website_id = s.website_id
                WHERE s.store_id = :storeId
            ";
            $params1 = ["storeId" => $storeId];
            $rootCategoryId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery1, $params1);

            $sqlQuery2 = "
                SELECT
                    a.attribute_id
                FROM eav_entity_type t
                JOIN eav_attribute a ON a.entity_type_id = t.entity_type_id
                WHERE t.entity_type_code = 'catalog_category' AND a.attribute_code = 'name'
            ";
            $attributeId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery2);

            $sqlQuery3 = "
                SELECT
                    tmp.entity_id AS category_id,
                    COALESCE (tmp.local_value, tmp.global_value) AS name,
                    tmp.position,
                    tmp.`level`,
                    tmp.parent_id,
                    tmp.path
                FROM
                (
                    SELECT
                        IF (v.store_id = 0, v.value, null) AS global_value,
                        IF (v.store_id > 0, v.value, null) AS local_value,
                        c.entity_id,
                        c.position,
                        c.`level`,
                        c.parent_id,
                        c.path
                    FROM catalog_category_entity c
                    JOIN catalog_category_entity_varchar v ON v.entity_id = c.entity_id AND v.attribute_id = :attributeId
                    WHERE (c.path = :path1 OR c.path LIKE :path2) AND (v.store_id = 0 OR v.store_id = :storeId)
                ) tmp
                GROUP BY tmp.entity_id
                ORDER BY tmp.path #tmp.level, tmp.position, tmp.parent_id
            ";
            $params3 = [
                "attributeId" => $attributeId,
                "path1" => "1/{$rootCategoryId}",
                "path2" => "1/{$rootCategoryId}/%",
                "storeId" => $storeId
            ];
            $data["data"] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery3, $params3);
            if (!empty($data["data"])) {
                $data["status"] = 1;
            }
        }

        return $data;

    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $sqlQuery = "
            SELECT
                a.attribute_id,
                a.attribute_code,
                a.frontend_label
            FROM eav_entity_type t
            JOIN eav_attribute a ON a.entity_type_id = t.entity_type_id
            WHERE
                t.entity_type_code = 'catalog_product'
                AND a.backend_type = 'int'
                AND a.frontend_input = 'select'
                AND (a.source_model IS NULL OR a.source_model = 'eav/entity_attribute_source_table')
            ORDER BY a.frontend_label
        ";
        return [
            "status" => 1,
            "data" => Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery)
        ];

    }

    /**
     * @param array $input
     * @return array
     */
    public function getAttributeData($input)
    {
        if (empty($input["attributeId"])) {
            return [
                "status" => 1,
                "message" => "Missing attribute id!\n"
            ];
        }
        $attributeId = intval($input["attributeId"]);

        $sqlQuery = "
            SELECT
                t.option_id,
                t.language_code,
                COUNT(DISTINCT(t.`value`)) AS option_values_qty,
                GROUP_CONCAT(DISTINCT(t.`value`) SEPARATOR \"\t\") AS option_values
            FROM (
                SELECT
                    eaov.*,
                    COALESCE(d3.value, d2.value, d1.value) AS language_code
                FROM eav_attribute ea
                JOIN eav_attribute_option eao ON eao.attribute_id = ea.attribute_id
                JOIN eav_attribute_option_value eaov ON eaov.option_id = eao.option_id
                JOIN core_store s ON s.store_id = eaov.store_id
                LEFT JOIN core_config_data d1 ON d1.path = 'general/locale/code' AND d1.scope_id = 0
                LEFT JOIN core_config_data d2 ON d2.path = 'general/locale/code' AND d2.scope = 'websites' AND d2.scope_id = s.website_id
                LEFT JOIN core_config_data d3 ON d3.path = 'general/locale/code' AND d3.scope = 'stores' AND d3.scope_id = s.store_id
                WHERE a.attribute_id = :attributeId
            ) t
            GROUP BY t.option_id, t.language_code
            ORDER BY t.option_id, t.language_code
        ";
        $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "attributeId" => $attributeId]);

        // Make option_values an array of values
        foreach(array_keys($data) as $index) {
            $data[$index]["option_values"] = !empty($data[$index]["option_values"]) ? explode("\t", $data[$index]["option_values"]) : [];
        }

        return [
            "status" => 1,
            "data" => $data
        ];
    }

    /**
     * @param array $input
     * @return array
     */
    public function getDefaultAttributeData($input)
    {
        if (empty($input["attributeId"])) {
            return [
                "status" => 1,
                "message" => "Missing attribute id!\n"
            ];
        }
        $attributeId = intval($input["attributeId"]);

        $sqlQuery = "
            SELECT
                v.option_id,
                d.`value` AS language_code,
                v.`value` AS option_value
            FROM eav_attribute a
            JOIN eav_attribute_option o ON o.attribute_id = a.attribute_id
            JOIN eav_attribute_option_value v ON v.option_id = o.option_id
            JOIN core_config_data d ON d.path = 'general/locale/code' AND v.store_id = 0 AND d.scope_id = 0 AND d.scope = 'default'
            WHERE a.attribute_id = :attributeId
            ORDER BY v.option_id ASC, d.value DESC
        ";
        $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "attributeId" => $attributeId]);

        return [
            "status" => 1,
            "data" => $data
        ];
    }

    /**
     * @param array $input
     * @return array
     */
    public function setAttributeData($input)
    {
        /*
         * 1) Fetch list of stores grouped by their language.
         *
         * 2) Try to find attribute option matching default store.
         *    If no match is found, create it.
         *
         * 3) Loop through source language values and try matching value in destination stores.
         *    If option value exists but is wrong, change it.
         *    If option value is missing, create it.
         *
         */

        if (empty($input["attribute_code"])) {
            return [
                "status" => 1,
                "message" => "Missing attribute code!\n"
            ];
        }

        if (empty($input["data"][0])) {
            return [
                "status" => 1,
                "message" => "Missing attribute data!\n"
            ];
        }

        $storeLanguages = $this->getStoreLanguages();
        if (empty($storeLanguages)) {
            return [
                "status" => 1,
                "message" => "Target does not have any languages set!\n"
            ];
        }

        // Fetch all local options - if any
        $localAttributeOptions = $this->getAttributeOptions($input["attribute_code"]);

        // Process data in batches, grouping by option id
        // Set first batch to process by using first element in source data
        $sourceOptionId = intval($input["data"][0]["option_id"]);

        // Add exit marker to end of source data
        $input["data"][] = [ "option_id" => null ];

        $batch = [];
        $message = "";
        foreach ($input["data"] as $dataRow) {

            // Collect all values for same option id
            if (intval($dataRow["option_id"]) === $sourceOptionId) {
                $batch[ $dataRow["language_code"] ] = $dataRow["option_values"];
                continue;
            }

            // If option id != $sourceOptionId, we either have next option id or we reached the end of input data.
            // In both cases, $batch should contain data for each language from source.
            // Let's iterate over each language in batch and create data for missing values.
            foreach ($batch as $remoteLanguageCode => $remoteOptionValues) {
                foreach ($storeLanguages as $localLanguage) {
                    $storeId = intval($localLanguage["store_id"]);
                    $localLanguageCode = $localLanguage["language_code"];
                    if ($remoteLanguageCode == $localLanguageCode) {

                        $optionValue = $remoteOptionValues[0];
                        if (!empty($remoteOptionValues[1])) {
                            $message .= "Found multiple option values ('" . implode("', '", $remoteOptionValues) . "') for {$localLanguageCode}, using {$optionValue}\n";
                        }

                        // We found language match and have selected option value, now find out if option values exists
                        $valueId = $this->findOptionValue($localAttributeOptions, $storeId, $optionValue);
                        if (!$valueId) {

                        }

                        return [
                            "status" => 1,
                            "message" => "Debug",
                            "debug" => [
                                "data" => $input["data"],
                                "localAttributeOptions" => $localAttributeOptions,
                                "storeLanguages" => $storeLanguages,
                                "localLanguage" => $localLanguage,
                                "batch" => $batch,
                            ]
                        ];
                    }
                }
            }

            // If we reached end of list, we are done
            if ($dataRow["option_id"] === null) {
                break;
            }

            // Reset for next batch and add first data
            $batch = [];
            $batch[ $dataRow["language_code"] ] = $dataRow["option_value"];
        }

        return [
            "status" => 1,
            "message" => $message,
        ];

    }

    /**
     * @param array $input
     * @return array
     */
    public function setDefaultAttributeData($input)
    {
        if (empty($input["attribute_code"])) {
            return [
                "status" => 1,
                "message" => "Missing attribute code!\n"
            ];
        }

        if (empty($input["data"][0])) {
            return [
                "status" => 1,
                "message" => "Missing attribute data!\n"
            ];
        }

        $msg = "";
        $storeId = 0;
        $faultyOptions = [];
        $newOptionsQty = 0;
        $oldOptionsQty = 0;
        $localOptions = $this->getAttributeOptions($input["attribute_code"]);

        foreach ($input["data"] as $dataRow) {

            $localOptionId = $this->findOptionValue($localOptions, $storeId, $dataRow["option_value"]);
            if (!empty($localOptionId)) {
                $oldOptionsQty++;
                continue;
            }

            $id = Mage::helper("integration")->createOption($input["attribute_code"], $dataRow["option_value"]);
            if ($id === null) {
                $faultyOptions[] = $dataRow["option_value"];
            } else {
                $newOptionsQty++;
            }
        }

        if ($oldOptionsQty) {
            if (!$newOptionsQty && empty($faultyOptions)) {
                $msg .= "Already had all options.\n";
            } elseif ($oldOptionsQty > 1) {
                $msg .= "Already had {$oldOptionsQty} options.\n";
            } else {
                $msg .= "Already had 1 option value.\n";
            }
        }

        if ($newOptionsQty) {
            if ($newOptionsQty > 1) {
                $msg .= "Successfully created {$newOptionsQty} new options.\n";
            } else {
                $msg .= "Successfully created 1 new option.\n";
            }
        }

        if (!empty($faultyOptions)) {
            if (count($faultyOptions) > 1) {
                $msg .= "Unable to create options values: ";
            } else {
                $msg .= "Unable to create options value: ";
            }
            $msg .= "'" . implode("','", $faultyOptions) . "'";
        }

        return [
            "status" => 1,
            "message" => $msg
        ];
    }

    /**
     * @return array
     */
    public function getStoreLanguages()
    {
        $sqlQuery = "
            SELECT
                s.store_id,
                COALESCE(d3.value, d2.value, d1.value) AS language_code
            FROM core_store s
            LEFT JOIN core_config_data d1 ON d1.path = 'general/locale/code' AND d1.scope_id = 0
            LEFT JOIN core_config_data d2 ON d2.path = 'general/locale/code' AND d2.scope = 'websites' AND d2.scope_id = s.website_id
            LEFT JOIN core_config_data d3 ON d3.path = 'general/locale/code' AND d3.scope = 'stores' AND d3.scope_id = s.store_id
            ORDER BY s.store_id
        ";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery);
    }

    /**
     * @param array $attributeCode
     * @return array
     */
    public function getAttributeOptions($attributeCode)
    {
        $sqlQuery = "
            SELECT
                eaov.*,
                COALESCE(d3.value, d2.value, d1.value) AS language_code
            FROM eav_attribute ea
            JOIN eav_attribute_option eao ON eao.attribute_id = ea.attribute_id
            JOIN eav_attribute_option_value eaov ON eaov.option_id = eao.option_id
            JOIN core_store s ON s.store_id = eaov.store_id
            LEFT JOIN core_config_data d1 ON d1.path = 'general/locale/code' AND d1.scope_id = 0
            LEFT JOIN core_config_data d2 ON d2.path = 'general/locale/code' AND d2.scope = 'websites' AND d2.scope_id = s.website_id
            LEFT JOIN core_config_data d3 ON d3.path = 'general/locale/code' AND d3.scope = 'stores' AND d3.scope_id = s.store_id
            WHERE
                ea.attribute_code = :attributeCode
                AND ea.entity_type_id = :typeId
        ";
        $params = [
            "attributeCode" => $attributeCode,
            "typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()
        ];
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);
    }

    /**
     * @param array $attributeOptionValues
     * @param int $storeId
     * @param string $optionValue
     * @return array
     */
    public function findOptionValue($attributeOptionValues, $storeId, $optionValue)
    {
        foreach ($attributeOptionValues as $row) {
            if ($row["store_id"] != $storeId) {
                continue;
            }
            if ($row["value"] == $optionValue) {
                return $row["value_id"];
            }
        }
        return null;
    }

    /**
     * @param array $input
     * @return array
     */
    public function setProductsInCategory($input)
    {
        $returnData = [
            "status" => null,
            "message" => ""
        ];

        if (!empty($input["store_id"])) {
            $storeId = intval($input["store_id"]);
        }
        // Had problems with local context, using global for now.
        $storeId = 0;

        if (!empty($input["category_id"])) {
            $categoryId = intval($input["category_id"]);
        }

        $overwritePosition = !empty($input["position"]);

        try {
            if (!empty($input["skus"])) {

                $model = Mage::getSingleton("catalog/category_api");
                $placeholders = [];
                $skuToId = [];

                // Build parameter list
                foreach ($input["skus"] as $skuIndex => $sku) {
                    $placeholders[":param{$skuIndex}"] = $sku;
                }

                // Use parameter list to fetch product id
                if (!empty($placeholders)) {
                    $placeholderString = implode(",", array_keys($placeholders));
                    $sqlQuery = "SELECT p.sku, p.entity_id FROM catalog_product_entity p WHERE p.sku IN ({$placeholderString})";
                    $skuToId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery, $placeholders);
                }

                // We now have a list of product id:s for all sku:s.
                if (!empty($skuToId)) {

                    $oldList = $model->assignedProducts($categoryId, $storeId);

                    // Check if we are supposed to overwrite current list of products in category
                    if (!empty($oldList) && !empty($input["overwrite"])) {
                        $returnData["message"] .= "Clearing all products from category";
                        foreach($oldList as $oldEntry) {
                            $model->removeProduct($categoryId, $oldEntry["product_id"]);
                        }
                        $returnData["message"] .= " - OK\n";
                        $oldList = [];
                    }

                    // Loop through list of sku/id and assign product to category
                    $position = 0;
                    foreach ($placeholders as $sku) {
                        if (array_key_exists($sku, $skuToId)) {
                            $productId = intval($skuToId[$sku]["entity_id"]);
                            if ($overwritePosition) {
                                $position++;
                            }
                            if (!empty($oldList)) {
                                foreach($oldList as $oldEntry) {
                                    if ($productId == $oldEntry["product_id"]) {
                                        $returnData["message"] .= "Updating product [{$sku}] to position {$position}";
                                        $model->updateProduct($categoryId, $productId, $position);
                                        $returnData["message"] .= " - OK\n";
                                        continue 2;
                                    }
                                }
                            }
                            try {
                                $returnData["message"] .= "Adding product [{$sku}] to position {$position}";
                                $model->assignProduct($categoryId, $productId, $position);
                                $returnData["message"] .= " - OK\n";
                            } catch (Exception $exception) {
                                $returnData["message"] .= " - Failed!\n";
                                if ($exception->getMessage() == "not_exists") {
                                    $returnData["message"] .= "Product [{$sku}] is not active in selected store!\n";
                                    $returnData["status"] = 0;
                                    continue;
                                } else {
                                    $returnData["message"] .= "Exception: " . $exception->getMessage() . "\n";
                                    $returnData["message"] .= $exception->getTraceAsString();
                                    $returnData["status"] = 0;
                                    break;
                                }
                            }
                        } else {
                            $returnData["message"] .= "Product [{$sku}] does not exist in selected Magento instance!\n";
                            $returnData["status"] = 0;
                        }
                    }
                } else {
                    $returnData["message"] .= "Found no matching products for wanted list of SKUs\n";
                }
            } else {
                $returnData["message"] .= "Missing list of SKUs\n";
            }
        } catch (Exception $exception) {
            $returnData["message"] .= "Exception: " . $exception->getMessage() . "\n";
            $returnData["message"] .= $exception->getTraceAsString();
            $returnData["status"] = 0;
            Mage::helper("integration")->logException($exception, "Exception while assigning products to category!");
        }

        if ($returnData["status"] === null) {
            $returnData["status"] = 1;
        }

        if (empty($returnData["message"])) {
            $returnData["message"] = "Nothing done!";
        }

        return $returnData;

    }

    /**
     * @param array $input
     * @return array
     */
    public function getQuotesForCustomer($input)
    {
        $data = [];

        if (!empty($input["customer_id"]) && is_numeric($input["customer_id"])) {
            $customerId = intval($input["customer_id"]);
            $sqlQuery = "
                SELECT
                    q.entity_id AS quote_id,
                    q.store_id,
                    cs.`name` AS store_name,
                    ce.email,
                    q.created_at,
                    qi.sku,
                    qi.`name`,
                    q.grand_total,
                    qp.additional_information
                FROM sales_flat_quote q
                JOIN sales_flat_quote_payment qp ON qp.quote_id = q.entity_id
                JOIN sales_flat_quote_item qi ON qi.quote_id = q.entity_id
                JOIN core_store cs ON cs.store_id = q.store_id
                JOIN customer_entity ce ON ce.entity_id = q.customer_id
                WHERE qp.additional_information IS NOT NULL AND q.store_id IN (6,7,8,9) AND q.customer_id = :customerId
            ";
            $data["status"] = 1;
            $data["message"] = "Got info for customer #{$customerId}";
            $data["data"] = [];
            $quotes = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "customerId" => $customerId ]);
            foreach ($quotes as $quote) {
                $additionalInformation = unserialize($quote["additional_information"]);
                if (!empty($additionalInformation["intent_id"])) {
                    $data["data"][] = [
                        "quote_id" => intval($quote["quote_id"]),
                        "store_id" => intval($quote["store_id"]),
                        "store_name" => $quote["store_name"],
                        "email" => $quote["email"],
                        "created_at" => $quote["created_at"],
                        "sku" => $quote["sku"],
                        "item_name" => $quote["name"],
                        "grand_total" => floatval($quote["grand_total"]),
                        "intent_id" => $additionalInformation["intent_id"]
                    ];
                }
            }
        } else {
            $data["status"] = 1;
            $data["message"] = "Unknown customer\n";
        }

        return $data;
    }

    public function purgeProducts($doIndex = false, $doEcho = false, $doLog = false)
    {
        $stats = [
            "initialQty" => 0,
            "filteredQty" => 0,
            "totalAffectedRows" => 0,
            "updatedProductsQty" => 0,
            "deletedProductsQty" => 0,
            "actions" => []
        ];

        $msg = "Product purge started.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) Mage::log($msg, Zend_Log::DEBUG, "producthandler");

        // Fetch list of SKU:s from Magento where SKU matches those from Visma (only simple and virtual products) (matching example: '12345' or '12345-Strukt')
        $sqlQuery1 = "SELECT cpe.sku, cpe.entity_id FROM catalog_product_entity cpe WHERE cpe.type_id IN ('simple','virtual') AND cpe.sku REGEXP '^[0-9]{5}(-[a-zA-Z]+)?$'";
        $magentoList = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery1);
        if (empty($magentoList)) {
            $msg = "Found no products matching SKU filter";
            if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg;
            if ($doLog) Mage::log($msg, Zend_Log::DEBUG, "producthandler");
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        }
        $stats["initialQty"] = count($magentoList);
        $msg = "Found {$stats["initialQty"]} mathing products to check in Visma.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) Mage::helper("integration")->log($msg);

        // Split result list in batches
        $batches = array_chunk($magentoList, 100, true);
        foreach ($batches as $batch) {
            // Fetch status from Visma
            $inStr = implode("', '", array_keys($batch));
            $sqlQuery2 = "SELECT Prod.ProdNo AS sku, Prod.Gr2 AS visma_status, Prod.ChDt FROM Prod WHERE Prod.ProdNo IN ('{$inStr}')";
            $vismaList = Mage::helper("integration")->getVismaDB()->query($sqlQuery2, array_keys($batch));
            foreach ($vismaList as $row) {

                if (!array_key_exists($row["sku"], $magentoList)) {
                    // Only process products that actually exist in Magento
                    continue;
                }

                if ($row["visma_status"] > 0 && $row["visma_status"] < 4) {

                    // If Visma status is ok (1, 2 or 3), remove product from list
                    unset($magentoList[$row["sku"]]);

                } else {

                    // Copy some data from Visma for later use
                    $magentoList[$row["sku"]]["ChDt"] = $row["ChDt"];
                    $magentoList[$row["sku"]]["visma_status"] = $row["visma_status"];
                }
            }
        }
        $stats["filteredQty"] = count($magentoList);

        if (empty($magentoList)) {
            $msg = "Found no products with status >= 4 in Visma";
            if ($doLog) Mage::helper("integration")->log($msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        } else {
            $msg = "Checked in Visma and found {$stats["filteredQty"]} disabled or archived products.";
            $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
            if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
            if ($doLog) Mage::helper("integration")->log($msg);
        }

        // Fetch info for attribute 'status'
        $sqlQuery3 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => "status"
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery3, $params);
        if (empty($eava["attribute_id"])) {
            $msg = "Magento is missing attribute 'status'?!";
            if ($doLog) Mage::helper("integration")->log($msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        }

        // Delete local value for attribute "status", if any exist.
        $sqlQuery4 = "DELETE cpex.* FROM catalog_product_entity_int cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery4);

        // Calculate "timestamp" (Visma uses INT for dates...).
        // Products changed on or before this date is considered ok to delete.
        $lastYear = intval(date("Ymd", strtotime("-1 year")));

        try {
            foreach ($magentoList as $row) {
                $stmt->bindValue("typeId", $params["typeId"]);
                $stmt->bindValue("attributeId", intval($eava["attribute_id"]));
                $stmt->bindValue("productId", intval($row["entity_id"]));
                $stmt->execute();
                $affectedRows = $stmt->rowCount();
                $stats["totalAffectedRows"] += $affectedRows;

                $globalProduct = Mage::getModel("catalog/product")->load($row["entity_id"]);

                // Find out what to do with product:
                if ($row["visma_status"] == Awardit_Integration_Model_Cli_ProductImport::VISMA_PRODUCT_STATUS_ARCHIVED && $row["ChDt"] <= $lastYear) {

                    // Delete products with status 6 in Visma, that has not been modified since 1 year.
                    // $globalProduct->delete();
                    // Just log delete for now
                    Mage::helper("integration")->log("Was about to delete product [{$row["sku"]}]");
                    $stats["deletedProductsQty"]++;

                } elseif ((!$doIndex && $affectedRows > 0) || $globalProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {

                    // Disable products that had local settings for "status" or where "status" is "enabled"
                    $globalProduct->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
                    $globalProduct->save();
                    $stats["updatedProductsQty"]++;

                }
            }
        } catch (Exception $exception) {
            $msg = "Exception handling product '{$row["sku"]}'!";
            Mage::helper("integration")->logException($exception, $msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        }

        $msg = "Deletes and updates done.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) Mage::helper("integration")->log($msg);

        if ($doIndex && ($stats["totalAffectedRows"] > 0 || $stats["updatedProductsQty"] > 0 || $stats["deletedProductsQty"] > 0)) {
            $indexCodes = [
                "catalog_product_price" => true,
                "catalog_product_flat" => true
            ];

            try {
                foreach ($indexCodes as $indexCode => $indexStatus) {
                    if ($indexStatus) {

                        $msg = "Running indexer for '{$indexCode}'.";
                        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
                        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
                        if ($doLog) Mage::helper("integration")->log($msg);

                        $process = Mage::getModel("index/indexer")->getProcessByCode($indexCode);
                        $process->reindexAll();

                        $msg = "Index done for '{$indexCode}'.";
                        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
                        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
                        if ($doLog) Mage::helper("integration")->log($msg);
                    }
                }
            } catch (Exception $exception) {
                $msg = "Exception running indexer for '{$indexCode}'!";
                Mage::helper("integration")->logException($exception, $msg);
                return [
                    "status" => 1,
                    "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                    "debug" => $stats
                ];
            }
        }

        $msg = "Purge done. Deleted {$stats["totalAffectedRows"]} local values, disabled {$stats["updatedProductsQty"]} and deleted {$stats["deletedProductsQty"]} products.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doLog) Mage::helper("integration")->log($msg);
        return [
            "status" => 1,
            "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
            "debug" => $stats
        ];

    }

    public function listDisabledProducts()
    {
        $stats = [
            "initialQty" => 0,
            "filteredQty" => 0,
            "totalLocalValues" => 0,
            "totalUpdateableProducts" => 0
        ];
        echo "Listing products from this Magento having status >= 4 in Visma.\n";

        // Fetch list of SKU:s from Magento where SKU matches those from Visma (only simple and virtual products) (matching example: '12345' or '12345-Strukt')
        $sqlQuery1 = "SELECT cpe.sku, cpe.entity_id FROM catalog_product_entity cpe WHERE cpe.type_id IN ('simple','virtual') AND cpe.sku REGEXP '^[0-9]{5}(-[a-zA-Z]+)?$'";
        $magentoList = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery1);
        if (empty($magentoList)) {
            echo "Found no products matching SKU filter.\n";
            return;
        }
        $stats["initialQty"] = count($magentoList);
        echo "Found {$stats["initialQty"]} mathing products to check in Visma.\n";

        // Split result list in batches
        $batches = array_chunk($magentoList, 100, true);
        foreach ($batches as $batch) {
            // Fetch status from Visma
            $inStr = implode("', '", array_keys($batch));
            $sqlQuery2 = "SELECT Prod.ProdNo AS sku, Prod.Gr2 AS visma_status FROM Prod WHERE Prod.ProdNo IN ('{$inStr}')";
            $vismaList = Mage::helper("integration")->getVismaDB()->query($sqlQuery2, array_keys($batch));
            foreach ($vismaList as $row) {
                // If Visma status is ok, remove product from list
                if ($row["visma_status"] > 0 && $row["visma_status"] < 4 && array_key_exists($row["sku"], $magentoList)) {
                    unset($magentoList[$row["sku"]]);
                }
            }
        }
        $stats["filteredQty"] = count($magentoList);

        if (empty($magentoList)) {
            echo "Found no products with status >= 4 in Visma.\n";
            return;
        } else {
            echo "Checked in Visma and found {$stats["filteredQty"]} disabled products.\n";
        }

        // Fetch info for attribute 'status'
        $sqlQuery3 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => "status"
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery3, $params);
        if (empty($eava["attribute_id"])) {
            echo "Magento is missing attribute 'status'?!\n";
            return;
        }

        $sqlQuery4 = "SELECT cpex.store_id, cpex.value AS localValuesQty FROM catalog_product_entity_int cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";

        foreach ($magentoList as $row) {
            $p = [
                "typeId" => $params["typeId"],
                "attributeId" => intval($eava["attribute_id"]),
                "productId" => intval($row["entity_id"])
            ];
            $localValues = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery4, $p);
            $localValuesQty = count($localValues);
            $stats["totalLocalValues"] += $localValuesQty;

            $globalProduct = Mage::getModel("catalog/product")->load($row["entity_id"]);
            if ($localValuesQty > 0 || $globalProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
                if ($localValuesQty > 0) {
                    echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' was about do be disabled and have local value in stores (" . implode(",", array_keys($localValues)) . ").\n";
                } else {
                    echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' was about do be disabled and have no local values.\n";
                }
                $stats["totalUpdateableProducts"]++;
            } else {
                echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' is already disabled.\n";
            }
        }

        echo "Would have deleted {$stats["totalLocalValues"]} local values and disabled {$stats["totalUpdateableProducts"]} products.\n";
        return;
    }

    /**
     * @param array $input
     * @return array
     */
    public function resetProduct($input)
    {
        $isUpdated = false;
        $data = [
            "status" => 1
        ];

        if (empty($input["attributes"])) {
            $data["message"] = "No attributes to reset?\n";
            return $data;
        }

        if (empty($input["sku"])) {
            $data["message"] = "Missing sku.\n";
            return $data;
        }
        $sku = $input["sku"];

        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (empty($productId)) {
            $data["message"] = "Product [{$sku}] does not exist at destination\n";
            return $data;
        }

        // Remove local values for supplied attributes
        foreach (array_keys($input["attributes"]) as $attributeCode) {
            $affectedRows = Mage::helper("integration")->deleteLocalAttributeValues($productId, $attributeCode);
            if ($affectedRows) {
                $isUpdated = true;
            }
        }

        // Need to remove local values before loading product, that's why we loop twice
        $globalProduct = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);
        foreach ($input["attributes"] as $attributeCode => $attributeValue) {
            // We use "-" as indicator for when attribute does not need to be updated
            if ($attributeValue !== "-" && $globalProduct->getData($attributeCode) != $attributeValue) {
                $globalProduct->setData($attributeCode, $attributeValue);
                $isUpdated = true;
            }
        }

        if ($isUpdated) {
            // Save triggers calculation of upsert index
            $globalProduct->save();
        }

        return $data;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getCategoryTree($input)
    {
        $defaultCategoryId = 1;
        $defaultParentId = 0;
        $defaultPosition = 0;
        if (!empty($input["store"])) {
            $storeId = intval($input["store"]);
            $defaultCategoryId = intval(Mage::app()->getStore($storeId)->getRootCategoryId());
        }

        $parents = [];
        $categoryCollection = Mage::getModel("catalog/category")
            ->getCollection()
            ->addAttributeToSelect(["name", "position", "level"])
            ->setOrder("level","ASC")
            ->setOrder("position","ASC");

        foreach ($categoryCollection as $category) {
            $parentId = intval($category->getParentId());
            $position = intval($category->getPosition());
            $id = intval($category->getEntityId());
            $parents[ $parentId ][ $position ] = [
                "id" => $id,
                "name" => $category->getName(),
                "level" => intval($category->getLevel()),
                "position" => $position
            ];

            if ($id === $defaultCategoryId) {
                $defaultParentId = $parentId;
                $defaultPosition = $position;
            }
        }

        return [
            "status" => 1,
            "data" => $this->treeCreator([], $parents, $parents[ $defaultParentId ][ $defaultPosition ])
        ];

    }

    /**
     * @param array $tree
     * @param array $parents
     * @param array $current
     * @return array
     */
    public function treeCreator($tree, $parents, $current)
    {
        $tree[] = $current;
        $newParentId = $current["id"];

        if (array_key_exists($newParentId, $parents)) {
            foreach ($parents[$newParentId] as $child) {
                $tree = $this->treeCreator($tree, $parents, $child);
            }
        }

        return $tree;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getCategoryData($input)
    {
        $storeId = 0;
        $rootCategoryId = 1;
        $copyProductList = false;
        $categoryData = [];

        if (!empty($input["store"])) {
            $storeId = intval($input["store"]);
            $rootCategoryId = intval(Mage::app()->getStore($storeId)->getRootCategoryId());
        }

        if (!empty($input["category"])) {
            $rootCategoryId = intval($input["category"]);
        }

        if (!empty($input["copy_prod"])) {
            $copyProductList = true;
        }
        $categoryData["default"] = $this->getCategoryDataForStore(0, $rootCategoryId);

        // Fetch list of products assigned to categories
        if ($copyProductList && !empty($categoryData["default"]["root"]["path"])) {
            $path = $categoryData["default"]["root"]["path"];
            $sqlQuery = "
                SELECT
                    c.entity_id AS category_id,
                    CONCAT(
                        '[',
                        GROUP_CONCAT(
                            JSON_OBJECT('sku', e.sku, 'position', p.`position`)
                            SEPARATOR ','
                        ),
                        ']'
                    ) AS products
                FROM catalog_category_entity c
                JOIN catalog_category_product p ON p.category_id = c.entity_id
                JOIN catalog_product_entity e ON e.entity_id = p.product_id
                WHERE (c.path = '{$path}' OR c.path LIKE '{$path}/%')
                GROUP BY c.entity_id
                ORDER BY c.parent_id, c.entity_id
            ";
            $categoryData["default"]["products"] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery);
            foreach (array_keys($categoryData["default"]["products"]) as $categoryId) {
                $categoryData["default"]["products"][$categoryId] = empty($categoryData["default"]["products"][$categoryId]["products"]) ? [] : json_decode($categoryData["default"]["products"][$categoryId]["products"], true);
            }
        }

        if (!empty($storeId)) {
            $categoryData["store"] = $this->getCategoryDataForStore($storeId, $rootCategoryId);
        }

        $qty = count($categoryData["default"]["children"]) +1;
        return [
            "status" => 1,
            "message" => "Got data from {$qty} categories.\n",
            "data" => $categoryData
        ];
    }

    /**
     * @param int $storeId
     * @param int $rootCategoryId
     * @return array
     */
    public function getCategoryDataForStore($storeId, $rootCategoryId)
    {
        $categoryData = [
            "root" => [],
            "children" => []
        ];

        $rootCategory = Mage::getModel("catalog/category")->setStoreId($storeId)->load($rootCategoryId);
        $categoryData["root"] = $rootCategory->getData();
        $path = $rootCategory->getPath();

        $categoryCollection = Mage::getModel("catalog/category")
            ->setStoreId($storeId)
            ->getCollection()
            ->addAttributeToSelect("*")
            ->addFieldToFilter("path", [ "like" => "{$path}/%" ] )
            ->setOrder("parent_id","ASC")
            ->setOrder("entity_id","ASC");

        foreach ($categoryCollection as $category) {
            $categoryData["children"][] = $category->getData();
        }

        return $categoryData;
    }

    /**
     * @param array $input
     * @return array
     */
    public function setCategoryData($input)
    {
        $storeId = 0;
        $clearDestination = $input["clear_destination"] ? true : false;
        $copyParent = $input["copy_parent"] ? true : false;
        $targetPosition = 1;
        $translation = [];
        $createdQty = 0;
        $msg = "";

        if (!empty($input["store"])) {
            $storeId = intval($input["store"]);
        }

        if (empty($input["category"])) {
            return [
                "status" => 1,
                "message" => "Missing destination category!\n",
            ];
        }
        $targetCategoryId = intval($input["category"]);

        if (empty($input["data"]["default"]["root"])) {
            return [
                "status" => 1,
                "message" => "Missing category data for default store!\n",
            ];
        }

        if (!empty($storeId) && empty($input["data"]["store"]["children"])) {
            return [
                "status" => 1,
                "message" => "Missing category data for store {$storeId}!\n",
            ];
        }

        $sourceCategoryId = intval($input["data"]["default"]["root"]["entity_id"]);
        $targetCategory = Mage::getModel("catalog/category")->load($targetCategoryId);

        if ($clearDestination) {

            if ($copyParent) {

                // If we copy source parent AND clear destination, we need to use parent of current target as new target.
                // Since we delete current target category, all its children will be deleted too.
                $parentCategoryId = $targetCategory->getParentId();
                $targetPosition = $targetCategory->getPosition();
                $targetCategory->delete();
                $targetCategory = Mage::getModel("catalog/category")->load($parentCategoryId);
                
            } else {

                // If we just clear destination, fetch all children to current target category
                $path = $targetCategory->getPath();
                $categoryCollection = Mage::getModel("catalog/category")
                    ->getCollection()
                    ->addAttributeToSelect("*")
                    ->addFieldToFilter("path", [ "like" => "{$path}/%" ] );

                foreach ($categoryCollection as $category) {

                    // We only need to delete first level of children, the rest will be deleted automatically.
                    $parentId = intval($category->getParentId());
                    if ($parentId == $targetCategoryId) {
                        $category->delete();
                    }
                }
            }
        }

        if ($copyParent) {

            // If we copy source parent, that category becomes the new target
            $category = Mage::getModel("catalog/category")
                ->setData($this->filterCategoryData($input["data"]["default"]["root"]))
                ->setPath($targetCategory->getPath())
                ->setPosition($targetPosition)
                ->save();
            $msg .= $this->copyCategoryImage($category->getImage(), $input);
            $createdQty++;
            $targetCategoryId = intval($category->getEntityId());

            // Assign products
            if (!empty($input["data"]["default"]["products"][$sourceCategoryId])) {
                $positions = $category->getProductsPosition();
                $addedProduct = false;
                foreach ($input["data"]["default"]["products"][$sourceCategoryId] as $prod) {
                    $productId = Mage::getModel("catalog/product")->getIdBySku($prod["sku"]);
                    if (!empty($productId) && !array_key_exists($productId, $positions)) {
                        $addedProduct = true;
                        $positions[$productId] = $prod["position"];
                    }
                }
                if ($addedProduct) {
                    $category->setPostedProducts($positions);
                    $category->save();
                }
            }

            // Set data for selected store
            if ($storeId && !empty($input["data"]["store"]["root"])) {
                $category->setStoreId($storeId)
                    ->setData($this->filterCategoryData($input["data"]["store"]["root"]))
                    ->save();
            }

            $targetCategory = Mage::getModel("catalog/category")->load($targetCategoryId);

        }

        $translation[$sourceCategoryId] = $targetCategoryId;

        foreach ($input["data"]["default"]["children"] as $index => $categoryData) {
            $sourceCategoryId = intval($categoryData["entity_id"]);
            $sourceParentId = intval($categoryData["parent_id"]);
            $targetId = intval($targetCategory->getEntityId());

            if (empty($translation[$sourceParentId])) {
                // This is bad, just bail out!
                return [
                    "status" => 1,
                    "message" => $msg . "Found no translation for source id at target!\n",
                    "debug" => [
                        "input" => $input,
                        "translation" => $translation,
                    ]
                ];
            }

            if ($targetId != $translation[$sourceParentId]) {
                $targetId = $translation[$sourceParentId];
                $targetCategory = Mage::getModel("catalog/category")->load($targetId);
            }

            // Create category for default store (admin)
            $category = Mage::getModel("catalog/category")
                ->setData($this->filterCategoryData($categoryData))
                ->setPath($targetCategory->getPath())
                ->save();
            $newCategoryId = intval($category->getEntityId());
            $msg .= $this->copyCategoryImage($category->getImage(), $input);

            // Assign products
            if (!empty($input["data"]["default"]["products"][$sourceCategoryId])) {
                $positions = $category->getProductsPosition();
                $addedProduct = false;
                foreach ($input["data"]["default"]["products"][$sourceCategoryId] as $prod) {
                    $productId = Mage::getModel("catalog/product")->getIdBySku($prod["sku"]);
                    if (!empty($productId) && !array_key_exists($productId, $positions)) {
                        $positions[$productId] = $prod["position"];
                        $addedProduct = true;
                    }
                }
                if ($addedProduct) {
                    $category->setPostedProducts($positions);
                    $category->save();
                }
            }

            // Set data for selected store
            if ($storeId && !empty($input["data"]["store"]["children"][$index])) {
                $localData = $this->filterChangedData($this->filterCategoryData($categoryData), $this->filterCategoryData($input["data"]["store"]["children"][$index]));
                if (!empty($localData)) {
                    $localCategory = Mage::getModel("catalog/category")
                        ->setStoreId($storeId)
                        ->load($newCategoryId);
                    foreach ($localData as $key => $val) {
                        $localCategory->setData($key, $val);
                    }
                    $localCategory->save();
                }
            }

            $translation[intval($categoryData["entity_id"])] = $newCategoryId;
            $createdQty++;
        }

        return [
            "status" => 1,
            "message" => $msg . "Created {$createdQty} categories.\n",
            "debug" => [
                "input" => $input,
                "translation" => $translation,
            ]
        ];
    }

    /**
     * @param array $categoryData
     * @return array
     */
    public function filterCategoryData($categoryData)
    {
        $keysToRemove = [
            "attribute_set_id",
            "available_sort_by",
            "children_count",
            "created_at",
            "custom_apply_to_products",
            "custom_use_parent_settings",
            "entity_id",
            "entity_type_id",
            "level",
            "parent_id",
            "path",
            "store_id",
            "updated_at",
            "url_path"
        ];

        foreach ($keysToRemove as $key) {
            unset($categoryData[$key]);
        }

        return $categoryData;
    }

    /**
     * @param array $dataSet1
     * @param array $dataSet2
     * @return array
     */
    public function filterChangedData($dataSet1, $dataSet2)
    {
        $result = [];
        foreach (array_keys($dataSet1) as $key) {
            if (array_key_exists($key, $dataSet2)) {
                if ($dataSet1[$key] !== $dataSet2[$key]) {
                    $result[$key] = $dataSet2[$key];
                }
            }
        }
        return $result;
    }

    /**
     * @param string $imageFilename
     * @param array $input
     * @return string
     */
    public function copyCategoryImage($imageFilename, $input)
    {
        $msg = "";
        if (empty($imageFilename)) {
            return $msg;
        }

        if ($input["instance"] === $input["source_instance"]) {
            return $msg;
        }
        if (empty($input["admin_url"])) {
            return $msg;
        }

        $filename = Mage::getBaseDir("media") . "/catalog/category/" . $imageFilename;
        // $msg .= "filename: {$filename}\n";
        if (file_exists($filename)) {
            $msg .= "Image \"{$imageFilename}\" already exists at destination.\n";
            return $msg;
        }

        $url = $input["admin_url"] . Awardit_Integration_Model_Producthandler::MEDIA_URL_FOR_CATEGORY_IMAGES . $imageFilename;
        // $msg .= "URL: {$url}\n";

        $fp = fopen ($filename, "w+");
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_TIMEOUT, 50);
        curl_setopt($ch, CURLOPT_FILE, $fp); // Give curl the file pointer so that it can write to it
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_exec($ch); // Data is written to file, we do not need to fetch it into a variable

        $info = curl_getinfo($ch);
        if ( !(!empty($info["http_code"]) && $info["http_code"] == 200)) {
            $msg .= "Bad http status code!\n";
            $msg .= print_r($info, true);
            return $msg;
        }
        if ( !(!empty($info["content_type"]) && in_array($info["content_type"], Awardit_Integration_Model_Producthandler::IMAGE_CONTENT_TYPES))) {
            $msg .= "Wrong content type for image!\n";
            $msg .= print_r($info, true);
            return $msg;
        }

        $errno = curl_errno($ch);
        if ($errno) {
            $msg .= "Got error {$errno} when fetching image from source!\n";
            $msg .= print_r($info, true);
        } else {
            $msg .= "Copied image \"{$imageFilename}\" from source.\n";
        }

        fclose($fp);
        curl_close($ch);

        return $msg;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getSpecialData($input)
    {
        return [
            "status" => 0,
            "message" => "Function not implemented!\n",
            "debug" => [
                "input" => $input
            ]
        ];
    }

    /**
     * @param array $input
     * @return array
     */
    public function setSpecialData($input)
    {
        return [
            "status" => 0,
            "message" => "Function not implemented!\n",
            "debug" => [
                "input" => $input
            ]
        ];
    }

}