<?php

class Awardit_Integration_Model_Cli_Stock extends Awardit_Integration_Model_Cli {

    // -m=Stock -f=TestDBAccess
    public function CLI_TestDBAccess()
    {
        try {
            echo "Testing DB access.\n";
            $sqlQuery = "SELECT * FROM stock LIMIT 10";
            $db = Mage::getSingleton("core/resource")->getConnection("stock_handler_read");
            $stockData = $db->fetchAll($sqlQuery);
            $rowCount = count($stockData);
            echo "Got {$rowCount} rows of data.\n";

        } catch (Exception $exception) {
            echo "Exception while testing DB access!\n";
            echo $exception->getMessage() . "\n";
            echo $exception->getTraceAsString() . "\n";
        }
    }

    // -m=Stock -f=Sync
    public function CLI_Sync($param)
    {
        $this->syncLocalStock();
    }

    // -m=Stock -f=SyncAll
    public function CLI_SyncAll($param)
    {
        $this->syncGlobalStock($param);
    }

    // -m=Stock -f=UpdateProducts
    public function CLI_UpdateProducts($param)
    {
        $this->updateProducts();
    }

    // -m=Stock -f=Reindex
    public function CLI_Reindex($param)
    {
        $this->doReindex();
    }

    public function syncLocalStock()
    {
        $syncStatus = false;
        Mage::log("Scanning for products that needs to sync local stock", Zend_Log::DEBUG);
        try {
            $sqlQuery = "
                SELECT
                    tmp.sku,
                    tmp.product_id,
                    tmp.min_qty,
                    tmp.qty,
                    tmp.is_in_stock
                FROM (
                    SELECT
                        p.sku,
                        s.product_id,
                        CAST(s.min_qty AS unsigned) AS min_qty,
                        CAST(s.qty AS unsigned) AS qty,
                        MAX(2 - i.value) AS is_enabled,
                        MAX(s.is_in_stock) AS is_in_stock
                    FROM catalog_product_entity p
                    JOIN cataloginventory_stock_item s ON s.product_id = p.entity_id
                    JOIN catalog_product_entity_int i ON i.entity_id = p.entity_id
                    JOIN eav_attribute a ON a.attribute_id = i.attribute_id
                    WHERE s.use_config_manage_stock = 0 AND s.manage_stock = 1 AND a.attribute_code = 'status'
                    GROUP BY p.sku
                    ORDER BY p.sku
                ) tmp
                WHERE tmp.is_enabled > 0
            ";
            $productData = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery);

            if (empty($productData)) {
                return;
            }

            $db = Mage::getSingleton("core/resource")->getConnection("stock_handler_read");
            $select = $db->select()
                    ->from(["s" => "stock"], ["s.sku", "external_qty" => "SUM(s.qty)"])
                    ->where("s.warehouse_id = ?", intval(Mage::getStoreConfig("integration/external_stock/warehouse_id")))
                    ->where("s.sku IN (?)", array_map(function($k) { return (string)$k; }, array_keys($productData)))
                    ->group("s.sku");
            $stockData = $db->fetchAssoc($select->__toString());

            $externalItemsQty = count($stockData);
            $actualUpdates = 0;
            Mage::log("Found {$externalItemsQty} products to potentially update.", Zend_Log::DEBUG);

            foreach ($stockData as $sku => $stockRow) {
                if (!empty($productData[$sku])) {

                    // Check if local stock qty differs from external stock qty
                    if ($productData[$sku]["qty"] != $stockRow["external_qty"]) {
                        $productData[$sku]["external_qty"] = $stockRow["external_qty"];
                        $this->updateLocalStock($productData[$sku]);
                        $actualUpdates++;
                        usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);

                    // Check if product has out of stock flag when stock qty indicates it should be in stock
                    } elseif ($productData[$sku]["qty"] > $productData[$sku]["min_qty"] && $productData[$sku]["is_in_stock"] = 0) {
                        $productData[$sku]["external_qty"] = $stockRow["external_qty"];
                        $this->updateLocalStock($productData[$sku]);
                        $actualUpdates++;
                        usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                    }
                }
            }

            // Do reindexing after successfull product sync and/or successfull stock update
            if ($syncStatus || !empty($actualUpdates)) {
                Mage::log("Found {$actualUpdates} products to actually update.", Zend_Log::DEBUG);
                $this->doReindex();
            }

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while syncing local stock!");
        }
    }

    public function syncGlobalStock($param)
    {
        // readAllStockDataFromVisma()->updateStockData();
        $wantedParams = [
            "timelimit" => "bool",
            "lastSync" => "timestamp",
            "warehouseId" => "int",
            "truncate" => "bool"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $useTimelimit = $extractedParams["timelimit"] ?? false;
        $lastSyncTimestamp = $extractedParams["lastSync"] ?? 0;
        $warehouseId = $extractedParams["warehouseId"] ?? 1;
        $truncateStock = $extractedParams["truncate"] ?? false;

        $limit = "";

        switch ($warehouseId) {
            case 2:
                $stcbal = "11,13";
                break;
            case 1:
            default:
                $stcbal = "1,3";
                break;
        }

        if ($this->getDebugMode()) {
            $specialDebugFilename = "/var/log/stock_raw.log";
            $specialDebugErrorFilename = "/var/log/stock_errors.log";
            $specialDebugUpdatesFilename = "/var/log/stock_updates.log";
        }

        if ($useTimelimit) {
            if (empty($lastSyncTimestamp)) {
                $sqlQuery = "SELECT MAX(synced_at) as last_sync FROM stock";
                $lastSync = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchOne($sqlQuery);
                if (empty($lastSync)) {
                    $lastSyncTimestamp = strtotime('-1 day');
                } else {
                    $lastSyncTimestamp = strtotime($lastSync);
                }
            }

            if (!empty($lastSyncTimestamp)) {
                $lastSyncDate = intval(date('Ymd', $lastSyncTimestamp));
                $lastSyncTime = intval(date('Hi', $lastSyncTimestamp));
                $dateStamp = date('Y-m-d H:i:s', $lastSyncTimestamp);
                Mage::log("Fetching stock data for all products changed since {$dateStamp}.", Zend_Log::DEBUG);
                $limit = sprintf("AND (p1.ChDt > %d OR (p1.ChDt = %d AND p1.ChTm > %d))", $lastSyncDate, $lastSyncDate, $lastSyncTime);
            }
        } else {
            Mage::log("Fetching stock data for all products.", Zend_Log::DEBUG);
        }

        try {
            $sqlQuery1 = sprintf(
                "SELECT
                    p1.ProdNo as sku,
                    p1.StcNo,
                    p2.Gr2,
                    p1.StcInc,
                    p1.Bal,
                    p1.ShpRsv,
                    p1.ShpRsvIn,
                    p1.PicNotR,
                    p1.InO,
                    p1.ChUsr,
                    p1.ChDt,
                    p1.ChTm,
                    CONCAT(
                        p1.ProdNo, ',\"',
                        p2.Descr, '\",',
                        p1.StcNo, ',',
                            p2.Gr2, ',',
                            CAST(p1.StcInc AS int), ',',
                            CAST(p1.Bal AS int), ',',
                            CAST(p1.ShpRsv AS int), ',',
                            CAST(p1.PicNotR AS int), ',',
                            CAST(p1.InO AS int), ',\"',
                        p1.ChUsr, '\",',
                        FORMAT(p1.ChDt,'####-##-##'), ',',
                        FORMAT(p1.ChTm,'00:0#'), ',',
                        CAST(p1.StcInc + p1.Bal - (p1.ShpRsv + p1.ShpRsvIn) - p1.PicNotR AS int)
                    ) AS debug,
                    CASE WHEN p2.Gr2 = 4 THEN 0 ELSE CASE WHEN p1.ChUsr IN ('plocklisteanv','test') AND p1.ChDt = %d AND p1.ChTm > %d THEN 0 ELSE 1 END END AS gen_err
                FROM StcBal p1
                JOIN Prod p2 ON p2.ProdNo = p1.ProdNo
                WHERE
                    p1.StcNo IN (%s)
                    AND p2.Gr7 = 1
                    AND p2.Gr2 BETWEEN 2 AND 4
                    %s
                ORDER BY p1.ProdNo, p1.StcNo",
                intval(date('Ymd')),
                115,
                $stcbal,
                $limit
            );


            $rawData = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery1);
            if (empty($rawData)) {
                Mage::log("Fond no stock updates.", Zend_Log::DEBUG);
                return;
            }

            if ($this->getDebugMode()) {
                file_put_contents($specialDebugFilename, "ProdNo,Descr,StcNo,Gr2,StcInc,Bal,ShpRsv,PicNotR,InO,ChUsr,ChDt,ChTm,x2,err\n", FILE_APPEND);
            }

            $rawData[] = false; // Add one last element that marks the end of data
            $previousSku = null;
            $stockErrors = [];
            $tmpStockData = [];
            $previousSkuError = false;
            $potentialError = [];
            $vismaStockData = [];

            foreach ($rawData as $rawDataRow) {

                // Check to se that we didn't reached the end
                if ($rawDataRow !== false) {
                    $sku = $rawDataRow['sku'];
                    $stockId = intval($rawDataRow['StcNo']);
                } else {
                    $sku = null;
                }

                // Check to se if we got new SKU
                if ($sku !== $previousSku) {

                    if ($previousSku !== null) {
                        // Save tmpStockData
                        $vismaStockData[] = $tmpStockData;
                    }

                    // If we had errors, save that for later
                    if ($previousSkuError) {
                        $stockErrors = array_merge($stockErrors, $potentialError);
                    }

                    // Check if we reached the end
                    if ($sku === null) {
                        break; // De end wass nådd, break out
                    }

                    // Setup for new SKU
                    $tmpStockData = [
                        "sku" => $sku,
                        "qty" => 0
                    ];
                    $previousSku = $sku;
                    $previousSkuError = false;
                    $potentialError = [];
                }

                // Calculate stocka data
                if ($sku === $previousSku) {
                    if ($this->getDebugMode()) {
                        file_put_contents($specialDebugFilename, sprintf("%s,%s\n", $rawDataRow["debug"], $rawDataRow["gen_err"]), FILE_APPEND);
                    }

                    // Only calculate real stock qty for products NOT having status 4 (Slutsåld).
                    if ($rawDataRow['Gr2'] != 4) {
                        // [Reserverat mot lager]
                        $x1 = $rawDataRow['ShpRsv'];

                        // [Reserverbart mot lager]
                        $x2 = $rawDataRow['StcInc'] + $rawDataRow['Bal'] - ($rawDataRow['ShpRsv'] + $rawDataRow['ShpRsvIn']) - $rawDataRow['PicNotR'];

                        // [I order]
                        $x3 = $rawDataRow['InO'] < 0 ? 0 : $rawDataRow['InO'];

                        // [Reserverat mot lager] (x1) => ShpRsv
                        // [Reserverbart mot lager] (x2) => [Realiserat saldo] + [Orealiserad lagerökn.] - [Reserverad] - [Plockat utan reservering]
                        // [Realiserat saldo] => Bal
                        // [Orealiserad lagerökn.] => StcInc
                        // [Reserverad] => [Reserverat mot lager] + [Reserverat mot ökning]
                        // [Reserverat mot lager] => ShpRsv
                        // [Reserverat mot ökning] => ShpRsvIn
                        // [Plockat utan reservering] PicNotR
                        // [I order] (x3) => InO

                        switch ($stockId) {
                            case 1:
                            case 11:
                                // Different calculations based on $x1 and $x3
                                if ($x1 > $x3) {
                                    // [Reserverbart mot lager]
                                    $qty = $x2;
                                } else {
                                    // [Reserverbart mot lager] + ( [Reserverat mot lager] - [I order] )
                                    $qty = $x2 + ( $x1 - $x3 );
                                }
                                $tmpStockData['qty'] += $qty;
                                break;

                            case 3:
                            case 13:
                                // [Reserverat mot lager] + [Reserverbart mot lager]
                                $qty = $x1 + $x2;
                                $tmpStockData['qty'] += $qty;
                                break;

                            default:
                                // We got an unknow stock id!
                                $qty = '-';
                                $previousSkuError = true;
                                break;
                        }
                    } else {
                        // Always set stock qty to zero for products with status 4 (Slutsåld).
                        $qty = 0;
                        $tmpStockData['qty'] += $qty;
                    }
                    $potentialError[] = "{$rawDataRow['debug']},{$rawDataRow['gen_err']},{$qty}";

                    if (!empty($rawDataRow['gen_err'])) {
                        // Save error
                        $previousSkuError = true;
                    }
                }
            }

            // Check to se if we got errors to report
            if (!empty($stockErrors)) {
                // Send email with all errors
                $subject = "Stock sync status";
                $body = "The following products did not match filter for succsessfull regeneration in Visma\n\n";
                $body .= "ProdNo,Descr,StcNo,Gr2,StcInc,Bal,ShpRsv,PicNotR,InO,ChUsr,ChDt,ChTm,x2,err\n";
                if ($this->getDebugMode()) {
                    file_put_contents($specialDebugErrorFilename, "ProdNo,Descr,StcNo,Gr2,StcInc,Bal(x3),ShpRsv(x1),PicNotR,InO,ChUsr,ChDt,ChTm,x2,err\n", FILE_APPEND);
                }
                foreach ($stockErrors as $errRow) {
                    $body .= "{$errRow}\n";
                    if ($this->getDebugMode()) {
                        file_put_contents($specialDebugErrorFilename, $errRow . "\n", FILE_APPEND);
                    }
                }
                Mage::helper($this->_defaultHelper)->sendAdminEmail($subject, $body, ['it','stock']);
            }

            if (empty($vismaStockData)) {
                Mage::log("Found raw stock data but no updates was generated!", Zend_Log::ERR);
                return;
            }

            // Begin: Process stock override
            $sqlQuery2 = "SELECT sku, qty, delete_on_next_run FROM stock_override WHERE warehouse_id = :warehouseId";
            $overrideData = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchAssoc($sqlQuery2, [ "warehouseId" => $warehouseId ]);

            if (!empty($overrideData)) {
                foreach ($vismaStockData as $index => $row) {
                    $sku = $row["sku"];
                    if (array_key_exists($sku, $overrideData)) {
                        if (empty($overrideData[$sku]["delete_on_next_run"])) {
                            $newQty = intval($row["qty"]) + intval($overrideData[$sku]["qty"]);
                            $vismaStockData[$index]["qty"] = $newQty;
                            Mage::log(
                                sprintf(
                                    "Overriding stock quantity for [%s]. %d + %d = %d",
                                    $sku,
                                    $row["qty"],
                                    $overrideData[$sku]["qty"],
                                    $newQty
                                ),
                                Zend_Log::DEBUG
                            );
                        }
                        unset($overrideData[$sku]);
                    }
                }

                // If we still have override data, we need to do some more processing
                if (!empty($overrideData)) {
                    $deleteStock = [];
                    foreach ($overrideData as $sku => $row) {
                        if (empty($row["delete_on_next_run"])) {
                            // We need to create stock data for overridden items that don't have stock data from Visma
                            $vismaStockData[] = [
                                "sku" => $sku,
                                "qty" => $row["qty"]
                            ];
                        } else {
                            // Save sku for later deletion
                            $deleteStock[] = $sku;
                        }
                    }

                    // We need to delete real stock data for overridden products, where override data is to be deleted.
                    if (!empty($deleteStock)) {
                        Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->delete(["external_stock", "stock"], [ "sku IN (?)" => $deleteStock, "warehouse_id = ?" => $warehouseId]);
                    }
                }

                // Delete override data marked for deletion
                Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->delete(["external_stock", "stock_override"], [ "warehouse_id = ?" => $warehouseId, "delete_on_next_run > ?" => 0]);
            }
            // End: Process stock override

            $sqlValues = [];
            $maxIterations = 50;
            $iteration = 0;
            $batch = 1;

            // Split updates in batches
            foreach ($vismaStockData as $stockData) {
                $sku = $stockData['sku'];
                $qty = $stockData['qty'] < 0 ? 0 : $stockData['qty'];

                $sqlValues[$batch][$iteration++] = sprintf(
                    "('%s',%d,0,0,%d,NOW(),NOW())",
                    $sku,
                    $warehouseId,
                    $qty
                );

                if ($this->getDebugMode()) {
                    file_put_contents($specialDebugUpdatesFilename, sprintf("%s,%d,%d\n", $sku, $stockData["qty"], $qty), FILE_APPEND);
                }

                if ($iteration === $maxIterations) {
                    $iteration = 0;
                    $batch++;
                }
            }

            if (empty($sqlValues)) {
                Mage::log("Stock updates was generated but no batch was created!", Zend_Log::ERR);
                return;
            }

            if ($truncateStock) {
                Mage::log("Truncating stock data.", Zend_Log::DEBUG);
                Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->exec("TRUNCATE stock");
            }

            $batchCount = count($sqlValues);
            for ($batch = 1; $batch <= $batchCount; $batch++) {
                $wantedRows = count($sqlValues[$batch]);
                $sqlUpdate = "INSERT INTO stock (sku,warehouse_id,magento_id,store_id,qty,changed_at,synced_at) VALUES ";
                $sqlUpdate .= implode(',', $sqlValues[$batch]);
                $sqlUpdate .= " ON DUPLICATE KEY UPDATE qty = VALUES(qty), changed_at = VALUES(changed_at), synced_at = VALUES(synced_at)";

                Mage::log(sprintf("Updating %d rows of stock_data, batch %d of %d.", $wantedRows, $batch, $batchCount), Zend_Log::DEBUG);

                $affectedRows = Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->exec($sqlUpdate);
                // Skip this "error" logging for now because it's useless (see below)
                /*
                  if($affectedRows < $wantedRows) {
                  // One problem here: Affected rows can be between 0 and double the amount of rows.
                  // If the "on duplicate key" is triggerd, the affected rows are 2, not 1.
                  $this->getHelper()->log(LOG_ERR, "Should have updated {$wantedRows} rows but only updated {$affectedRows}.");
                  }
                 */

                // Just log the amount of updates the query generated
                Mage::log(sprintf("Actual updated rows was: %d.", $affectedRows), Zend_Log::DEBUG);
            }
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while syncing stock data from Visma!");
        }
    }

    public function updateProducts()
    {
        $startTime = time();
        Mage::log("Updating products from table 'product_update'", Zend_Log::DEBUG);

        $updates = [];
        $totalUpdatedQty = 0;
        $warehouseId = intval(Mage::getStoreConfig("integration/external_stock/warehouse_id"));
        $magentoId = intval(Mage::getStoreConfig("integration/general/magento_instance"));

        try {
            // Fetch all products tagged to be updated
            $sqlQuery1 = "
                SELECT
                    pu.product_id,
                    pu.sku,
                    SUM(s.qty) AS qty
                FROM product_update pu
                JOIN stock s ON s.sku = pu.sku AND s.warehouse_id = pu.warehouse_id AND (s.magento_id = 0 OR s.magento_id = pu.warehouse_id)
                WHERE pu.magento_id = :magentoId AND pu.warehouse_id = :warehouseId AND pu.tagged_at IS NOT NULL
                GROUP BY pu.sku
                ORDER BY pu.tagged_at
            ";
            $params1 = [
                "warehouseId" => $warehouseId,
                "magentoId" => $magentoId
            ];
            $externalStockRows = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchAll($sqlQuery1, $params1);

            $rowQty = count($externalStockRows);
            if (!$rowQty) {
                return;
            }

            // Ok, we got products in need of stock updates, now initialize queries and variables
            $sqlQuery2a = "
                SELECT
                    p.sku,
                    s.product_id,
                    min_qty,
                    s.qty
                FROM catalog_product_entity p
                JOIN cataloginventory_stock_item s ON s.product_id = p.entity_id
                WHERE
                    s.use_config_manage_stock = 0
                    AND s.manage_stock = 1
                    AND p.sku IN (%ph)
                GROUP BY p.sku
            ";
            $sqlQuery3a = "UPDATE product_update SET tagged_at = NULL WHERE warehouse_id = :warehouseId AND magento_id = :magentoId AND product_id IN (%ph)";
            $batch = 0;
            $batchSize = Awardit_Integration_Helper_Data::STOCK_UPDATE_BATCH_SIZE;
            $placeholders = [];
            $index = 0;

            // Do this until we processed all stock rows
            do {

                // Remember $stockRows index and its sku
                $placeholders[ $externalStockRows[$index]["sku"] ] = $index;
                $index++;

                // If we have reached batchSize OR if we have done all stock rows
                if ($index % $batchSize === 0 || $index >= $rowQty) {

                    // Prepare query to fetch product data from local Magento instance
                    $sqlQuery2b = str_replace ("%ph", ":ph" . implode(", :ph", $placeholders), $sqlQuery2a);
                    $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_read")->prepare($sqlQuery2b);
                    foreach ($placeholders as $sku => $idx) {
                        $stmt1->bindValue("ph{$idx}", $sku);
                    }
                    $stmt1->execute();
                    $result = $stmt1->fetchAll();
                    $updates = [];

                    // Parse result and update stock in local Magento instance
                    foreach ($result as $row) {

                        // Filter out famous "this should not happen" values
                        if (!array_key_exists($row["sku"], $placeholders) || !array_key_exists($placeholders[ $row["sku"] ], $externalStockRows)) {
                            continue;
                        }

                        // Update local stock only if needed
                        if ($row["qty"] != $externalStockRows[ $placeholders[ $row["sku"] ] ]["qty"]) {

                            $status = $this->forceUpdateLocalStock([
                                "product_id" => $row["product_id"],
                                "external_qty" => intval($externalStockRows[ $placeholders[ $row["sku"] ] ]["qty"]),
                                "min_qty" => intval($row["min_qty"]),
                                "sku" => $row["sku"],
                                "qty" => intval($row["qty"]),
                            ]);

                            if ($status) {

                                // If update was successfull, sleep for a while and remember product_id for later
                                usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                                $updates[] = $externalStockRows[ $placeholders[ $row["sku"] ] ]["product_id"];
                                $totalUpdatedQty++;

                            }
                        } else {
                            // No update is needed, but we still want to remember product_id for later
                            $updates[] = $externalStockRows[ $placeholders[ $row["sku"] ] ]["product_id"];
                        }

                    }

                    // Clear 'tagged_at' in table 'product_update' for remembered products
                    if (!empty($updates)) {

                        $sqlQuery3b = str_replace ("%ph", ":ph" . implode(", :ph", array_keys($updates)), $sqlQuery3a);
                        $stmt2 = Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->prepare($sqlQuery3b);
                        $stmt2->bindValue("warehouseId", $warehouseId);
                        $stmt2->bindValue("magentoId", $magentoId);
                        foreach ($updates as $idx => $productId) {
                            $stmt2->bindValue("ph{$idx}", intval($productId));
                        }
                        $stmt2->execute();
                        $affectedRows = $stmt2->rowCount();
                        if ($affectedRows > 0) {
                            usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                        }

                    }

                    // Increment and reset
                    $batch++;
                    $placeholders = [];

                }

                // Continue until we reached end of stock rows
            } while($index < $rowQty);

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating local stock for products tagged to be updated!");
        }

        // Update products set to not handle stock to always be in stock
        if (false): // Skip this stage for now
        try {

            $sqlUpdate4a = "
                UPDATE cataloginventory_stock_status css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = 1, css.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND css.stock_status = 0
            ";
            $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4a);
            $stmt1->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt1->execute();
            $updatedRows = $stmt1->rowCount();

            $sqlUpdate4b = "
                UPDATE cataloginventory_stock_status_idx css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = 1, css.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND css.stock_status = 0
            ";
            $stmt2 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4b);
            $stmt2->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt2->execute();
            $updatedRows += $stmt2->rowCount();

            $sqlUpdate4c = "
                UPDATE cataloginventory_stock_item csi
                SET csi.is_in_stock = 1, csi.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND csi.is_in_stock = 0
            ";
            $stmt3 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4c);
            $stmt3->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt3->execute();
            $updatedRows += $stmt3->rowCount();

            if ($updatedRows > 0) {
                usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                Mage::log("Updated {$updatedRows} rows for products that were set to be out-of-stock even though they are set to not handle stock.", Zend_Log::DEBUG);
            }

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating products to always be in stock when set to not handle stock!");
        }
        endif;

        // Do reindexing after successfull product sync and/or successfull stock update
        if (!empty($externalStockRows)) {
            $this->doReindex();
        }

        $runningTime = time() - $startTime;
        $hh = floor($runningTime / 3600);
        $timeLeft = $runningTime % 3600;
        $mm = floor($timeLeft / 60);
        $ss = $timeLeft % 60;
        Mage::log("Updated {$totalUpdatedQty} products in {$hh} hours, {$mm} minutes and {$ss} seconds.", Zend_Log::DEBUG);

    }

    public function updateLocalStock($stockRow)
    {
        try {
            $productId = $stockRow["product_id"];
            $isInStock = $stockRow["external_qty"] >= $stockRow["min_qty"] ? Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK : Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK;

            $sqlQuery1 = "
                UPDATE cataloginventory_stock_status css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = :isInStock, css.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery1);
            $stmt1->bindValue("qty", $stockRow["external_qty"]);
            $stmt1->bindValue("isInStock", $isInStock);
            $stmt1->bindValue("productId", $productId);
            $stmt1->execute();
            $updatedRows = $stmt1->rowCount();

            $sqlQuery2 = "
                UPDATE cataloginventory_stock_status_idx css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = :isInStock, css.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt2 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery2);
            $stmt2->bindValue("qty", $stockRow["external_qty"]);
            $stmt2->bindValue("isInStock", $isInStock);
            $stmt2->bindValue("productId", $productId);
            $stmt2->execute();
            $updatedRows += $stmt2->rowCount();

            $sqlQuery3 = "
                UPDATE cataloginventory_stock_item csi
                SET csi.is_in_stock = :isInStock, csi.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt3 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery3);
            $stmt3->bindValue("qty", $stockRow["external_qty"]);
            $stmt3->bindValue("isInStock", $isInStock);
            $stmt3->bindValue("productId", $productId);
            $stmt3->execute();
            $updatedRows += $stmt3->rowCount();

            if (!empty($updatedRows)) {
                Mage::log("Product [{$stockRow["sku"]}] was updated (updated rows: {$updatedRows}), changed local stock from {$stockRow["qty"]} to {$stockRow["external_qty"]}", Zend_Log::DEBUG);
            }
            return true;

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception in syncIsInStock()!");
            return false;
        }
    }

    public function forceUpdateLocalStock($stockRow)
    {
        try {
            $stock = Mage::getModel("catalog/product")->load($stockRow["product_id"])->getStockItem();

            $stock->setQty($stockRow["external_qty"]);
            $stock->setIsInStock($stockRow["external_qty"] >= $stockRow["min_qty"] ? Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK : Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK);
            $stock->forceSave();

            Mage::log("Product [{$stockRow["sku"]}] was updated, changed local stock from {$stockRow["qty"]} to {$stockRow["external_qty"]}", Zend_Log::DEBUG);
            return true;

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating stock for product [{$stockRow["sku"]}]!");
            return false;
        }
    }

    public function doReindex()
    {
        $indexCodes = [
            "cataloginventory_stock" => true,
            "catalog_product_price" => false,
            "catalog_product_flat" => false
        ];

        try {
            foreach ($indexCodes as $indexCode => $indexStatus) {
                if ($indexStatus) {
                    Mage::log("Running reindexing process for {$indexCode}", Zend_Log::DEBUG);
                    $process = Mage::getModel("index/indexer")->getProcessByCode($indexCode);
                    $process->reindexAll();
                }
            }
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception in doReindex() where indexCode = '{$indexCode}'!");
            return false;
        }
    }

}
