<?php
class Awardit_Integration_Model_Cli_Cron extends Awardit_Integration_Model_Cli {

    /**
     * @var int
     */
    const CRON_DEFAULT_ITERATION_LIMIT = 999;

    /**
     * @var string
     */
    const CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT = "+30 minutes";     // Time limit for importing products

    /**
     * @var string
     */
    const CRON_MAX_RUNNING_TIME_ORDER_PROCESSIING = "+5 minutes";   // Time limit for processing orders

    protected $_defaultProductImportModel = "integration/Cli_ProductImport";
    protected $_defaultOrderExportModel = "integration/Cli_OrderExport";
    protected $_maxRunningTime = null;

    /**
     * @return int
     */
    public function getLimit()
    {
        if (parent::getLimit() != Awardit_Integration_Model_Cli::CLI_DEFAULT_ITERATION_LIMIT) {
            return parent::getLimit();
        } else {
            return Awardit_Integration_Model_Cli_Cron::CRON_DEFAULT_ITERATION_LIMIT;
        }
    }

    public function setMaxRunningTime($offset)
    {
        if (!empty($offset)) {
            $this->_maxRunningTime = new DateTime(date("Y-m-d H:i:s", strtotime($offset)));
        } else {
            $this->_maxRunningTime = null;;
        }
    }

    public function getMaxRunningTime()
    {
        return $this->_maxRunningTime;
    }

    // Return true while max running time is not reached OR no max running time is specified
    public function checkRunningTime($now = null)
    {
        if ($now === null) {
            $now = new DateTime("now");
        }
        if ($this->_maxRunningTime !== null) {
            return $now < $this->_maxRunningTime;
        }
        return true;
    }

    // -m=Cron -f=importProducts
    public function CLI_importProducts($param)
    {
        $start = new DateTime("now");
        Mage::helper($this->_defaultHelper)->log("Cron: Starting to import updated products.");

        $wantedParams = [ "updateStock" => "int" ];
        $extractedParams = $this->extractParams($param, $wantedParams);
        $updateStock = null;
        if (array_key_exists("updateStock", $extractedParams)) {
            $updateStock = empty($extractedParams["updateStock"]) ? false : true;
        }

        $processes = [
            "tag" => true,
            "import" => true,
        ];
        $this->_processProducts($processes, $updateStock);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: product script ran for {$diff->format("%H:%I:%S")}\n");
    }

    // -m=Cron -f=prepareOrders
    public function CLI_scanOrders($param)
    {
        $start = new DateTime("now");
        Mage::helper($this->_defaultHelper)->log("Cron: Starting to scan for new orders.");
        $this->setMaxRunningTime(null);

        $processes = [
            "scan" => true,
            "prepare" => false,
            "update" => false
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: Order script ran for {$diff->format("%H:%I:%S")}\n");

    }

    // -m=Cron -f=prepareOrders
    public function CLI_prepareOrders($param)
    {
        $start = new DateTime("now");
        Mage::helper($this->_defaultHelper)->log("Cron: Starting to prepare all orders.");
        $this->setMaxRunningTime(null);

        $processes = [
            "scan" => true,
            "prepare" => true,
            "update" => false
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: Order script ran for {$diff->format("%H:%I:%S")}\n");

    }

    // -m=Cron -f=updateOrders
    public function CLI_updateOrders($param)
    {
        $start = new DateTime("now");
        Mage::helper($this->_defaultHelper)->log("Cron: Starting to update all orders.");
        $this->setMaxRunningTime(null);

        $wantedParams = [
            "fromStatus" => "int",
            "toStatus" => "int",
            "status" => "int"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["fromStatus"])) {
            $this->setExportStatusFrom($extractedParams["fromStatus"]);
        }
        if (!empty($extractedParams["toStatus"])) {
            $this->setExportStatusTo($extractedParams["toStatus"]);
        }
        if (!empty($extractedParams["status"])) {
            $this->setExportStatusFrom($extractedParams["status"]);
            $this->setExportStatusTo($extractedParams["status"]);
        }

        $processes = [
            "scan" => false,
            "prepare" => false,
            "update" => true
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: Order script ran for {$diff->format("%H:%I:%S")}\n");

    }

    // -m=Cron -f=processOrders
    public function CLI_processOrders($param)
    {
        $start = new DateTime("now");
        Mage::helper($this->_defaultHelper)->log("Cron: Starting to process all orders.");
        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_ORDER_PROCESSIING);

        $wantedParams = [
            "fromStatus" => "int",
            "toStatus" => "int",
            "status" => "int"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["fromStatus"])) {
            $this->setExportStatusFrom($extractedParams["fromStatus"]);
        }
        if (!empty($extractedParams["toStatus"])) {
            $this->setExportStatusTo($extractedParams["toStatus"]);
        }
        if (!empty($extractedParams["status"])) {
            $this->setExportStatusFrom($extractedParams["status"]);
            $this->setExportStatusTo($extractedParams["status"]);
        }

        $processes = [
            "scan" => true,
            "prepare" => true,
            "update" => true
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: Order script ran for {$diff->format("%H:%I:%S")}\n");

    }

    // -m=Cron -f=exportOrders
    public function CLI_exportOrders($param)
    {
        Mage::helper($this->_defaultHelper)->log("Cron: this function is deprecated!");
    }

    // -m=Cron -f=timedPricesInside
    public function CLI_timedPricesInside($param)
    {
        Mage::helper($this->_defaultHelper)->log("Cron: Starting timed prices (inside) processing.");
        $this->_processTimedPrices("inside");
    }

    // -m=Cron -f=timedPricesOutside
    public function CLI_timedPricesOutside($param)
    {
        Mage::helper($this->_defaultHelper)->log("Cron: Starting timed prices (outside) processing.");
        $this->_processTimedPrices("outside");
    }

    protected function _processProducts($processes, $updateStock)
    {
        if (empty($processes)) {
            return;
        }

        $model = Mage::getModel("integration/Cli_LocalProductImport") ?: Mage::getModel($this->_defaultProductImportModel);
        $model->setLimit($this->getLimit());
        $model->setDebugMode($this->getDebugMode());
        if ($updateStock !== null) {
            $model->setUpdateStock($updateStock);
        }

        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT);
        $taggedProducts = 0;
        $importedProducts = 0;

        if (!empty($processes["tag"])) {
            $productList = $model->getAllUpdatedProducts();
            $model->tagProductsForImport($productList);
            $taggedProducts = count($productList);
            Mage::helper($this->_defaultHelper)->log("Tagged {$taggedProducts} products for import.");
        }

        if (!empty($processes["import"])) {
            do {
                $integrationProduct = $model->getProductToImport();
                if (empty($integrationProduct)) {
                    break;
                }

                $model->mergeDataToProduct($integrationProduct);
                $model->updateIntegrationData($integrationProduct);
                if ($integrationProduct["import_status"] == Awardit_Integration_Model_Cli_ProductImport::IMPORT_STATUS_FETCHED) {
                    $model->importSingleProduct($integrationProduct);
                    $model->updateIntegrationData($integrationProduct);
                }

                if ($this->getLimit() > 0) {
                    if ($importedProducts++ >= $this->getLimit()) {
                        Mage::helper($this->_defaultHelper)->log("Limit reached, exiting.");
                        break;
                    }
                }

            } while ($this->checkRunningTime());

            if (!$this->checkRunningTime()) {
                Mage::helper($this->_defaultHelper)->log("Time limit reached, exiting.");
            }
        }

        Mage::helper($this->_defaultHelper)->log("Imported {$importedProducts} products.");

    }

    protected function _processOrders($processes)
    {
        if (empty($processes)) {
            return;
        }

        $model = Mage::getModel("integration/Cli_LocalOrderExport") ?: Mage::getModel($this->_defaultOrderExportModel);
        $model->setLimit($this->getLimit());
        $model->setDebugMode($this->getDebugMode());
        $model->setExportStatusFrom($this->getExportStatusFrom());
        $model->setExportStatusTo($this->getExportStatusTo());

        $preparedOrders = 0;
        $updatedOrders = 0;
        $iterations = 0;

        if (!empty($processes["scan"])) {
            $model->scanForNewOrders();
        }

        if (!empty($processes["prepare"])) {
            $integrationOrders = $model->getAllOrdersToExport();

            if (!empty($integrationOrders)) {
                $maxQtyToExport = count($integrationOrders);

                Mage::helper($this->_defaultHelper)->log("Found {$maxQtyToExport} order to prepare.");

                foreach ($integrationOrders as $integrationOrder) {
                    $model->prepareSingleOrder($integrationOrder);

                    $preparedOrders++;
                    if ($this->getLimit() > 0 && $preparedOrders >= $this->getLimit()) {
                        Mage::helper($this->_defaultHelper)->log("Limit reached, exiting.");
                        break;
                    }

                    if (!$this->checkRunningTime()) {
                        break;
                    }
                }
            }
        }

        if (!empty($processes["update"]) && $this->checkRunningTime()) {

            // Update from EXPORT_STATUS_TRANSFERRED1 to EXPORT_STATUS_TRANSFERRED2
            $model->updateIntegrationOrderStatusSpecial();

            $ordersToUpdate = $model->getAllOrdersToUpdate();
            if (!empty($ordersToUpdate)) {
                $maxQtyToUpdate = count($ordersToUpdate);
                Mage::helper($this->_defaultHelper)->log("Found {$maxQtyToUpdate} orders to update.");
                foreach ($ordersToUpdate as $integrationOrder) {
                    $model->updateSingleOrder($integrationOrder);
                    $updatedOrders += $model->getWasLastOrderUpdated() ? 1 : 0;
                    $iterations++;
                    if ($this->getLimit() > 0) {
                        if ($iterations >= $this->getLimit()) {
                            Mage::helper($this->_defaultHelper)->log("Limit reached, exiting.");
                            break;
                        }
                    }

                    if (!$this->checkRunningTime()) {
                        break;
                    }
                }
            }
        }

        if (!$this->checkRunningTime()) {
            Mage::helper($this->_defaultHelper)->log("Time limit reached, exiting.");
        }

        if (!empty($preparedOrders) && !empty($updatedOrders)) {
            Mage::helper($this->_defaultHelper)->log("Prepared {$preparedOrders} and updated {$updatedOrders} orders.");
        } elseif(!empty($preparedOrders)) {
            Mage::helper($this->_defaultHelper)->log("Prepared {$preparedOrders} orders.");
        } elseif(!empty($updatedOrders)) {
            Mage::helper($this->_defaultHelper)->log("Updated {$updatedOrders} orders.");
        } else {
            Mage::helper($this->_defaultHelper)->log("Prepared and updated no orders.");
        }

    }

    protected function _processTimedPrices($mode)
    {
        $start = new DateTime("now");
        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT);

        $model = $this->getTimedPricesModel();
        if ($mode == "inside") {
            $model->setDoInside(true);
            $model->setDoOutside(false);
            $model->setDateAdjustment("tomorrow 00:01:00");
        } elseif ($mode == "outside") {
            $model->setDoInside(true);
            $model->setDoOutside(true);
        } else {
            throw new Exception("Unknown mode: {$mode}");
        }

        $model->processTimedPrices();
        Mage::helper($this->_defaultHelper)->log("Cron: Running reindexing process.");
        $model->doReindex();

        $end = new DateTime("now");
        $diff = $start->diff($end);
        Mage::helper($this->_defaultHelper)->log("Cron: Timed prices processing script ran for {$diff->format("%H:%I:%S")}\n");

    }

}
