<?php

class Awardit_Integration_Model_Cli_PointsCoreOrderExport extends Awardit_Integration_Model_Cli_OrderExport {

    protected $_defaultHelper = "integration/PointsCore";
    protected $_includePointsTax = true;

    /*
     * Inherited actions:
     *
     * -m=PointsCoreOrderExport -f=scanForNewOrders
     * -m=PointsCoreOrderExport -f=prepareSingleOrder -p="integrationId:<integrationId>|incrementId:<incrementId>"
     * -m=PointsCoreOrderExport -f=prepareAllOrders
     * -m=PointsCoreOrderExport -f=transferAllOrders
     * -m=PointsCoreOrderExport -f=updateSingleOrder -p="integrationId:<integrationId>|incrementId:<incrementId>"
     * -m=PointsCoreOrderExport -f=updateAllOrders
     * -m=PointsCoreOrderExport -f=testXML -p="integrationId:<integrationId>|incrementId:<incrementId>"
     *
     */

    // -m=PointsCoreOrderExport -f=debugExportOrders
    public function CLI_debugExportOrders($param)
    {
        $this->setLimit(Awardit_Integration_Model_Cli_Cron::CRON_DEFAULT_ITERATION_LIMIT);

        $integrationOrders = $this->getOrdersByAttribute("export_status", self::EXPORT_STATUS_TRANSFERRED1, false);

        if (empty($integrationOrders)) {
            Mage::helper($this->_defaultHelper)->log("Found no orders to debug.");
            return true;
        }

        $maxQtyToExport = count($integrationOrders);
        Mage::helper($this->_defaultHelper)->log("Found {$maxQtyToExport} orders to debug.");

        $dir = Awardit_Integration_Helper_Data::$basePath . "debug" . DS . date("Y-m");
        if (!is_dir($dir) && !mkdir($dir, 0777, true)) {
            Mage::helper($this->_defaultHelper)->log("Unable to create directory {$dir}", LOG_ERR);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Unable to create directory {$dir}",
                "Check directory permissions!",
                ["it"]
            );
            return false;
        }

        foreach ($integrationOrders as $integrationOrder) {
            $orderData = $this->getOrderData($integrationOrder);
            $xmlData = $this->createXmlData($orderData);

            $dom = new DOMDocument("1.0", "UTF-8");
            $dom->formatOutput = true;
            $dom->loadXML(utf8_encode(simplexml_load_string($xmlData->asXML(), null, LIBXML_NOCDATA)->asXML()));

            $filename = "{$orderData["increment_id"]}.{$orderData["visma_order_db"]}.xml";
            $dom->save($dir . DS . $filename);
        }

    }

    protected function _addPaymentData(&$orderData, $order)
    {
        $status = parent::_addPaymentData($orderData, $order);

        if ($status && $orderData["xmlData"]["order_head"]["Ord_Inf4"] == "Gratis") {
            if ($order->getData("points_points") > 0) {
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Poäng";
            }
        }

        return $status;
    }

    protected function _addPriceData(&$orderData, $order)
    {
        $status = parent::_addPriceData($orderData, $order);

        if (!$status) {
            return $status;
        }

        /*
        Query to get values:

        SELECT
            tmp.*,
            tmp.Shipping_OrdLn_Free1 - (tmp.Discount_OrdLn_Free1 * (tmp.Shipping_OrdLn_Free1 / (tmp.Shipping_OrdLn_Free1 + tmp.Item_OrdLn_Free1))) AS Correct_Shipping_OrdLn_Free1,
            tmp.Item_OrdLn_Free1 - (tmp.Discount_OrdLn_Free1 * (tmp.Item_OrdLn_Free1 / (tmp.Shipping_OrdLn_Free1 + tmp.Item_OrdLn_Free1))) AS Correct_Item_OrdLn_Free1,
            tmp.Shipping_OrdLn_Free1 + Item_OrdLn_Free1 - Discount_OrdLn_Free1 AS total_points_reference,
            (tmp.Shipping_OrdLn_Free1 - (tmp.Discount_OrdLn_Free1 * (tmp.Shipping_OrdLn_Free1 / (tmp.Shipping_OrdLn_Free1 + tmp.Item_OrdLn_Free1)))) + (tmp.Item_OrdLn_Free1 - (tmp.Discount_OrdLn_Free1 * (tmp.Item_OrdLn_Free1 / (tmp.Shipping_OrdLn_Free1 + tmp.Item_OrdLn_Free1)))) AS valculated_total_points
        FROM (
            SELECT
                # Order
                sfo.increment_id,

                # Shipping
                sfo.base_shipping_incl_tax / ((100 + sfoi.tax_percent) / 100) AS Shipping_OrdLn_Price,
                (sfo.base_shipping_amount + sfo.base_shipping_tax_amount) - (sfo.base_points_shipping_total + sfo.base_points_shipping_tax_total) AS Shipping_OrdLn_Free3,
                sfo.points_shipping_points + sfo.points_shipping_tax_points AS Shipping_OrdLn_Free1,

                # Discounts
                sfo.base_discount_amount / ((100 + sfoi.tax_percent) / 100) AS Discount_OrdLn_Price,
                sfo.base_discount_amount + (sfo.base_points_discount_total + sfo.base_points_discount_tax_total) AS Discount_OrdLn_Free3,
                -(sfo.points_discount_points + sfo.points_discount_tax_points) AS Discount_OrdLn_Free1,

                # Items
                sfoi.item_id,
                sfoi.sku,
                sfoi.product_type,
                sfoi.parent_item_id,
                sfoi.tax_percent,
                sfoi.base_original_price / ((100 + sfoi.tax_percent) / 100) AS Item_OrdLn_Price,
                (sfoi.base_row_total + sfoi.base_tax_amount + sfoi.base_hidden_tax_amount) - sfoi.base_discount_amount - (sfoi.base_points_row_total + sfoi.base_points_row_tax_total) AS Item_OrdLn_Free3,
                sfoi.points_row_points + sfoi.points_row_tax_points AS Item_OrdLn_Free1,
                (sfoi.base_points_row_total + sfoi.base_points_row_tax_total) / (sfoi.points_row_points + sfoi.points_row_tax_points) AS points_ratio

            FROM sales_flat_order sfo
            JOIN sales_flat_order_item sfoi ON sfoi.order_id = sfo.entity_id
            WHERE sfo.increment_id = '<icrement_id>'
        ) tmp


        Points core values explained:

            points_xxx_value: Value in points, excluding tax
            points_xxx_tax_value: Value of tax in points
            points_xxx_points: Number of points spent, excluding tax
            points_xxx_tax_points: Number of points spent in tax
            points_xxx_total: Amount of currency the points cover, exlcuding tax
            points_xxx_tax_total: Amount of currency the points cover in tax
            base_points_xxx_total: Amount of currency the points cover, exlcuding tax
            base_points_xxx_tax_total: Amount of currency the points cover in tax

        */

        // Get item data
        $items = $order->getItemsCollection()->getItems();
        $haveChildren = false;
        foreach ($items as $item) {
            $itemId = intval($item->getData("item_id"));
            $product = Mage::getModel("catalog/product")->setStoreId($order->getStoreId())->load($item->getProductId());

            $orderData["price_data"]["items"][$itemId]["original_points"] = $item->getPointsRowValue() + ($this->_includePointsTax ? $item->getPointsRowTaxValue() : 0);
            $orderData["price_data"]["items"][$itemId]["row_total_points"] = $item->getPointsRowPoints() + ($this->_includePointsTax ? $item->getPointsRowTaxPoints() : 0);

            // We need to recalculate row_total_money because points may have been used
            if (array_key_exists("row_total_money", $orderData["price_data"]["items"][$itemId])) {
                // If item is a parent without children, i.e. NOT config or bundle, remove points value in money from row_total_money
                $orderData["price_data"]["items"][$itemId]["row_total_money"] -= ($item->getBasePointsRowTotal() + $item->getBasePointsRowTaxTotal());
            } else {
                // If item is a parent with children, i.e. config or bundle, save points value in money for later use
                $orderData["price_data"]["items"][$itemId]["row_total_money"] = ($item->getBasePointsRowTotal() + $item->getBasePointsRowTaxTotal());
            }

            // If item have parent, calculate child data
            $parentId = intval($item->getData("parent_item_id"));
            if ($parentId > 0) {
                $haveChildren = true;
                if (empty($orderData["price_data"]["items"][$parentId]["children"])) {
                    $orderData["price_data"]["items"][$parentId]["children"] = [];
                    $orderData["price_data"]["items"][$parentId]["child_points_sum"] = 0;
                    $orderData["price_data"]["items"][$parentId]["child_qty"] = 0;
                }
                $orderData["price_data"]["items"][$parentId]["children"][] = $itemId;
                $orderData["price_data"]["items"][$parentId]["child_qty"] += $item->getQtyOrdered();
                $orderData["price_data"]["items"][$parentId]["child_points_sum"] += ($item->getQtyOrdered() * $orderData["price_data"]["items"][$itemId]["original_points"]);
            }

        }

        // If there are children, calculate prices for them and remove parent
        if ($haveChildren) {
            foreach (array_keys($orderData["price_data"]["items"]) as $itemId) {
                if (!empty($orderData["price_data"]["items"][$itemId]["children"])) {
                    foreach ($orderData["price_data"]["items"][$itemId]["children"] as $childId) {

                        // If sum of child points > 0, use points to calculate ratio
                        if ($orderData["price_data"]["items"][$itemId]["child_points_sum"] > 0) {
                            $orderData["price_data"]["items"][$childId]["child_points_rate"] = ($orderData["price_data"]["items"][$childId]["original_points"] * $orderData["price_data"]["items"][$childId]["qty"]) / $orderData["price_data"]["items"][$itemId]["child_points_sum"];

                        // If no sum > 0 was found, just calculate ratio based on number of children
                        } else {
                            $orderData["price_data"]["items"][$childId]["child_points_rate"] = $orderData["price_data"]["items"][$childId]["qty"] / $orderData["price_data"]["items"][$itemId]["child_qty"];
                        }

                        // Distribute points from parent to child, using previously calculated ratio
                        $orderData["price_data"]["items"][$childId]["row_total_points"] = $orderData["price_data"]["items"][$itemId]["row_total_points"] * $orderData["price_data"]["items"][$childId]["child_points_rate"];

                        // We need to recalculate row_total_money because points may have been used, using previously calculated ratio
                        $orderData["price_data"]["items"][$childId]["row_total_money"] -= ($orderData["price_data"]["items"][$itemId]["row_total_money"] * $orderData["price_data"]["items"][$childId]["child_points_rate"]);

                    }

                    // Remove parent
                    unset($orderData["price_data"]["items"][$itemId]);
                }
            }
        }

        // Calculate shipping
        if ($order->getData("points_shipping_value") > 0) {
            $originalMoney = $order->getData("shipping_incl_tax");
            $originalPoints = $item->getData("points_shipping_value") + $item->getData("points_shipping_tax_value");
            $pointsPaid = $order->getData("points_shipping_points") + $order->getData("points_shipping_tax_points");

            // We need to recalculate money paid because points may have been used
            $pointsInMoney = $order->getBasePointsShippingTotal() + $order->getBasePointsShippingTaxTotal();

            if ($orderData["visma_order_db"] == Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {

                // If order have Sweden as Visma company db, distribute shipping points propotionally for each tax used for physical products
                foreach (array_keys($orderData["price_data"]["taxes"]["physical"]["tax"]) as $taxPercent) {
                    $orderData["price_data"]["shipping"][$taxPercent]["original_price"] =  $originalMoney * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"];
                    $orderData["price_data"]["shipping"][$taxPercent]["original_points"] = $originalPoints * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"];
                    $orderData["price_data"]["shipping"][$taxPercent]["row_total_money"] -= ($pointsInMoney * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"]);
                    $orderData["price_data"]["shipping"][$taxPercent]["row_total_points"] = $pointsPaid * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"];

                }

            } else {

                // If order does NOT have Sweden as Visma company db, use max tax for shipping fee
                $taxData = Mage::helper($this->_defaultHelper)->getMaxTaxForCurrencyAndCountry($orderData["order_currency_id"], $orderData["order_country_id"]);
                $orderData["price_data"]["shipping"][ $taxData["tax_percent"] ]["original_price"] =  $originalMoney;
                $orderData["price_data"]["shipping"][ $taxData["tax_percent"] ]["original_points"] = $originalPoints;
                $orderData["price_data"]["shipping"][ $taxData["tax_percent"] ]["row_total_money"] -= $pointsInMoney;
                $orderData["price_data"]["shipping"][ $taxData["tax_percent"] ]["row_total_points"] = $pointsPaid;

            }
        }

        // Calculate discount
        if ($order->getDiscountAmount() < 0) {

            $orderTotalDiscountMoney = $order->getBasePointsDiscountTotal() + $order->getBasePointsDiscountTaxTotal();
            $orderTotalDiscountPoints = $order->getPointsDiscountPoints() + $order->getPointsDiscountTaxPoints();

            foreach (array_keys($orderData["price_data"]["taxes"]["all"]["tax"]) as $taxPercent) {

                // We need to recalculate money paid because points may have been used
                foreach (array_keys($orderData["price_data"]["discounts"]["rows"]) as $index) {
                    if ($orderData["price_data"]["discounts"]["rows"][$index]["tax_percent"] == $taxPercent) {
                        $orderData["price_data"]["discounts"]["rows"][$index]["row_total_money"] -= (-$orderTotalDiscountMoney * $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["ratio"]);
                        $orderData["price_data"]["discounts"]["rows"][$index]["row_total_points"] = (-$orderTotalDiscountPoints * $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["ratio"]);
                        break;
                    }
                }
            }

            // We need to adjust points spent (OrdLn_Free1) to include discount, for items and shipping

            // Calculate new values for shipping points and money
            foreach (array_keys($orderData["price_data"]["shipping"]) as $taxPercent) {
                $ratio = $orderData["price_data"]["shipping"][$taxPercent]["discount"] / ($orderData["price_data"]["discounts"]["sum"] ?: 1);
                $orderData["price_data"]["shipping"][$taxPercent]["row_total_points"] += ($orderTotalDiscountPoints * $ratio);
                $orderData["price_data"]["shipping"][$taxPercent]["row_total_money"] -= ($orderTotalDiscountMoney * $ratio);
            }

            // Calculate new values for items points and money
            foreach (array_keys($orderData["price_data"]["items"]) as $itemId) {
                $ratio = $orderData["price_data"]["items"][$itemId]["discount"] / ($orderData["price_data"]["discounts"]["sum"] ?: 1);
                $orderData["price_data"]["items"][$itemId]["row_total_points"] += ($orderTotalDiscountPoints * $ratio);
                $orderData["price_data"]["items"][$itemId]["row_total_money"] -= ($orderTotalDiscountMoney * $ratio);
            }
        }

        return true;
    }

    protected function _prepareDiscounts(&$orderData, $order)
    {
        $status = parent::_prepareDiscounts($orderData, $order);

        if ($status) {
            foreach ($orderData["price_data"]["discounts"]["rows"] as $index => $discount) {
                if (!empty($discount["row_total_points"])) {
                    $orderData["xmlData"]["discount"][$index]["OrdLn_Free1"] = $discount["row_total_points"];
                }
            }
        }

        return true;
    }

    protected function _prepareShipment(&$orderData, $order)
    {
        $status = parent::_prepareShipment($orderData, $order);

        if ($status) {
            foreach ($orderData["price_data"]["shipping"] as $index => $shipping) {
                if (!empty($shipping["row_total_points"])) {
                    $orderData["xmlData"]["shipping"][$index]["OrdLn_Free1"] = $shipping["row_total_points"];
                }
            }
        }

        return $status;
    }

    protected function _prepareItems(&$orderData, $order)
    {
        $status = parent::_prepareItems($orderData, $order);

        if ($status) {
            foreach ($orderData["price_data"]["items"] as $index => $item) {
                if (!empty($item["row_total_points"])) {
                    $orderData["xmlData"]["items"][$index]["OrdLn_Free1"] = $item["row_total_points"];
                }
            }
        }

        return $status;
    }

    // Expand (some) products with SKU ending in "-strukt".
    public function expandStruct(&$orderData, $order)
    {
        foreach (array_keys($orderData["xmlData"]["items"]) as $rowIndex) {
            if ($this->isStructToExpand($orderData["xmlData"]["items"][$rowIndex]["OrdLn_ProdNo"])) {
                $structChildren = $this->getStructChildren(
                    $orderData["xmlData"]["items"][$rowIndex]["OrdLn_ProdNo"],
                    $orderData["xmlData"]["order_head"]["Ord_R1"],
                    $orderData["xmlData"]["order_head"]["Ord_Cur"]
                );
                if (!empty($structChildren)) {
                    $orderData["struct_items"][$rowIndex] = [
                        "visma_children" => $structChildren,
                        "order_children" => [],
                        "sum_money" => 0,
                        "sum_points" => 0,
                        "children_qty" => 0,
                    ];
                }
            }
        }

        if (empty($orderData["struct_items"])) {
            return true;
        }

        $orderCustomerGroupId = $order->getCustomerGroupId();
        $vismaCustomerGroupId = null;
        if (!empty($orderCustomerGroupId) && $orderCustomerGroupId > 1) {
            $vismaCustomerGroupId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne("SELECT visma_group_id FROM customer_group WHERE customer_group_id = ?", $orderCustomerGroupId);
        }

        foreach (array_keys($orderData["struct_items"]) as $rowIndex) {

            // Prepare for calculation of ratio for price

            // Loop children to calculate prices.
            foreach (array_keys($orderData["struct_items"][$rowIndex]["visma_children"]) as $sku) {
                // Prices from table is excluding VAT.
                // Prices for calculations needs to be including VAT

                // Get VAT percentage for child.
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"] = Mage::helper($this->_defaultHelper)->getTaxPercentFromVismaTaxId($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_id"], $orderData["order_currency_id"], $orderData["order_country_id"]);
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"] = (100 + $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"]) / 100;

                // Get price/points for 1 unit of this child
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit_money"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"] * ((!empty($vismaCustomerGroupId) && array_key_exists($vismaCustomerGroupId, $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"])) ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"][$vismaCustomerGroupId] : $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"][0]);
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit_points"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"] * ((!empty($vismaCustomerGroupId) && array_key_exists($vismaCustomerGroupId, $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points"])) ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points"][$vismaCustomerGroupId] : $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points"][0]);

                // Get total price/points for all units of this child
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_money"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit_money"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_points"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit_points"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];

                // Add to total price/points for all units of all children
                $orderData["struct_items"][$rowIndex]["sum_money"] += $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_money"];
                $orderData["struct_items"][$rowIndex]["sum_points"] += $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_points"];

                // Count all children
                $orderData["struct_items"][$rowIndex]["children_qty"] += $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];
            }

            // Loop children again to create new item(s) for order
            foreach (array_keys($orderData["struct_items"][$rowIndex]["visma_children"]) as $sku) {

                // NoInvoAb is total units therefore we must multiply units of parent with units of child.
                $totalQty = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"] * $orderData["xmlData"]["items"][$rowIndex]["OrdLn_NoInvoAb"];

                // Calculate ratio by dividing (total child price) by (total price of all children).
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"] = $orderData["struct_items"][$rowIndex]["sum_money"] > 0 ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_money"] / $orderData["struct_items"][$rowIndex]["sum_money"] : null;
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"] = $orderData["struct_items"][$rowIndex]["sum_points"] > 0 ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["row_total_points"] / $orderData["struct_items"][$rowIndex]["sum_points"] : null;

                // Try to figure out rato of (price and points) on each child in relation to parent.
                if (empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"]) && empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"])) {
                    // Use qty of cildren as fallback for ratio calculation.
                    $qtyRatio = ($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"] ?: 1) / ($orderData["struct_items"][$rowIndex]["children_qty"] ?: count($orderData["struct_items"][$rowIndex]["visma_children"]) ?: 1);
                    $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"] = $qtyRatio;
                    $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"] = $qtyRatio;
                } elseif (empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"]) && !empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"])) {
                    // Use points ratio as fallback if no money ratio was found
                    $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"];
                } elseif (!empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"]) && empty($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"])) {
                    // Use money ratio as fallback if no points ratio was found
                    $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"];
                }

                // Calculation of price must be done including VAT, to get price correct if child have different VAT than parent.
                $totalPriceInc = $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Price"] * ((100 + $orderData["xmlData"]["items"][$rowIndex]["OrdLn_VatRt"]) / 100);

                // Multiply by ratio to get correct price for this child
                $priceInc = $totalPriceInc * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"];

                // Calculate price excluding VAT using factor from child
                $priceEx = $priceInc / $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"];

                // Price from parent is per unit of parent. We need to divide by number of units per child.
                $price = $priceEx / $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];

                $newItem = [
                    "OrdLn_ProdNo" => $sku,
                    "OrdLn_NoInvoAb" => $totalQty,
                    "OrdLn_Price" => $price, // Price per unit !
                    "OrdLn_VatRt" => $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"],
                    "OrdLn_VatNo" => $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_id"],
                    "OrdLn_ExRt" => $orderData["xmlData"]["order_head"]["Ord_ExRt"],
                ];

                // If needed, split cash paid from parent to child
                if (array_key_exists("OrdLn_Free3", $orderData["xmlData"]["items"][$rowIndex]) && $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"] > 0) {
                    $newItem["OrdLn_Free3"] = $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["money_ratio"];
                }

                // If needed, split points paid from parent to child
                if (array_key_exists("OrdLn_Free1", $orderData["xmlData"]["items"][$rowIndex]) && $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free1"] > 0) {
                    $newItem["OrdLn_Free1"] = $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free1"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["points_ratio"];
                }

                if ($orderData["visma_order_db"] != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {
                     // Vissa produkter skall INTE undantas lagerhantering i andra bolag än Sverige
                    if (!Mage::helper($this->_defaultHelper)->isExceptionToException($newItem["OrdLn_ProdNo"])) {
                        $newItem["OrdLn_ExcPrint"] = 16384; // Undantas lagerhantering i andra bolag än Sverige
                    }
                }

                // Add child to order items
                $orderData["struct_items"][$rowIndex]["order_children"][$sku] = $newItem;
            }

            // Reset values in parent
            $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Price"] = 0;
            if (array_key_exists("OrdLn_Free1", $orderData["xmlData"]["items"][$rowIndex])) {
                unset($orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free1"]);
            }
            if (array_key_exists("OrdLn_Free3", $orderData["xmlData"]["items"][$rowIndex])) {
                unset($orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"]);
            }

        }

        return true;
    }

    public function getStructChildren($prodNo, $shopId, $currency)
    {
        $childProducts = [];
        $sqlQuery = "
            SELECT
                Struct.ProdNo,
                Struct.SubProd,
                Struct.NoPerStr AS qty,
                Prod.ProdTp AS product_type,
                PrDcMat.VatNo AS tax_id,
                PrDcMat.SugPr AS suggested_price,
                PrDcMat.Cur AS currency_code,
                PrDcMat.R1 AS shop_id,
                PrDcMat.SalePr AS price,
                PrDcMat.SalePr2 AS points,
                PrDcMat.CustPrG2 AS customer_group_id,
                PrDcMat.FrDt AS from_date,
                PrDcMat.ToDt AS to_date
            FROM Struct
            JOIN Prod ON Prod.ProdNo = Struct.SubProd
            LEFT JOIN PrDcMat ON PrDcMat.ProdNo = Struct.SubProd AND PrDcMat.Cur = :currency AND (PrDcMat.R1 = :shopId OR (PrDcMat.R1 = 0 AND PrDcMat.VatNo > 0))
            WHERE Struct.ProdNo = :prodNo
            ORDER BY Struct.ProdNo, Struct.SubProd, PrDcMat.R1, PrDcMat.CustPrG2
        ";
        $params = [
            "prodNo" => $prodNo,
            "shopId" => $shopId,
            "currency" => $currency
        ];
        $result = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery, $params);
        $containsDatePrices = false;
        foreach ($result as $priceRow) {
            if ($priceRow["tax_id"] > 0) {
                $childProducts[ $priceRow["SubProd"] ]["tax_id"] = $priceRow["tax_id"];
                $childProducts[ $priceRow["SubProd"] ]["product_type"] = $priceRow["product_type"];
                $childProducts[ $priceRow["SubProd"] ]["qty"] = $priceRow["qty"];
                continue; // row with tax_id does not have any prices
            }

            // Extract points
            $childProducts[ $priceRow["SubProd"] ]["points"][ $priceRow["customer_group_id"] ] = $priceRow["points"];

            // Extract price
            if (empty($priceRow["from_date"]) && empty($priceRow["to_date"])) {
                $childProducts[ $priceRow["SubProd"] ]["prices"][ $priceRow["customer_group_id"] ] = $priceRow["price"];
            } else {
                $containsDatePrices = true;
            }
        }

        if ($containsDatePrices) {
            foreach ($result as $priceRow) {
                if (!empty($priceRow["from_date"]) || !empty($priceRow["to_date"])) {
                    if ($this->getTimedPricesModel()->checkTimedDates($priceRow["from_date"], $priceRow["to_date"]) === true) {
                        $childProducts[ $priceRow["SubProd"] ]["prices"][ $priceRow["customer_group_id"] ] = $priceRow["price"];
                    }
                }
            }
        }

        return $childProducts;
    }

}
