<?php

class Awardit_Integration_Model_Cli_Accrual extends Awardit_Integration_Model_Cli {

    protected $_defaultHelper = "integration/PointsCore";

    const ACCRUAL_ORDER_PAYOUT_STATUS_WAITING = 0;
    const ACCRUAL_ORDER_PAYOUT_STATUS_DONE = 1;
    const ACCRUAL_ORDER_PAYOUT_STATUS_CANCELED = 8;
    const ACCRUAL_ORDER_PAYOUT_STATUS_ERROR = 9;

    const ACCRUAL_ORDER_PAYOUT_TRIES_MAX = 3;
    const ACCRUAL_ORDER_ID_PREFIX = "AC_";
    const ACCRUAL_ACL_ORDER_INFO = "admin/sales/accrual_info";

    const SAS_API_POINTS_TYPE_EXTRA = "EXTRA";
    const SAS_API_POINTS_TYPE_BASIC = "BASIC";
    const MAGENTO_ATTRIBUTE_CODE_SAS_CUSTOMER_EB = "sas_eb_no";

    protected $_sasApiModel = null;
    protected $_sasApiToken = null;
    protected $_maxTries = null;

    // -m=Accrual -f=ProcessSingleOrder -p="orderId:<order_id>|incrementId:<increment_id>"
    public function CLI_ProcessSingleOrder($param)
    {
        $wantedParams = [
            "orderId" => "int",
            "incrementId" => "string"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $accrualOrders = [];
        if (!empty($extractedParams["orderId"])) {
            $accrualOrders[0] = $this->getAccrualOrdersByAttribute("order_id", $extractedParams["orderId"]);
            if (empty($accrualOrders[0])) {
                Mage::helper($this->_defaultHelper)->log("Unable to load accrual order #{$extractedParams["orderId"]}!", LOG_ERR);
                return;
            }
        } elseif (!empty($extractedParams["incrementId"])) {
            $accrualOrders[0] = $this->getAccrualOrdersByAttribute("increment_id", $extractedParams["incrementId"]);
            if (empty($accrualOrders[0])) {
                Mage::helper($this->_defaultHelper)->log("Unable to load accrual order {$extractedParams["incrementId"]}!", LOG_ERR);
                return;
            }
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
            return;
        }

        if ($accrualOrders[0]["payout_status"] == self::ACCRUAL_ORDER_PAYOUT_STATUS_DONE) {
            Mage::helper($this->_defaultHelper)->log("Accrual already processed for order {$extractedParams["incrementId"]}");
            return;
        }

        $this->processAccrualOrders($accrualOrders);
        $this->updateAccrualOrders($accrualOrders);

    }

    // -m=Accrual -f=ProcessSingleOrder -p="dateOverride:<date>"
    public function CLI_ProcessAllOrders($param)
    {
        $wantedParams = [
            "dateOverride" => "string"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $date = date("Y-m-d", empty($extractedParams["dateOverride"]) ? time() : strtotime($extractedParams["dateOverride"]));

        $accrualOrders = $this->getAccrualOrdersByAttribute("to_process", $date);
        $this->processAccrualOrders($accrualOrders);
        $this->updateAccrualOrders($accrualOrders);
    }

    // -m=Accrual -f=ListOrdersWithAttribute -p="attributeCode:<attribute_code>,attributeValue:<value>"
    public function CLI_ListOrdersWithAttribute($param)
    {
        $wantedParams = [
            "attributeCode" => "string",
            "attributeValue" => "string"
        ];

        $extractedParams = $this->extractParams($param, $wantedParams);
        if (!empty($extractedParams["attributeCode"]) || !empty($extractedParams["attributeValue"])) {
            echo "Missing parameter!\n";
            return;
        }

        $accrualOrders = $this->getAccrualOrdersByAttribute($extractedParams["attributeCode"], $extractedParams["attributeValue"]);
        echo print_r($accrualOrders, true);
    }

    public function getMaxTries()
    {
        if ($this->_maxTries === null) {
            $this->_maxTries = self::ACCRUAL_ORDER_PAYOUT_TRIES_MAX;
        }
        return $this->_maxTries;
    }

    public function setMaxTries($maxTries)
    {
        if ($maxTries === null) {
            $maxTries = self::ACCRUAL_ORDER_PAYOUT_TRIES_MAX;
        }
        $this->_maxTries = $maxTries;
    }

    public function getAccrualOrdersByAttribute($attributeName, $attributeValue)
    {
        $params = [];

        switch ($attributeName) {
            case "order_id":
                $params["orderId"] = $attributeValue;
                $where = "ca.order_id = :orderId";
                $fetchRow = true;
                break;

            case "increment_id":
                $params["incrementId"] = $attributeValue;
                $where = "ca.increment_id = :incrementId";
                $fetchRow = true;
                break;

            case "payout_status":
                $params["payoutStatus"] = $attributeValue;
                $where = "ca.payout_status = :payoutStatus";
                $fetchRow = false;
                break;

            case "to_process":
                $params["payoutDate"] = date("Y-m-d", strtotime($attributeValue));
                $params["statusWaiting"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_WAITING;
                $params["statusError"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
                $params["maxTries"] = $this->getMaxTries();
                $where = "(ca.payout_status = :statusWaiting AND ca.payout_date <= :payoutDate) OR (ca.payout_status = :statusError AND ca.payout_tries < :maxTries)";
                $fetchRow = false;
                break;

            default:
                return [];
        }

        if ($this->getLimit() > 0) {
            $limit = " LIMIT " . intval($this->getLimit());
        } else {
            $limit = "";
        }

        $sqlQuery = "
            SELECT
                ca.*,
                sfo.created_at,
                sfo.store_id,
                cio.visma_db
            FROM " . Awardit_Integration_Helper_PointsCore::INTEGRATION_TABLE_ACCRUAL . " ca
            JOIN " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " cio ON cio.magento_order_id = ca.order_id
            JOIN sales_flat_order sfo ON sfo.entity_id = ca.order_id
            WHERE {$where}
            ORDER BY ca.payout_date ASC, ca.order_id ASC
            {$limit}
        ";

        if ($fetchRow) {
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, $params);
        } else {
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);
        }
    }

    public function getVismaOrderStatus($incrementId, $vismaDB)
    {
        $sqlQuery = "SELECT Ord.Gr FROM {$vismaDB}.dbo.Ord WHERE Ord.Inf = :incrementId";
        $params = [ "incrementId" => $incrementId ];
        return Mage::helper($this->_defaultHelper)->getVismaDB()->fetchOne($sqlQuery, $params);

    }

    public function processAccrualOrders(&$accrualOrders)
    {
        $iterations = 0;

        foreach (array_keys($accrualOrders) as $index) {

            if ($this->getLimit() > 0) {
                if ($iterations++ >= $this->getLimit()) {
                    break;
                }
            }

            // Load Magento order
            $magentoOrder = Mage::getModel("sales/order")->load($accrualOrders[$index]["order_id"]);
            if ($magentoOrder->getId() != $accrualOrders[$index]["order_id"]) {
                $accrualOrders[$index]["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
                Mage::helper($this->_defaultHelper)->log("Unable to load accrual order {$accrualOrders[$index]["increment_id"]}!", LOG_ERR);
                continue;
            }

            // Try to extract sas_eb_no from extra_data
            if (empty($accrualOrders[$index]["extra_data"])) {
                $extraData = @json_decode($accrualOrders[$index]["extra_data"], true);
                if (!empty($extraData["sas_eb_no"])) {
                    $accrualOrders[$index]["customer_eb_no"] = $extraData["sas_eb_no"];
                }
            }

            // If customer_eb_no is empty, try to get it using Magento customer
            if (empty($accrualOrders[$index]["customer_eb_no"])) {
                $customerId = $magentoOrder->getCustomerId();
                if (!empty($customerId)) {
                    $customer = Mage::getModel("customer/customer")->load($customerId);
                    $accrualOrders[$index]["customer_eb_no"] = $customer->getData(self::MAGENTO_ATTRIBUTE_CODE_SAS_CUSTOMER_EB);
                }
            }

            // If customer_eb_no still is empty, log it and continue with next
            if (empty($accrualOrders[$index]["customer_eb_no"])) {
                $accrualOrders[$index]["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
                $msg = "Unable to get customer EB number from order {$accrualOrders[$index]["increment_id"]}!";
                Mage::helper($this->_defaultHelper)->log($msg, LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Error in accrual for {$accrualOrders[$index]["increment_id"]}",
                    $msg,
                    ["it"]
                );
                continue;
            }

            switch ($magentoOrder->getStatus()) {
                case "processing":
                case "complete":
                    // Ok to do accrual payout
                    break;

                case "canceled":
                case "closed":
                    // NOT ok to do accrual payout
                    $accrualOrders[$index]["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_CANCELED;
                    Mage::helper($this->_defaultHelper)->log("Accrual order {$accrualOrders[$index]["increment_id"]} is canceled in Visma");
                    continue 2;

                default:
                    // NOT ok to do accrual payout
                    $accrualOrders[$index]["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
                    Mage::helper($this->_defaultHelper)->log("Accrual order {$accrualOrders[$index]["increment_id"]} have unprocessable status '{$magentoOrder->getStatus()}' in Magento!", LOG_ERR);
                    continue 2;
            }

            // Check order in Visma
            $vismaDB = $accrualOrders[$index]["visma_db"] ?: Mage::helper($this->_defaultHelper)->getVismaOrderDB($magentoOrder->getStoreId());
            $vismaStatus = $this->getVismaOrderStatus($accrualOrders[$index]["increment_id"], $vismaDB);

            // If order was placed in other company than Sweden, try Sweden too
            if ($vismaStatus != Awardit_Integration_Model_Cli_OrderExport::VISMA_ORDER_STATUS_DELIVERED && $vismaDB != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {
                $vismaDB = Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB;
                $vismaStatus = $this->getVismaOrderStatus($accrualOrders[$index]["increment_id"], $vismaDB);
            }

            if ($vismaStatus == Awardit_Integration_Model_Cli_OrderExport::VISMA_ORDER_STATUS_CANCELED) {
                $accrualOrders[$index]["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_CANCELED;
                Mage::helper($this->_defaultHelper)->log("Accrual order {$accrualOrders[$index]["increment_id"]} is canceled in Visma");
                continue;
            }

            if ($vismaStatus != Awardit_Integration_Model_Cli_OrderExport::VISMA_ORDER_STATUS_DELIVERED) {
                Mage::helper($this->_defaultHelper)->log("Accrual order {$accrualOrders[$index]["increment_id"]} is not delivered yet");
                continue;
            }

            $this->doAccrualPayout($accrualOrders[$index]);
        }
    }

    public function doAccrualPayout(&$accrualOrder)
    {
        try {
            $accrualOrder["store"] = Mage::getModel("core/store")->load($accrualOrder["store_id"]);
            $token = $this->getSasApiToken($accrualOrder);
            if (empty($token)) {
                Mage::helper($this->_defaultHelper)->log("Unable to get SAS API token!", LOG_ERR);
                $accrualOrder["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
                return;
            }
        } catch (Exception $exception) {
            $msg = "Exception while getting SAS API token for order {$accrualOrder["increment_id"]}!";
            Mage::helper($this->_defaultHelper)->log($msg, LOG_ERR);
            Mage::helper($this->_defaultHelper)->logException($exception, $msg);
            $accrualOrder["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
            return;
        }

        try {
            $transData = $this->getSasApiModel()->registerSale(
                $accrualOrder["store"],
                $token,
                $accrualOrder["customer_eb_no"],
                $accrualOrder["payout_points"],
                self::ACCRUAL_ORDER_ID_PREFIX . $accrualOrder["increment_id"],
                strtotime($accrualOrder["created_at"]),
                self::SAS_API_POINTS_TYPE_EXTRA
            );
        } catch (Exception $exception) {
            $msg = "Exception while registering accrual for order {$accrualOrder["increment_id"]}!";
            Mage::helper($this->_defaultHelper)->log($msg, LOG_ERR);
            Mage::helper($this->_defaultHelper)->logException($exception, $msg);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Error in accrual for {$accrualOrder["increment_id"]}",
                $msg . "\nException: {$exception->getMessage()}\nTrace:\n{$exception->getTraceAsString()}",
                ["it"]
            );
            $accrualOrder["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
            return;
        }

        if (empty($transData["transactionId"])) {
            $msg = "Got no transaction id when registering accrual for order {$accrualOrder["increment_id"]}!";
            Mage::helper($this->_defaultHelper)->log($msg, LOG_ERR);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Error in accrual for {$accrualOrder["increment_id"]}",
                $msg . "\ntransData: " . print_r($transData, true),
                ["it"]
            );
            $accrualOrder["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_ERROR;
            return;
        }

        $accrualOrder["transaction_id"] = $transData["transactionId"];
        $accrualOrder["payout_status"] = self::ACCRUAL_ORDER_PAYOUT_STATUS_DONE;

    }

    public function getSasApiModel()
    {
        if (empty($this->_sasApiModel)) {
            $this->_sasApiModel = Mage::getModel("sas/Api");
        }
        return $this->_sasApiModel;
    }

    public function getSasApiToken($accrualOrder)
    {
        if (empty($this->_sasApiToken)) {
            $this->_sasApiToken = $this->getSasApiModel()->getToken($accrualOrder["store"]);
        }

        // TODO: check if token is expired, if so, get new one

        if ($this->getDebugMode()) {
            Mage::log(print_r($this->_sasApiToken, true), LOG_DEBUG, "api-token-debug.log", true);
        }

        return $this->_sasApiToken;
    }

    public function updateAccrualOrders($accrualOrders)
    {
        try {
            $sqlQuery = "UPDATE " . Awardit_Integration_Helper_PointsCore::INTEGRATION_TABLE_ACCRUAL . " SET payout_tries = :payouTries, payout_status = :payoutStatus, transaction_id = :transId WHERE order_id = :orderId";
            $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);

            foreach (array_keys($accrualOrders) as $index) {

                // Only update data if payout status is something else than WAITING.
                if ($accrualOrders[$index]["payout_status"] != self::ACCRUAL_ORDER_PAYOUT_STATUS_WAITING) {
                    $stmt->bindValue("payouTries", intval($accrualOrders[$index]["payout_tries"]) + 1);
                    $stmt->bindValue("payoutStatus", $accrualOrders[$index]["payout_status"]);
                    $stmt->bindValue("transId", $accrualOrders[$index]["transaction_id"]);
                    $stmt->bindValue("orderId", intval($accrualOrders[$index]["order_id"]));
                    $stmt->execute();

                    if ($accrualOrders[$index]["payout_status"] == self::ACCRUAL_ORDER_PAYOUT_STATUS_DONE) {
                        Mage::helper($this->_defaultHelper)->log("Accrual successfully registered for order {$accrualOrders[$index]["increment_id"]}");
                    }
                }

                // If update was successfull or not needed, remove it to keep count if we later get an exception
                unset($accrualOrders[$index]);

            }
        } catch (Exception $exception) {
            $msg = "Exception while updating accrual!";
            Mage::helper($this->_defaultHelper)->log($msg, LOG_ERR);
            if (!empty($accrualOrders)) {
                $body = "Have the following data left to update:\n";
                $body .= print_r($accrualOrders, true);
                Mage::helper($this->_defaultHelper)->log($body, LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail($msg, $body, [ "it" ]);
            }
            Mage::helper($this->_defaultHelper)->logException($exception, $msg);
        }
    }

}
