<?php

class Awardit_Integration_Model_Cli_ProductImport extends Awardit_Integration_Model_Cli {

    const IMPORT_STATUS_NEW      = 0; // New
    const IMPORT_STATUS_FETCHED  = 1; // Data for import fetched
    const IMPORT_STATUS_COMPLETE = 2; // Product successfully imported
    const IMPORT_STATUS_SKIP     = 8; // Product have status indicating it is no longer active
    const IMPORT_STATUS_ERROR    = 9; // Product import generated error

    /*
      beskriving av "status"
      1 = Släcks ej. Magento skall ej sköta lagerstatus
      2 = Släcks. Magento sköter lagerstatus
      3 = Utgående produkt. Samma som 2.
      4 = Ej i lager. Utgången, visas ej.
      5 = ej i stock, men produkten skall visas
      6 = ej i stock. visas ej.
      7 = ej i stock. visas ej.
     */
    const VISMA_PRODUCT_STATUS_UNKNOWN  = 0; // Visma product status unknown
    const VISMA_PRODUCT_STATUS_ACTIVE   = 1; // Always active
    const VISMA_PRODUCT_STATUS_TURN_OFF = 2; // Deactivate if out-of-stock, reactivate when in-stock
    const VISMA_PRODUCT_STATUS_LAST     = 3; // Product is still in-stock but about to expire
    const VISMA_PRODUCT_STATUS_EXPIRED  = 4; // Product has expired
    const VISMA_PRODUCT_STATUS_SOLD_OUT = 5; // Product is still active, but sold out
    const VISMA_PRODUCT_STATUS_ARCHIVED = 6; // Product is no longer available
    const VISMA_PRODUCT_STATUS_SEASON   = 7; // Product is seasonal

    const VISMA_PRODUCT_TYPE_SIMPLE = 1;
    const VISMA_PRODUCT_TYPE_VIRTUAL = 2;

    const ESTIMATED_DELIVERY_DATE_OFFSET = "+4 days";

    protected $_updateStock = true;
    protected $_updatePrice = true;

    private $extraVismaProductDataFields = [];
    private $extraVismaProductPriceFields = [];

    // -m=ProductImport -f=tagAllProducts
    public function CLI_tagAllProducts($param)
    {
        $allProducts = $this->getAllProducts();
        $productsToTagQty = count($allProducts);
        Mage::helper($this->_defaultHelper)->log("Found {$productsToTagQty} products to tag for import.");
        $this->tagProductsForImport($allProducts);
        Mage::helper($this->_defaultHelper)->log("Done!");
    }

    // -m=ProductImport -f=tagUpdatedProducts
    public function CLI_tagUpdatedProducts($param)
    {
        $updatedProducts = $this->getAllUpdatedProducts();
        $productsToTagQty = count($updatedProducts);
        Mage::helper($this->_defaultHelper)->log("Found {$productsToTagQty} products to tag for import.");
        $this->tagProductsForImport($updatedProducts);
        Mage::helper($this->_defaultHelper)->log("Done!");

    }

    // -m=ProductImport -f=tagSingleProduct -p="sku:<sku>"
    public function CLI_tagSingleProduct($param)
    {
        $wantedParams = [ "!sku" => "string" ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        // Create filter to only include products with prices in this Magento instance
        $vismaShopIds = Mage::helper($this->_defaultHelper)->getInstanceUniqueVismaShopIds();
        if (empty($vismaShopIds) || !is_array($vismaShopIds)) {
            return;
        }
        $vismaShopIdString = implode(",", $vismaShopIds);

        // Make sure sku exists in this Magento instance
        $sqlQuery = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND PrDcMat.R1 IN ({$vismaShopIdString}) AND Prod.ProdNo = :prodNo";
        $params = [ "prodNo" => $extractedParams["sku"] ];
        $prodNo = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchOne($sqlQuery, $params);
        if ($prodNo != $extractedParams["sku"]) {
            Mage::helper($this->_defaultHelper)->log("Product [{$extractedParams["sku"]}] does not belong in this Magento instance!", LOG_ERR);
            return;
        }

        Mage::helper($this->_defaultHelper)->log("Tag single product for import: {$prodNo}");
        $this->tagProductsForImport([$prodNo]);
        Mage::helper($this->_defaultHelper)->log("Done!");

    }

    // -m=ProductImport -f=importAllProducts -p="updateStock:<0|1>"
    public function CLI_importAllProducts($param)
    {
        $wantedParams = [ "updateStock" => "int" ];
        $extractedParams = $this->extractParams($param, $wantedParams);
        if (array_key_exists("updateStock", $extractedParams)) {
            $this->setUpdateStock(empty($extractedParams["updateStock"]) ? false : true);
        }
        $importedProducts = 0;

        $maxQty = $this->getMaxQtyToImport();
        Mage::helper($this->_defaultHelper)->log("Starting to import products (max {$this->getLimit()}). Found {$maxQty} in queue.");
        $integrationProduct = $this->getProductToImport();
        while (!empty($integrationProduct)) {

            $this->mergeDataToProduct($integrationProduct);
            $this->updateIntegrationData($integrationProduct);
            if ($integrationProduct["import_status"] == Awardit_Integration_Model_Cli_ProductImport::IMPORT_STATUS_FETCHED) {
                $this->importSingleProduct($integrationProduct, $importedProducts+1, $maxQty);
                $this->updateIntegrationData($integrationProduct);
            }

            $importedProducts++;
            if ($this->getLimit() > 0) {
                if ($importedProducts >= $this->getLimit()) {
                    break;
                }
            }
            $integrationProduct = $this->getProductToImport();
        }

        Mage::helper($this->_defaultHelper)->log("Imported {$importedProducts} products.");

    }

    // -m=ProductImport -f=importSingleProduct -p="sku:<sku>[,updateStock:<0|1>]"
    public function CLI_importSingleProduct($param)
    {
        Mage::helper($this->_defaultHelper)->log("Starting to import single product.");

        $wantedParams = [ "!sku" => "string", "updateStock" => "int" ];
        $extractedParams = $this->extractParams($param, $wantedParams);
        if (array_key_exists("updateStock", $extractedParams)) {
            $this->setUpdateStock(empty($extractedParams["updateStock"]) ? false : true);
        }

        if (array_key_exists("updateStock", $extractedParams)) {
            $this->setUpdateStock(empty($extractedParams["updateStock"]) ? false : true);
        }
        $integrationProduct = $this->getProductToImport($extractedParams["sku"]);
        if (empty($integrationProduct)) {
            echo "Found no product to import with sku = {$extractedParams["sku"]}";
            return;
        }

        $this->mergeDataToProduct($integrationProduct);
        $this->updateIntegrationData($integrationProduct);
        if ($integrationProduct["import_status"] == Awardit_Integration_Model_Cli_ProductImport::IMPORT_STATUS_FETCHED) {
            $this->importSingleProduct($integrationProduct);
            $this->updateIntegrationData($integrationProduct);
        }

    }

    // -m=ProductImport -f=purgeProducts
    public function CLI_purgeProducts($param)
    {
        // ($doIndex = false, $doEcho = false, $doLog = false)
        $data = Mage::getModel("integration/Producthandler")->purgeProducts(true, $this->getDebugMode(), true);
        if ($this->getDebugMode()) {
            echo $data["message"] . "\n";
        }
    }

    // -m=ProductImport -f=listDisabledProducts
    public function CLI_listDisabledProducts($param)
    {
        // ($doIndex = false, $doEcho = false, $doLog = false)
        $data = Mage::getModel("integration/Producthandler")->listDisabledProducts(true, $this->getDebugMode(), true);
        if ($this->getDebugMode()) {
            echo $data["message"] . "\n";
        }
    }

    public function getExtraVismaProductDataFields()
    {
        return $this->extraVismaProductDataFields;
    }

    public function getExtraVismaProductDataFieldsAsString()
    {
        $extraFields = [];

        foreach ($this->getExtraVismaProductDataFields() as $field => $name) {
            $extraFields[] = "{$field} AS {$name}";
        }

        return implode(",", $extraFields) . (!empty($extraFields) ? "," : "");

    }

    public function getExtraVismaProductPriceFields()
    {
        return $this->extraVismaProductPriceFields;
    }

    public function getExtraVismaProductPriceFieldsAsString()
    {
        $extraFields = [];

        foreach ($this->getExtraVismaProductPriceFields() as $field => $name) {
            $extraFields[] = "{$field} AS {$name}";
        }

        return implode(",", $extraFields) . (!empty($extraFields) ? "," : "");

    }

    public function setUpdateStock($mode)
    {
        $this->_updateStock = $mode ? true : false;
    }

    public function getUpdateStock()
    {
        return $this->_updateStock;
    }

    public function setUpdatePrice($mode)
    {
        $this->_updatePrice = $mode ? true : false;
    }

    public function getUpdatePrice($product, $storeId, $priceData)
    {
        return $this->_updatePrice;
    }

    public function getAllProducts()
    {
        // Create filter to only include products with prices in this Magento instance
        $vismaShopIds = Mage::helper($this->_defaultHelper)->getInstanceUniqueVismaShopIds();
        $skus = [];
        if (!empty($vismaShopIds)) {
            $vismaShopIdString = implode(",", $vismaShopIds);
            if (!empty($vismaShopIdString)) {
                Mage::helper($this->_defaultHelper)->log("Scanning for all products in this instance");

                $sqlQuery = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND PrDcMat.R1 IN ({$vismaShopIdString})";
                $products = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery);

                foreach ($products as $prodRow) {
                    $skus[] = $prodRow["ProdNo"];
                }
            }
        }
        return $skus;
    }

    public function getAllUpdatedProducts()
    {
        // Default time if we don't get one from the database
        $lastUpdateTimestamp = strtotime("-1 hour");
        $updatedProducts = [];

        $sqlQuery = "SELECT MAX(`created_at`) as last_update FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS;
        $timestamp = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery);
        if (!empty($timestamp)) {
            $lastUpdateTimestamp = strtotime($timestamp);
        }
        $lastUpdateDate = date("Ymd", $lastUpdateTimestamp);
        $lastUpdateTime = date("Hi", $lastUpdateTimestamp) - 0;

        // Create filter to only include products with prices in this Magento instance
        $vismaShopIds = Mage::helper($this->_defaultHelper)->getInstanceUniqueVismaShopIds();
        if (empty($vismaShopIds)) {
            Mage::helper($this->_defaultHelper)->log("Found no stores with Visma shop id!", LOG_ERR);
            return $updatedProducts;
        }
        $vismaShopIdString = implode(",", $vismaShopIds);
        if (empty($vismaShopIdString)) {
            Mage::helper($this->_defaultHelper)->log("Found no stores with Visma shop id!", LOG_ERR);
            return $updatedProducts;
        }

        $logDate = date("Y-m-d H:i:s", $lastUpdateTimestamp);
        Mage::helper($this->_defaultHelper)->log("Scanning for new and/or updated products since {$logDate}");

        // get latest products based on product-table
        $sqlQuery1 = "SELECT DISTINCT(Prod.ProdNo) AS ProdNo FROM Prod JOIN PrDcMat ON PrDcMat.ProdNo = Prod.ProdNo WHERE Prod.Gr7 = 1 AND (Prod.ChDt > {$lastUpdateDate} OR (Prod.ChDt = {$lastUpdateDate} AND Prod.ChTm > {$lastUpdateTime})) AND PrDcMat.R1 IN ({$vismaShopIdString})";
        $productData = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery1);

        // get latest products based on price-table
        $sqlQuery2 = "SELECT DISTINCT(p1.ProdNo) AS ProdNo FROM PrDcMat p1 JOIN Prod p2 ON p2.ProdNo = p1.ProdNo JOIN PrDcMat p3 ON p3.ProdNo = p2.ProdNo WHERE p2.Gr7 = 1 AND (p1.ChDt > {$lastUpdateDate} OR (p1.ChDt = {$lastUpdateDate} AND p1.ChTm > {$lastUpdateTime})) AND p3.R1 IN ({$vismaShopIdString})";
        $productPrice = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery2);

        // get latest products based on stock update
        $sqlQuery3 = "SELECT DISTINCT(p1.ProdNo) AS ProdNo FROM StcBal p1 JOIN Prod p2 ON p2.ProdNo = p1.ProdNo JOIN PrDcMat p3 ON p3.ProdNo = p2.ProdNo WHERE p2.Gr7 = 1 AND p2.Gr2 != 4 AND (p1.ChDt > {$lastUpdateDate} OR (p1.ChDt = {$lastUpdateDate} AND p1.ChTm > {$lastUpdateTime})) AND p3.R1 IN ({$vismaShopIdString})";
        $productStock = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery3);

        // join all above products
        foreach ($productData as $row) {
            $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
        }

        foreach ($productPrice as $row) {
            if (empty($updatedProducts[$row["ProdNo"]])) {
                $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
            }
        }

        foreach ($productStock as $row) {
            if (empty($updatedProducts[$row["ProdNo"]])) {
                $updatedProducts[$row["ProdNo"]] = $row["ProdNo"];
            }
        }

        return $updatedProducts;
    }

    public function tagProductsForImport($skus)
    {
        if (empty($skus) || !is_array($skus)) {
            return;
        }

        $skuStr = implode("'),('", $skus);
        $defaultVismaStatus = self::VISMA_PRODUCT_STATUS_UNKNOWN;
        $defaultImportStatus = self::IMPORT_STATUS_NEW;
        $sqlQuery = "
            INSERT INTO " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS . " (`sku`) VALUES ('{$skuStr}')
            ON DUPLICATE KEY UPDATE
                `created_at` = now(),
                `imported_at` = NULL,
                `visma_status` = {$defaultVismaStatus},
                `import_status` = {$defaultImportStatus},
                `price_data` = NULL,
                `stock_data` = NULL,
                `extra_data` = NULL
        ";
        Mage::getSingleton("core/resource")->getConnection("core_write")->exec($sqlQuery);
    }

    public function getMaxQtyToImport()
    {
        $sqlQuery = "SELECT count(id) AS max_qty FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS . " WHERE `import_status` < :importStatus";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, [ "importStatus" => self::IMPORT_STATUS_COMPLETE ]);
    }

    public function getProductToImport($sku = null)
    {
        if (empty($sku)) {
            $sqlQuery = "SELECT * FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS . " WHERE `import_status` < :importStatus ORDER BY `created_at` ASC, `id` ASC LIMIT 1";
            $integrationProduct = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, [ "importStatus" => self::IMPORT_STATUS_COMPLETE ]);
        } else {
            $sqlQuery = "SELECT * FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS . " WHERE `sku` = :sku";
            $integrationProduct = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, [ "sku" => $sku ]);
        }

        if (!empty($integrationProduct["price_data"])) {
            $integrationProduct["price_data"] = json_decode($integrationProduct["price_data"], true);
        }

        if (!empty($integrationProduct["stock_data"])) {
            $integrationProduct["stock_data"] = json_decode($integrationProduct["stock_data"], true);
        }

        if (!empty($integrationProduct["extra_data"])) {
            $integrationProduct["extra_data"] = json_decode($integrationProduct["extra_data"], true);
        }

        return $integrationProduct;

    }

    public function mergeDataToProduct(&$integrationProduct)
    {
        // Having problems with using sku as placeholder/parameter when sku is numeric.
        // ProdNo is specified as varchar and MSSQL freaks out if we ask for 12345 instead of '12345'
        $sku = $integrationProduct["sku"];

        $sqlQuery1 = "
            SELECT
                Prod.ProdNo AS sku,
                Prod.Descr AS name,
                Prod.Inf AS manufacturer_product_id,
                Prod.R3 AS manufacturer_id,
                CASE WHEN R3.Nm IS NOT NULL THEN R3.Nm ELSE '' END AS manufacturer_name,
                Prod.Gr2 AS visma_status,
                Prod.WdtU AS weight,
                Prod.ProdTp AS product_type,
                {$this->getExtraVismaProductDataFieldsAsString()}
                OrdLn.CfDelDt AS estimated_delivery_date
            FROM
                Prod
            LEFT JOIN
              R3 ON R3.RNo = Prod.R3
            LEFT JOIN
                OrdLn ON OrdLn.ProdNo = Prod.ProdNo AND OrdLn.SupNo > 0 AND OrdLn.CustNo = 0 AND OrdLn.NoInvoAb > 0
            WHERE
                Prod.ProdNo = '{$sku}'
        ";
        $integrationProduct["extra_data"] = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchRow($sqlQuery1);
        if (!empty($integrationProduct["extra_data"]["name"]) && !Mage::helper($this->_defaultHelper)->isUTF8($integrationProduct["extra_data"]["name"])) {
            $integrationProduct["extra_data"]["raw_name"] = $integrationProduct["extra_data"]["name"];
            $integrationProduct["extra_data"]["name"] = Mage::helper($this->_defaultHelper)->convertString($integrationProduct["extra_data"]["name"]);
        }
        if (!empty($integrationProduct["extra_data"]["manufacturer_name"]) && !Mage::helper($this->_defaultHelper)->isUTF8($integrationProduct["extra_data"]["manufacturer_name"])) {
            $integrationProduct["extra_data"]["raw_manufacturer_name"] = $integrationProduct["extra_data"]["manufacturer_name"];
            $integrationProduct["extra_data"]["manufacturer_name"] = Mage::helper($this->_defaultHelper)->convertString($integrationProduct["extra_data"]["manufacturer_name"]);
        }
        $integrationProduct["visma_status"] = $integrationProduct["extra_data"]["visma_status"];

        // Populate the extra_data field and clean up original data arrays
        $tmp = "{$integrationProduct["extra_data"]["estimated_delivery_date"]}";
        // Remove old value, replaced later if ok
        unset($integrationProduct["extra_data"]["estimated_delivery_date"]);
        if (!empty($tmp)) {
            if (strlen($tmp) == 8) {
                $tmpTime = new DateTime(date("Y-m-d", strtotime(self::ESTIMATED_DELIVERY_DATE_OFFSET, strtotime(substr($tmp, 0, 4) . "-" . substr($tmp, 4, 2) . "-" . substr($tmp, 6, 2)))));
                $now = new DateTime("now");
                if ($tmpTime > $now) {
                    $integrationProduct["extra_data"]["estimated_delivery_date"] = $tmpTime->format("Y-m-d");
                }
            }
        }

        // Create filter to only include products with prices in this Magento instance
        $vismaShopIds = Mage::helper($this->_defaultHelper)->getInstanceUniqueVismaShopIds();
        if (empty($vismaShopIds) || !is_array($vismaShopIds)) {
            $subject = "No Visma shop ids in this Magento instance";
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                $subject,
                sprintf("Magento instance %s have no Visma shop ids configured", Mage::helper($this->_defaultHelper)->getInstanceId()),
                [ "it" ]
            );
            throw new Exception($subject);
        }
        $vismaShopIdString = implode(",", $vismaShopIds);

        $sqlQuery2 = "
            SELECT
                PurcPr AS purchase_price,
                PurcCur AS purchase_currency_id,
                VatNo AS tax_id,
                SugPr AS suggested_price,
                Cur AS currency_id,
                R1 AS shop_id,
                SalePr AS price,
                CustPrG2 AS customer_group_id,
                FrDt AS from_date,
                ToDt AS to_date,
                ChUsr AS updated_by,
                ChDt AS updated_at_date,
                ChTm AS updated_at_time,
                {$this->getExtraVismaProductPriceFieldsAsString()}
                CASE WHEN  R1 > 0 AND Cur > 0 THEN CONCAT(CustPrG2, '-', R1, '-', Cur) ELSE NULL END AS price_index
            FROM
                PrDcMat
            WHERE
                ProdNo = '{$sku}'
                AND CustNo = 0
                AND NOT (PurcCur > 0 AND Cur > 0)
                AND R1 IN (0,{$vismaShopIdString})
            ORDER BY Cur, R1, CustPrG2, FrDt, ToDt
        ";
        $integrationProduct["price_data"] = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery2);

        // Even if we later skip stock updates, we still might need the data (for new products)
        $integrationProduct["stock_data"]["stock_balance"] = Mage::helper($this->_defaultHelper)->getVismaProductStock($sku);

        $integrationProduct["import_status"] = self::IMPORT_STATUS_FETCHED;

    }

    public function updateIntegrationData($integrationProduct)
    {
        $updatedAt = $integrationProduct["import_status"] == self::IMPORT_STATUS_COMPLETE ? "`imported_at` = now()" : "`imported_at` = null";
        $sqlQuery = "
            UPDATE " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_PRODUCTS . " SET
                `visma_status` = :vismaStatus,
                `import_status` = :importStatus,
                {$updatedAt},
                `extra_data` = :extraData,
                `price_data` = :priceData,
                `stock_data` = :stockData
            WHERE `id` = :id
        ";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->bindValue("vismaStatus", intval($integrationProduct["visma_status"]));
        $stmt->bindValue("importStatus", intval($integrationProduct["import_status"]));
        $stmt->bindValue("extraData", json_encode($integrationProduct["extra_data"]));
        $stmt->bindValue("priceData", json_encode($integrationProduct["price_data"]));
        $stmt->bindValue("stockData", json_encode($integrationProduct["stock_data"]));
        $stmt->bindValue("id", intval($integrationProduct["id"]));
        $stmt->execute();
        $affectedRows = $stmt->rowCount();
        if ($affectedRows < 1) {
            // No update was made, perhaps all supplied data had no updated values?
        }
    }

    public function importSingleProduct(&$integrationProduct, $iteration = 0, $maxQty = 0)
    {

        if (empty($integrationProduct["id"])) {
            throw new Exception("Unable to load integration product data.");
        }

        if ($integrationProduct["import_status"] != self::IMPORT_STATUS_FETCHED) {
            return true;
        }

        if ($iteration > 0 && $maxQty > 0) {
            Mage::helper($this->_defaultHelper)->log("Importing product [{$integrationProduct["sku"]}] ({$iteration}/{$maxQty})");
        } else {
            Mage::helper($this->_defaultHelper)->log("Importing product [{$integrationProduct["sku"]}]");
        }

        if (stripos($integrationProduct["extra_data"]["name"], "struktur") !== false) {
            Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] is a 'struktur', skipping it.");
            $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
            return true;
        }

        // Extract data, do modifications, etc (used for local instances and integration addons)
        $this->doLocalPreProcessingOfData($integrationProduct);

        // Check for timed prices
        $this->getTimedPricesModel()->parseTimedPrices($integrationProduct["sku"], $integrationProduct["price_data"]);

        // Find suggested prices and taxes
        $taxArray = [];
        $EURTaxId = 0;
        $msrpArray = [];
        foreach ($integrationProduct["price_data"] as $priceData) {
            $currencyId = intval($priceData["currency_id"]);
            if (!empty($priceData["tax_id"])) {
                $taxArray[$currencyId] = Mage::helper($this->_defaultHelper)->getMagentoTaxIdFromVismaTaxId($priceData["tax_id"], $currencyId);
                if ($currencyId == 99) {
                    $EURTaxId = $priceData["tax_id"];
                }
            }
            if (!empty($currencyId) && $priceData["suggested_price"] > 0) {
                $msrpArray[$currencyId] = $priceData["suggested_price"] - 0;
            }
        }

        // Find default suggested price and tax
        $suggestedPrice = 0;
        $defaultTaxId = null;
        if (array_key_exists(Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID, $msrpArray)) {
            $suggestedPrice = $msrpArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID];
        }
        if (array_key_exists(Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID, $taxArray)) {
            $defaultTaxId = $taxArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID];
        }

        // Initialize new data for product
        $productData = $integrationProduct["extra_data"];
        $productData["default_tax_class_id"] = $defaultTaxId;
        $productData["suggested_price"] = $suggestedPrice - 0;
        $productData["stock"] = [
            "stock_balance" => intval($integrationProduct["stock_data"]["stock_balance"])
        ];

        // Check if we have needed values
        if (empty($suggestedPrice)) {
            if ($productData["visma_status"] != self::VISMA_PRODUCT_STATUS_EXPIRED) {
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no suggested_price!", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Product [{$integrationProduct["sku"]}] have no suggested price!",
                    "Check product [{$integrationProduct["sku"]}] for missing suggested price for currency " . Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID . "\n",
                    ["it", "products"]
                );
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }
        }
        if ($defaultTaxId === null) {
            if ($productData["visma_status"] != self::VISMA_PRODUCT_STATUS_EXPIRED) {
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no default tax id!", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Product [{$integrationProduct["sku"]}] have no default tax id!",
                    "Check product [{$integrationProduct["sku"]}] for missing tax for currency " . Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID . "\n",
                    ["it", "products"]
                );
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }
        }
        if (empty($taxArray)) {
            if ($productData["visma_status"] != self::VISMA_PRODUCT_STATUS_EXPIRED) {
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no tax for any currency!", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Product [{$integrationProduct["sku"]}] have no tax for any currency!",
                    "Check product [{$integrationProduct["sku"]}] for missing tax id\n",
                    ["it", "products"]
                );
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }
        }

        // Calculate stock for X-products.
        $matches = [];
        if (preg_match("/^([0-9]+)X([0-9]+)$/", $integrationProduct["sku"], $matches) === 1) {
            if (!empty($matches) && count($matches) == 3) {

                $realSku = intval($matches[1]);
                $multiplier = intval($matches[2]);
                $realStockBalance = Mage::helper($this->_defaultHelper)->getVismaProductStock($realSku);

                if (empty($realStockBalance)) {
                    $productData["stock"]["stock_balance"] = 0;
                } else {
                    $productData["stock"]["stock_balance"] = floor($realStockBalance / $multiplier);
                }

                Mage::helper($this->_defaultHelper)->log("Found an 'X-Product': Real sku [{$realSku}] with {$realStockBalance} / {$multiplier} = {$productData["stock"]["stock_balance"]} in stock");
            }
        }

        // Get Magento store_id from Visma store_id and currency
        $productWebsites = Mage::helper($this->_defaultHelper)->getMagentoWebsiteIdsForProduct($integrationProduct["price_data"]);
        if (empty($productWebsites)) {
            Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no active stores in this Magento, skipping it.");
            $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
            return true;
        }

        $productData["magento_id"] = Mage::getModel("catalog/product")->getIdBySku($integrationProduct["sku"]);
        $isNew = empty($productData["magento_id"]);

        $productData["stock"]["use_config_manage_stock"] = 0; // Always override manage_stock
        switch (intval($productData["visma_status"])) {
            case self::VISMA_PRODUCT_STATUS_ACTIVE:
                $productData["stock"]["is_in_stock"] = Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK;
                $productData["stock"]["manage_stock"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_NO;
                $productData["stock"]["qty"] = Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY;
                $productData["magento_status"] = Mage_Catalog_Model_Product_Status::STATUS_ENABLED;
                break;

            case self::VISMA_PRODUCT_STATUS_TURN_OFF:
            case self::VISMA_PRODUCT_STATUS_LAST:
                $productData["stock"]["is_in_stock"] = $productData["stock"]["stock_balance"] > 0 ? Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK : Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK;
                $productData["stock"]["manage_stock"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES;
                $productData["stock"]["qty"] = $productData["stock"]["stock_balance"];
                $productData["magento_status"] = Mage_Catalog_Model_Product_Status::STATUS_ENABLED;
                break;

            case self::VISMA_PRODUCT_STATUS_EXPIRED:
                $productData["stock"]["is_in_stock"] = Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK;
                $productData["stock"]["manage_stock"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES;
                $productData["stock"]["qty"] = 0;
                $productData["magento_status"] = Mage_Catalog_Model_Product_Status::STATUS_DISABLED;
                $integrationProduct["extra_data"]["estimated_delivery_date"] = "";
                if ($isNew) {
                    Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] is new but have Visma status '{$productData["visma_status"]}', skipping it.");
                    $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
                    return true;
                }
                break;

            case self::VISMA_PRODUCT_STATUS_SOLD_OUT:
                $productData["stock"]["is_in_stock"] = $productData["stock"]["stock_balance"] > 0 ? Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK : Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK;
                $productData["stock"]["manage_stock"] = Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES;
                $productData["stock"]["qty"] = $productData["stock"]["stock_balance"];
                $productData["magento_status"] = Mage_Catalog_Model_Product_Status::STATUS_ENABLED;
                if ($isNew) {
                    Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] is new but have Visma status '{$productData["visma_status"]}', skipping it.");
                    $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
                    return true;
                }
                break;

            case self::VISMA_PRODUCT_STATUS_ARCHIVED:
            case self::VISMA_PRODUCT_STATUS_SEASON:
                // skip these products
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have status '{$productData["visma_status"]}', skipping it.");
                $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
                return true;

            default:
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have unknown status '{$productData["visma_status"]}', skipping it.", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Product [{$integrationProduct["sku"]}] have unknown status",
                    print_r($integrationProduct, true),
                    ["it"]
                );
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
        }

        // Not needed below, use $productData["stock"]["qty"] instead.
        unset($productData["stock"]["stock_balance"]);

        // create new product if needed
        if ($isNew) {
            // New products always start out as disabled
            $productData["magento_status"] = Mage_Catalog_Model_Product_Status::STATUS_DISABLED;

            if (Mage::getStoreConfig("integration/products/new_products")) {
                Mage::helper($this->_defaultHelper)->log("Found new product [{$integrationProduct["sku"]}], creating it.");

                $globalProduct = Mage::getModel("catalog/product");
                $globalProduct->setStoreId(0);
                $globalProduct->setSku($integrationProduct["sku"]);
                $globalProduct->setAttributeSetId(Mage::getModel("catalog/product")->getDefaultAttributeSetId());
                $globalProduct->setTypeId($integrationProduct["extra_data"]["product_type"] == self::VISMA_PRODUCT_TYPE_VIRTUAL ? Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL : Mage_Catalog_Model_Product_Type::TYPE_SIMPLE);
                $globalProduct->setCategoryIds([]);
                $globalProduct->setName($productData["name"]);
                $globalProduct->setDescription($productData["name"]);
                $globalProduct->setShortDescription($productData["name"]);
                $globalProduct->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH); // catalog & search
                $globalProduct->setStatus($productData["magento_status"]);
                $globalProduct->setWeight(0.0);
                $globalProduct->setPrice(0.0);
                $globalProduct->setTaxClassId($productData["default_tax_class_id"]);
                $globalProduct->setStockData( ["use_config_manage_stock" => 0, "manage_stock" => 0 ]);
                if (!empty($productData["magento_manufacturer_id"])) {
                    $globalProduct->setManufacturer($productData["magento_manufacturer_id"]);
                }

                $this->setDefaultNewProductExtraData($integrationProduct, $globalProduct, $productData);

                $globalProduct->save();

                // Wait for some time to let make Magento do its stuff
                usleep(Awardit_Integration_Helper_Data::PRODUCT_SAVE_SLEEP_TIME);

                // Make sure the product later on is a fresh reload
                unset($globalProduct);

                 // Update magento_id
                $productData["magento_id"] = Mage::getModel("catalog/product")->getIdBySku($integrationProduct["sku"]);

            } else {

                Mage::helper($this->_defaultHelper)->log("Found new product [{$integrationProduct["sku"]}], skipping it.");
                $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
                return true;

            }
        }

        $globalProduct = Mage::getModel("catalog/product")->load($productData["magento_id"]);
        $isUpdated = false;

        // Only disable products, never enable them
        if ($globalProduct->getData("status") == Mage_Catalog_Model_Product_Status::STATUS_ENABLED && $productData["magento_status"] == Mage_Catalog_Model_Product_Status::STATUS_DISABLED) {
            $globalProduct->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
            $isUpdated = true;

            // If this product existed before and it is about to be disabled, just disable it and do not update anything else.
            if (!$isNew) {
                // Make sure all values in local stores are deleted, only use default value
                $this->deleteLocalAttributeValues($globalProduct->getId(), "status");
                $globalProduct->save();
                $integrationProduct["import_status"] = self::IMPORT_STATUS_SKIP;
                Mage::helper($this->_defaultHelper)->log("Product was set to be disabled, do not update any more.");
                return true;
            }
        }

        if ($globalProduct->getData("price") != $productData["suggested_price"]) {
            if ($productData["suggested_price"] > 0) {
                $globalProduct->setPrice($productData["suggested_price"]);
                $isUpdated = true;
            } else {
                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no suggested_price!", LOG_ERR);
                Mage::helper($this->_defaultHelper)->log("productData: " . json_encode($productData), LOG_DEBUG);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Product [{$integrationProduct["sku"]}] have no suggested_price!",
                    "Check product [{$integrationProduct["sku"]}] for missing suggested price for currency " . Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID . "\n",
                    ["it", "products"]
                );
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }
        }

        if ($globalProduct->getData("msrp") != $msrpArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID]) {
            $globalProduct->setData("msrp", $msrpArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID]);
            $isUpdated = true;
        }

        if ($globalProduct->getTaxClassId() != $taxArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID]) {
            $globalProduct->setTaxClassId($taxArray[Awardit_Integration_Helper_Data::DEFAULT_CURRENCY_ID]);
            $isUpdated = true;
        }

        // Check all extra_data to see if product needs to be updated
        if (!empty($integrationProduct["extra_data"])) {
            foreach ($integrationProduct["extra_data"] as $key => $val) {

                if ($key === "manufacturer_name") {
                    $manufacturerId = empty($val) ? null : Mage::helper($this->_defaultHelper)->getManufacturerId($val);
                    if (!empty($manufacturerId) && $globalProduct->getManufacturer() != $manufacturerId) {
                        $globalProduct->setManufacturer($manufacturerId);
                        $isUpdated = true;
                    }
                    continue;
                }

                // Only set attributes that actually exist for product
                if ($globalProduct->getResource()->getAttribute($key)) {
                    switch ($key) {
                        case "weight":
                            $val = floatval($val);
                            if ($val < 0) {
                                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have negative value for {$key}!", LOG_ERR);
                                Mage::helper($this->_defaultHelper)->log("productData: " . json_encode($productData), LOG_DEBUG);
                                $val = abs($val);
                            }

                        default:
                            if ($globalProduct->getData($key) != $val) {
                                $globalProduct->setData($key, $val);
                                $isUpdated = true;
                            }
                    }
                }
            }
        }

        // Set local instance product data
        if ($this->setDefaultExistingProductExtraData($integrationProduct, $globalProduct, $productData)) {
            $isUpdated = true;
        }

        $stockItem = Mage::getModel("cataloginventory/stock_item")->loadByProduct($globalProduct);
        $clearExternalStock = false;
        if ($stockItem && ($isNew || $this->_updateStock)) {
            if (!$stockItem->getItemId()) {
                Mage::helper($this->_defaultHelper)->log("Product was missing stock data, creating new.");
                $stockItem->assignProduct($globalProduct);
            }
            $_raw_stock_data = $stockItem->getData();

            // Do a quick check if we need to update the product. Don't update if its not absolutely necessary
            if ($isNew || ($productData["stock"]["qty"] != $_raw_stock_data["qty"] && $_raw_stock_data["qty"] < 1000000) || $productData["stock"]["is_in_stock"] != $_raw_stock_data["is_in_stock"] || $productData["stock"]["manage_stock"] != $_raw_stock_data["manage_stock"]) {

                Mage::helper($this->_defaultHelper)->log("Stock needs update: " . json_encode($productData["stock"]));
                $stockItem->setData("use_config_manage_stock", $productData["stock"]["use_config_manage_stock"]);
                $stockItem->setData("manage_stock", $productData["stock"]["manage_stock"]);
                $stockItem->setData("is_in_stock", $productData["stock"]["is_in_stock"]);
                $stockItem->setData("qty", $productData["stock"]["qty"]);
                try {
                    $stockItem->save();
                } catch (Exception $exception) {
                    Mage::helper($this->_defaultHelper)->logException($exception, "Exception updating stock!");
                    $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                    return false;
                }
                $clearExternalStock = empty($productData["stock"]["manage_stock"]);
                $isUpdated = true;
            } else {
                Mage::helper($this->_defaultHelper)->log("No update of stock needed.");
            }
        }
        if ($clearExternalStock) {
            // Clear external stock from product that is set to not handle stock, since product status is global
            $sqlQuery = "DELETE FROM stock WHERE sku = '{$integrationProduct["sku"]}'";
            $affectedRows = Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->exec($sqlQuery);
            Mage::helper($this->_defaultHelper)->log("Clearing external stock data for [{$integrationProduct["sku"]}]. Affected rows: {$affectedRows}");
        }

        // Handle config for websites
        $currentWebsites = $globalProduct->getWebsiteIds();
        sort($currentWebsites, SORT_NUMERIC);
        if (count(array_diff($currentWebsites, $productWebsites)) > 0 || count(array_diff($productWebsites, $currentWebsites)) > 0) {

            Mage::helper($this->_defaultHelper)->log("Current websites for [{$integrationProduct["sku"]}]: " . implode(",", $currentWebsites));
            Mage::helper($this->_defaultHelper)->log("Wanted websites for  [{$integrationProduct["sku"]}]: " . implode(",", $productWebsites));

            $globalProduct->setWebsiteIds($productWebsites);

            // We need to save product here, beacuse we found out that we can't set prices on newley added websites

            try {
                $globalProduct->save();
                usleep(Awardit_Integration_Helper_Data::PRODUCT_SAVE_SLEEP_TIME);
            } catch (Exception $exception) {
                Mage::helper($this->_defaultHelper)->logException($exception, "Exception while saving!");
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }

             // Make sure the product is a fresh reload
            unset($globalProduct);

            $globalProduct = Mage::getModel("catalog/product")->load($productData["magento_id"]);
            Mage::helper($this->_defaultHelper)->log("Successfully saved and reloaded product after updating websites.");
            $isUpdated = false;
        }

        // Begin: Set group prices
        $rawProductGroupPrices = $globalProduct->getData("group_price");

        // Sometimes we get empty group prices, try to get them "the other way"
        if (empty($rawProductGroupPrices)) {
            $globalProduct->getGroupPrice();
            $rawProductGroupPrices = $globalProduct->getData("group_price");
        }

        if (!is_array($rawProductGroupPrices)) {
            $rawProductGroupPrices = [];
        }

        $disableProduct = false;
        $uniqueGroupPrices = [];
        $adminEmail = [
            "send_email" => false,
            "zero_price" => [
                "title" => "Product is missing price, check for empty price rows in Visma price table where:\n",
                "body" => ""
            ],
            "same_price" => [
                "title" => "Product have multiple price rows with same price, check for rows in Visma price table where:\n",
                "body" => ""
            ],
            "different_price" => [
                "title" => "Product have multiple price rows with different price, check for rows in Visma price table where:\n",
                "body" => ""
            ],
            "missing_tax" => [
                "title" => "Product have no tax id, check for rows in Visma price table where:\n",
                "body" => ""
            ],
        ];

        foreach ($productWebsites as $magentoWebsiteId) {
            $storeIds = Mage::helper($this->_defaultHelper)->getMagentoStoreIds($magentoWebsiteId);
            foreach ($storeIds as $localStoreId) {
                $vismaCurrencyId = intval(Mage::helper($this->_defaultHelper)->getVismaCurrencyId(Mage::getStoreConfig("currency/options/base", $localStoreId)));
                $vismaShopId = intval(Mage::getStoreConfig("integration/general/visma_shop_id", $localStoreId));
                $normalPriceGroupId = Mage::getStoreConfig("integration/products/normal_price_from_group", $localStoreId);
                $specialPriceGroupId = Mage::getStoreConfig("integration/products/special_price_from_group", $localStoreId);
                foreach ($integrationProduct["price_data"] as $index => $priceData) {
                    if (!empty($priceData["customer_group_id"]) && $priceData["shop_id"] == $vismaShopId && $priceData["currency_id"] == $vismaCurrencyId) {

                        // Check if price is 0 (zero)
                        if ($priceData["price"] == 0) {

                            // Try to get new price from some kind of override
                            $localProduct = Mage::getModel("catalog/product")->setStoreId($localStoreId)->load($productData["magento_id"]);
                            $newPrice = $this->getNewPriceIfZero($localProduct, $localStoreId, $priceData);

                            // Check if we got new price
                            if ($newPrice > 0) {

                                // Use new price
                                $integrationProduct["price_data"][$index]["overridden_price"] = $priceData["price"];
                                $priceData["price"] = $newPrice;
                                $integrationProduct["price_data"][$index]["price"] = $newPrice;

                            } else {

                                // Check if it's ok to use empty price
                                if (!Mage::helper($this->_defaultHelper)->isZeroPriceProduct($localProduct->getSku())) {

                                    // DO NOT ALLOW PRICE TO BE ZERO! mark it to be disabled and send email about this.
                                    Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have empty price in Visma!". LOG_ERR);
                                    $adminEmail["zero_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                    $adminEmail["send_email"] = true;
                                    $disableProduct = true;
                                    continue; // Next price row
                                }
                            }
                        }

                        if ($priceData["customer_group_id"] == $normalPriceGroupId) {
                            // Use this group price as normal price
                            continue;
                        } elseif ($priceData["customer_group_id"] == $specialPriceGroupId) {
                            // Use this group price as special price
                            continue;
                        }

                        // Add group price if visma customer group translates to one or more Magento customer groups
                        $magentoGroups = Mage::helper($this->_defaultHelper)->getMagentoGroupIds($priceData["customer_group_id"]);
                        foreach ($magentoGroups as $magentoGroupId) {
                            $key = "{$magentoWebsiteId}-{$magentoGroupId}";
                            if (!array_key_exists($key, $uniqueGroupPrices)) {

                                // Only add new group prices if there is nothing wrong with the price
                                $uniqueGroupPrices[$key] = [
                                    "website_id" => $magentoWebsiteId,
                                    "cust_group" => $magentoGroupId,
                                    "price" => $priceData["price"],
                                ];

                            } elseif($uniqueGroupPrices[$key]["price"] == $priceData["price"]) {

                                // Found multiple rows with same price, warn about this
                                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have multiple rows with same price in Visma!". LOG_ERR);
                                $adminEmail["same_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                $adminEmail["send_email"] = true;

                            } else {

                                // Found multiple rows with different prices, warn about this
                                Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have multiple rows with different price in Visma!", LOG_ERR);
                                $adminEmail["different_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                $adminEmail["send_email"] = true;
                                $disableProduct = true;
                                break;
                            }
                        }
                    }
                }
            }
        }

        if (!$disableProduct) {

            // Find out if group prices needs update
            $updateGroupPrices = false;
            if (!empty($uniqueGroupPrices)) {

                if (!empty($rawProductGroupPrices)) {

                    // We have both new and old group prices, check if there is any mismatch
                    if (count($uniqueGroupPrices) != count($rawProductGroupPrices)) {

                        // We have different amount of new and old group prices, mark group prices to be updated
                        $updateGroupPrices = true;

                    } else {

                        // We have same amount of new and old group prices, check each
                        foreach ($uniqueGroupPrices as $newGroupPrice) {
                            foreach ($rawProductGroupPrices as $oldGroupPrice) {
                                if ($oldGroupPrice["website_id"] == $newGroupPrice["website_id"] && $oldGroupPrice["cust_group"] == $newGroupPrice["cust_group"]) {
                                    if ($oldGroupPrice["price"] != $newGroupPrice["price"]) {

                                        // Yes, new group price have different value than old group price, mark group prices to be updated
                                        $updateGroupPrices = true;
                                        break 2; // bail out of new group price loop

                                    }

                                    break; // Bail out of old group price loop
                                }
                            }
                        }
                    }
                } else {

                    // If we have new group prices but no old ones, mark group prices to be updated
                    $updateGroupPrices = true;

                }

            } elseif (!empty($rawProductGroupPrices)) {

                // We have old group prices but no new ones, mark group prices to be updated
                $updateGroupPrices = true;

            }

            if ($updateGroupPrices) {
                $globalProduct->setData("group_price", array_values($uniqueGroupPrices));
                $isUpdated = true;
            }
        }
        // End: Group prices

        // Begin: set local price and attributes
        foreach ($productWebsites as $magentoWebsiteId) {

            $storeIds = Mage::helper($this->_defaultHelper)->getMagentoStoreIds($magentoWebsiteId);
            foreach ($storeIds as $localStoreId) {

                $localProduct = Mage::getModel("catalog/product")->setStoreId($localStoreId)->load($productData["magento_id"]);

                $normalPriceGroupId = Mage::getStoreConfig("integration/products/normal_price_from_group", $localStoreId);
                $specialPriceGroupId = Mage::getStoreConfig("integration/products/special_price_from_group", $localStoreId);
                $vismaCurrencyId = intval(Mage::helper($this->_defaultHelper)->getVismaCurrencyId(Mage::getStoreConfig("currency/options/base", $localStoreId)));
                $vismaShopId = Mage::getStoreConfig("integration/general/visma_shop_id", $localStoreId);
                $vismaCountryId = Mage::helper($this->_defaultHelper)->getVismaCountryId(Mage::getStoreConfig("general/store_information/merchant_country", $localStoreId));
                $clearSpecialPrice = true;
                $defaultMSRP = 0;
                $newPriceSet = null;

                // Loop through all price rows
                foreach ($integrationProduct["price_data"] as $index => $priceData) {

                    // Check if we found price row matching wanted shop id and currency
                    if ($priceData["shop_id"] == $vismaShopId && $priceData["currency_id"] == $vismaCurrencyId) {

                        // Check if we found price row for empty customer group id OR configured customer group id for normal price override
                        if ((empty($priceData["customer_group_id"]) && empty($normalPriceGroupId)) || ($priceData["customer_group_id"] == $normalPriceGroupId)) {

                            // Check if we found price row with empty customer group id AND price is 0 (zero)
                            if (empty($priceData["customer_group_id"]) && $priceData["price"] == 0) {

                                // Try to get new price from some kind of override
                                $newPrice = $this->getNewPriceIfZero($localProduct, $localStoreId, $priceData);

                                // Check if we got new price
                                if ($newPrice > 0) {

                                    // Use new price
                                    $integrationProduct["price_data"][$index]["overridden_price"] = $priceData["price"];
                                    $priceData["price"] = $newPrice;
                                    $integrationProduct["price_data"][$index]["price"] = $newPrice;

                                } else {

                                    // Check if it's ok to use empty price
                                    if (!Mage::helper($this->_defaultHelper)->isZeroPriceProduct($localProduct->getSku())) {

                                        // DO NOT ALLOW PRICE TO BE ZERO! Disable product in local config and send email about this.
                                        Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have empty price in Visma!". LOG_ERR);
                                        $adminEmail["zero_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                        $adminEmail["send_email"] = true;
                                        $disableProduct = true;
                                        continue 2; // Next store
                                    }
                                }
                            }

                            // Use price as default MSRP
                            $defaultMSRP = $priceData["price"];

                            // Check if we are supposed to update price AND price needs to be updated
                            if ($this->getUpdatePrice($localProduct, $localStoreId, $priceData) && $priceData["price"] != $localProduct->getData("price")) {

                                if ($newPriceSet === null) {

                                    // Set price
                                    $localProduct->setData("price", $priceData["price"])->getResource()->saveAttribute($localProduct, "price");
                                    $newPriceSet = $priceData["price"];
                                    $isUpdated = true;

                                } elseif($newPriceSet == $priceData["price"]) {

                                    // Price was previously updated with same price => more than one row in Visma price table with same currency/shop/customer group
                                    Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have multiple rows with same price in Visma!", LOG_ERR);
                                    $adminEmail["same_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                    $adminEmail["send_email"] = true;

                                } else {

                                    // Price was previously updated with different price => more than one row in Visma price table with same currency/shop/customer group
                                    Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have multiple rows with different price in Visma!", LOG_ERR);
                                    $adminEmail["different_price"]["body"] .= "Shop id = {$priceData["shop_id"]}, Currency = {$priceData["currency_id"]}, Customer group = {$priceData["customer_group_id"]}\n";
                                    $adminEmail["send_email"] = true;
                                    $disableProduct = true;
                                    continue 2; // Next store

                                }
                            }
                        }

                        // Check if we got price row with customer group id mathing configured customer group id for special price override
                        if (!empty($specialPriceGroupId) && $priceData["customer_group_id"] == $specialPriceGroupId) {

                            // Check to se if we need to update special price
                            if ($priceData["price"] != $localProduct->getData("special_price")) {

                                // Set special price on product
                                $localProduct->setData("special_price", $priceData["price"])->getResource()->saveAttribute($localProduct, "special_price");
                                $localProduct->setData("special_from_date", date("Y-m-d"))->getResource()->saveAttribute($localProduct, "special_from_date");
                                $isUpdated = true;
                            }

                            // We found special price override, no need to clear special price.
                            $clearSpecialPrice = false;
                        }

                    }
                }

                // Set tax id.
                $magentoTaxId = 0;
                if ($vismaCurrencyId == 99 && $EURTaxId > 0) {
                    // Use country to get actual tax id when currency is EUR (99)
                    $magentoTaxId = Mage::helper($this->_defaultHelper)->getMagentoTaxIdFromVismaTaxId($EURTaxId, $vismaCurrencyId, $vismaCountryId);
                }
                // If we did not get any tax id for currency 99 using country id, get one the normal way.
                if (empty($magentoTaxId)) {
                    if (array_key_exists($vismaCurrencyId, $taxArray)) {
                        $magentoTaxId = $taxArray[$vismaCurrencyId];
                    } else {
                        // Tax class id is missing for current currency
                        Mage::helper($this->_defaultHelper)->log("Product [{$integrationProduct["sku"]}] have no tax id for currency {$vismaCurrencyId}", LOG_ERR);
                        $adminEmail["missing_tax"]["body"] .= "Currency = {$vismaCurrencyId}\n";
                        $adminEmail["send_email"] = true;
                        $magentoTaxId = $localProduct->getData("tax_class_id");
                    }
                }
                if ($magentoTaxId != $localProduct->getData("tax_class_id")) {
                    $localProduct->setData("tax_class_id", $magentoTaxId)->getResource()->saveAttribute($localProduct, "tax_class_id");
                    $isUpdated = true;
                }

                // Set msrp
                if (array_key_exists($vismaCurrencyId, $msrpArray)) {
                    if ($msrpArray[$vismaCurrencyId] != $localProduct->getData("msrp")) {
                        $localProduct->setData("msrp", $msrpArray[$vismaCurrencyId])->getResource()->saveAttribute($localProduct, "msrp");
                        $isUpdated = true;
                    }
                } else {
                    // If no msrp exists for current currency, use current price.
                    if ($localProduct->getData("msrp") != $defaultMSRP) {
                        $localProduct->setData("msrp", $defaultMSRP)->getResource()->saveAttribute($localProduct, "msrp");
                        $isUpdated = true;
                    }
                }

                if (!empty($specialPriceGroupId) && $clearSpecialPrice && $localProduct->getData("special_price") > 0) {
                    // Clear special price and both special price dates
                    $localProduct->setData("special_price", "")->getResource()->saveAttribute($localProduct, "special_price");
                    $localProduct->setData("special_to_date", "")->getResource()->saveAttribute($localProduct, "special_to_date");
                    $localProduct->setData("special_from_date", "")->getResource()->saveAttribute($localProduct, "special_from_date");
                    $isUpdated = true;
                }

                // Set local product extra data
                if ($this->setLocalProductExtraData($integrationProduct, $localProduct, $productData)) {
                    $isUpdated = true;
                }
            }
        }
        // End: set local price and attributes

        // Check if we have admin email to send
        if ($adminEmail["send_email"]) {
            if ($disableProduct) {
                $adminEmailBody = "Product has been disabled! Someone needs to fix all errors in Visma price table and let the product be imported without errors before it is safe to enable it again.\n\n";
            } else {
                $adminEmailBody = "Product has been saved and is usable, but it is suggested to fix all errors in Visma price table as soon as possible.\n\n";
            }

            foreach ($adminEmail as $mail) {
                if (!empty($mail["body"])) {
                    $adminEmailBody .= $mail["title"] . $mail["body"] . "\n";
                }
            }

            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Product [{$integrationProduct["sku"]}] have price problems!",
                $adminEmailBody,
                ["it", "products"]
            );
        }

        if ($disableProduct) {

            // Disable product if it has errors
            try {
                $globalProduct->setData("status", Mage_Catalog_Model_Product_Status::STATUS_DISABLED)->getResource()->saveAttribute($localProduct, "status");
                $globalProduct->save();
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
            } catch (Exception $exception) {
                Mage::helper($this->_defaultHelper)->logException($exception, "Exception while saving!");
            }
            return false;

        } elseif ($isUpdated) {

            // Save global product if it is updated
            try {
                $globalProduct->save();
                Mage::helper($this->_defaultHelper)->log("Successfully saved product.");
            } catch (Exception $exception) {
                Mage::helper($this->_defaultHelper)->logException($exception, "Exception while saving!");
                $integrationProduct["import_status"] = self::IMPORT_STATUS_ERROR;
                return false;
            }

        } else {
            Mage::helper($this->_defaultHelper)->log("No save needed.");
        }

        $integrationProduct["import_status"] = self::IMPORT_STATUS_COMPLETE;
        $this->doLocalPostProcessingOfData($integrationProduct, $globalProduct);

        return true;
    }

    public function deleteLocalAttributeValues($productId, $attributeCode)
    {
        // Fetch attribute info for wanted attribute code
        $sqlQuery1 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => $attributeCode
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery1, $params);

        // If no attribute was found, return
        if (empty($eava)) {
            return null;
        }

        // Find table name for matching backend type
        $table = "";
        switch ($eava["backend_type"]) {
            case "int":
                $table = "catalog_product_entity_int";
                break;

            case "datetime":
                $table = "catalog_product_entity_datetime";
                break;

            case "decimal":
                $table = "catalog_product_entity_decimal";
                break;

            case "text":
                $table = "catalog_product_entity_text";
                break;

            case "varchar":
                $table = "catalog_product_entity_varchar";
                break;

            default:
                break;
        }

        // If no matching table was found, return
        if (empty($table)) {
            return null;
        }

        // Now try to delete all attribute values except from default store
        $sqlQuery2 = "DELETE cpex.* FROM {$table} cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery2);
        $stmt->bindValue("typeId", $params["typeId"]);
        $stmt->bindValue("attributeId", intval($eava["attribute_id"]));
        $stmt->bindValue("productId", intval($productId));
        $stmt->execute();
        $affectedRows = $stmt->rowCount();
        if ($affectedRows > 0) {
            Mage::helper($this->_defaultHelper)->log("Removed {$affectedRows} rows of local values for attribute '{$attributeCode}'");
        }

        return $affectedRows;

    }

    public function doLocalPreProcessingOfData(&$integrationProduct)
    {
        return;
    }

    public function doLocalPostProcessingOfData($integrationProduct, $product)
    {
        return;
    }

    public function setDefaultNewProductExtraData(&$integrationProduct, &$product, &$productData)
    {
        return;
    }

    public function setDefaultExistingProductExtraData($integrationProduct, &$product, &$productData)
    {
        return false;
    }

    public function setLocalProductExtraData($integrationProduct, &$product, &$productData)
    {
        return false;
    }

    public function getNewPriceIfZero($product, $storeId, $priceData)
    {
        return 0.0;
    }

}
