<?php

class Awardit_Integration_Model_Override_CatalogInventoryStockItem extends Mage_CatalogInventory_Model_Stock_Item
{

    private $_externalQty = [];
    private $_localQty = [];
    private $_externalStockEnabled = false;

    public function _construct()
    {
        $this->_externalStockEnabled = Mage::getStoreConfig('integration/external_stock/enabled', Mage::app()->getStore()->getId()) ? true : false;
        parent::_construct();
    }

    public function getQty()
    {
        // Get the local qty for later use
        $localQty = parent::getData('qty');

        // Check to see if external stock handling and stock management is activated
        if ($this->_externalStockEnabled && $this->getManageStock()) {

            // If so, do some setup
            $productId = $this->getProductId();

            // Check to se if product have cached local value
            if (array_key_exists($productId, $this->_localQty)) {
                return $localQty; // If so, return local value
            }

            // Check if qty is cached
            if (array_key_exists($productId, $this->_externalQty)) {
                return $this->_externalQty[$productId]; // If so, return cached value
            }

            // Try to fetch product (to get sku)
            $product = $this->getProduct();

            try {
                // Check if we already know about current product
                if (empty($product)) {

                    // If not, fetch SKU using product_id from stock_item
                    $sqlQuery = "SELECT sku FROM catalog_product_entity WHERE entity_id = ?";
                    $sku = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchOne($sqlQuery, $productId);
                } else {

                    // If so, get SKU from that product
                    $sku = $product->getSku();
                }
                $warehouseId = Mage::getStoreConfig('integration/external_stock/warehouse_id') - 0;
                $magentoId = Mage::getStoreConfig('integration/general/magento_instance') - 0;

                // Get qty for current product, adjusted for local variations based on warehouse, Magento instance and store_id.
                $sqlQuery = "SELECT stock.sku, SUM(stock.qty) AS qty FROM stock WHERE stock.warehouse_id = ? AND (stock.magento_id = 0 OR stock.magento_id = ?) AND (stock.store_id = 0 OR stock.store_id = ?) AND stock.sku = ? GROUP BY stock.sku";
                $params = [
                    $warehouseId,
                    $magentoId,
                    Mage::app()->getStore()->getId() - 0,
                    $sku
                ];
                $result = Mage::getSingleton('core/resource')->getConnection('stock_handler_read')->fetchRow($sqlQuery, $params);
            } catch (Exception $exception) {
                Mage::helper('integration')->logException($exception, "Exception while fetching data from external stock for product #{$productId}!");
                return $localQty;
            }

            // Check to se if current sku have external stock data
            if (empty($result['sku'])) {

                // If not, use loaclQty and cache this for later use (speedup)
                $this->_localQty[$productId] = $localQty;
                return $localQty;
            }

            // Cache the value to speed up subsequent queries
            $this->_externalQty[$productId] = $result['qty'];
            $productNeedsUpdate = false;

            if (intval($localQty) != intval($this->_externalQty[$productId])) {
                // Check if local qty is different than external qty

                // If so, update local stock
                $this->setQty($this->_externalQty[$productId]);
                $productNeedsUpdate = true;

            } elseif ($this->getIsInStock() && $this->_externalQty[$productId] <= $this->getMinQty()) {
                // Check to see product is in-stock when it shouldn't be

                // If so, set product to be out-of-stock
                $productNeedsUpdate = true;

            } elseif (!$this->getIsInStock() && $this->_externalQty[$productId] > $this->getMinQty()) {
                // Check to see product is out-of-stock when it shouldn't be

                // If so, set product to be in-stock
                $productNeedsUpdate = true;
            }

            // Check if product needs to be updated
            if ($productNeedsUpdate) {
                try {
                    // Save product data to table for later update
                    $sqlQuery = "INSERT INTO product_update (product_id, warehouse_id, magento_id, sku, tagged_at) VALUES ({$productId}, {$warehouseId}, {$magentoId}, '{$sku}', now()) ON DUPLICATE KEY UPDATE tagged_at = VALUES(tagged_at)";
                    Mage::getSingleton('core/resource')->getConnection('stock_handler_write')->exec($sqlQuery);
                } catch (Exception $exception) {
                    Mage::helper('integration')->logException($exception, "Exception while inserting data to product_update!\n{$sqlQuery}");
                }
            }

            return $this->_externalQty[$productId];
        }

        return $localQty;
    }

    public function getData($key = '', $index = null)
    {
        // Get the real data from parent
        $data = parent::getData($key, $index);

        // Check if external stock handling is activated
        if ($this->_externalStockEnabled) {

            // If so, check if data contains an array with qty in it
            if (!empty($data) && is_array($data) && array_key_exists('qty', $data)) {

                // If so, update data with the correct qty
                $data['qty'] = $this->getQty();

                // Check if data contains only qty
            } elseif (!empty($data) && $key == 'qty') {

                // If so, update data with the correct qty
                $data = $this->getQty();
            }
        }

        return $data;
    }

    // Needed to override this function to avoid infinite recursive calls
    public function getManageStock()
    {
        if ($this->getUseConfigManageStock()) {
            return (int) Mage::getStoreConfigFlag(self::XML_PATH_MANAGE_STOCK);
        }

        return parent::getData('manage_stock');
    }

    public function subtractQty($qty)
    {
        // Check to see if external stock handling is activated
        if ($this->_externalStockEnabled) {

            // If so, check if we are configured to subtract/add qty
            if ($this->canSubtractQty()) {

                // If so, do some setup
                $product = $this->getProduct();
                $productId = $this->getData('product_id');

                try {
                    // Check if we already know about current product
                    if (empty($product)) {

                        // If not, fetch SKU using product_id from stock_item
                        $sqlQuery = "SELECT sku FROM catalog_product_entity WHERE entity_id = ?";
                        $sku = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchOne($sqlQuery, $productId);
                    } else {

                        // If so, get SKU from that product
                        $sku = $product->getSku();
                    }

                    // Check if product have external stock
                    if ($this->_doesProductHaveExternalStock($sku)) {

                        // If so, decrease external stock using the current warehouse_id and only if we can get the whole qty.
                        $warehouseId = Mage::getStoreConfig('integration/external_stock/warehouse_id') - 0;
                        $sqlQuery = "UPDATE stock SET qty = qty - {$qty} WHERE warehouse_id = {$warehouseId} AND magento_id = 0 AND store_id = 0 AND sku = '{$sku}' AND qty >= {$qty}";
                        $affectedRows = Mage::getSingleton('core/resource')->getConnection('stock_handler_write')->exec($sqlQuery);

                        // Check to see if update went well
                        if ($affectedRows < 1) {

                            // If not, log this and throw exception
                            $helper = Mage::helper('integration');
                            $helper->log(LOG_ERR, "Unable to update stock for [{$sku}]");
                            Mage::throwException(Mage::helper('cataloginventory')->__('Not all products are available in the requested quantity'));
                        }

                        // Check if qty is cached
                        if (array_key_exists($productId, $this->_externalQty)) {
                            unset($this->_externalQty[$productId]); // If so, clear it, because it needs to be re-read later
                        }
                    }
                } catch (Exception $exception) {
                    Mage::helper('integration')->logException($exception, "Exception while subtracting {$qty} from external stock for product #{$productId}!");
                }
            }
        }

        return parent::subtractQty($qty);
    }

    public function addQty($qty)
    {
        // Check to see if external stock handling is activated
        if ($this->_externalStockEnabled) {

            // If so, check if we are configured to subtract/add qty
            if ($this->canSubtractQty()) {

                // Do some setup
                $product = $this->getProduct();
                $productId = $this->getData('product_id');

                try {
                    // Check if we already know about current product
                    if (empty($product)) {

                        // If not, fetch SKU using product_id from stock_item
                        $sqlQuery = "SELECT sku FROM catalog_product_entity WHERE entity_id = ?";
                        $sku = Mage::getSingleton('core/resource')->getConnection('core_read')->fetchOne($sqlQuery, $productId);
                    } else {

                        // If so, get SKU from that product
                        $sku = $product->getSku();
                    }

                    // Check if product have external stock
                    if ($this->_doesProductHaveExternalStock($sku)) {

                        // If so, increase external stock using the current warehouse_id.
                        $warehouseId = Mage::getStoreConfig('integration/external_stock/warehouse_id') - 0;
                        $sqlQuery = "UPDATE stock SET qty = qty + {$qty} WHERE warehouse_id = {$warehouseId} AND magento_id = 0 AND store_id = 0 AND sku = '{$sku}'";
                        $affectedRows = Mage::getSingleton('core/resource')->getConnection('stock_handler_write')->exec($sqlQuery);

                        // Check to see if update went well
                        if ($affectedRows < 1) {

                            // If not, log this
                            $helper = Mage::helper('integration');
                            $helper->log(LOG_ERR, "Unable to update stock for [{$sku}]");
                        } else {

                            // Otherwise, check if qty is cached
                            if (array_key_exists($productId, $this->_externalQty)) {

                                // If so, clear it, because it needs to be re-read later
                                unset($this->_externalQty[$productId]);
                            }
                        }
                    }
                } catch (Exception $exception) {
                    Mage::helper('integration')->logException($exception, "Exception while adding {$qty} to external stock for product #{$productId}!");
                }
            }
        }

        return parent::addQty($qty);
    }

    public function save()
    {
        // Check to see if external stock handling is activated
        if ($this->_externalStockEnabled) {

            // Check to see if stock is managed
            if ($this->getManageStock()) {

                // If so, do some setup
                $productId = parent::getData('product_id');

                // Check if qty is cached
                if (array_key_exists($productId, $this->_externalQty)) {

                    // Fetch local qty
                    $localQty = parent::getData('qty');

                    // Check if local and external qty differs
                    if ($localQty != $this->_externalQty[$productId]) {

                        // If so, set local qty to match external qty
                        parent::setQty($this->_externalQty[$productId]);
                    }
                }
            }
        }

        return parent::save();
    }

    protected function _doesProductHaveExternalStock($sku)
    {
        try {
            $sqlQuery = "SELECT stock.sku FROM stock WHERE stock.warehouse_id = ? AND stock.magento_id = 0 AND stock.store_id = 0 AND stock.sku = ?";
            $params = [
                Mage::getStoreConfig('integration/external_stock/warehouse_id') - 0,
                $sku
            ];
            $result = Mage::getSingleton('core/resource')->getConnection('stock_handler_read')->fetchOne($sqlQuery, $params);

            return !empty($result);
        } catch (Exception $exception) {
            Mage::helper('integration')->logException($exception, "Exception while checking if product [{$sku}] have external stock data!");
            return false;
        }
    }

}
