<?php

class Awardit_Integration_Model_Cli_Stock extends Awardit_Integration_Model_Cli {

    // -m=Stock -f=Sync
    public function CLI_Sync($param)
    {
        $this->syncLocalStock();
    }

    // -m=Stock -f=ProductUpdate
    public function CLI_UpdateProducts($param)
    {
        $this->updateProducts();
    }

    // -m=Stock -f=Reindex
    public function CLI_Reindex($param)
    {
        $this->doReindex();
    }

    public function syncLocalStock()
    {
        $syncStatus = false;
        Mage::helper($this->_defaultHelper)->log("Scanning for products that needs to sync local stock");
        try {
            $sqlQuery = "
                SELECT
                    tmp.sku,
                    tmp.product_id,
                    tmp.min_qty,
                    tmp.qty
                FROM (
                    SELECT
                        p.sku,
                        s.product_id,
                        CAST(s.min_qty AS unsigned) AS min_qty,
                        CAST(s.qty AS unsigned) AS qty,
                        SUM(i.value - 1) AS is_enabled
                    FROM catalog_product_entity p
                    JOIN cataloginventory_stock_item s ON s.product_id = p.entity_id
                    JOIN catalog_product_entity_int i ON i.entity_id = p.entity_id
                    JOIN eav_attribute a ON a.attribute_id = i.attribute_id
                    WHERE s.use_config_manage_stock = 0 AND s.manage_stock = 1 AND a.attribute_code = 'status'
                    GROUP BY p.sku
                    ORDER BY p.sku
                ) tmp
                WHERE tmp.is_enabled > 0
            ";
            $productData = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery);

            if (empty($productData)) {
                return;
            }

            $db = Mage::getSingleton("core/resource")->getConnection("stock_handler_read");
            $select = $db->select()
                    ->from(["s" => "stock"], ["s.sku", "external_qty" => "SUM(s.qty)"])
                    ->where("s.warehouse_id = ?", intval(Mage::getStoreConfig("integration/external_stock/warehouse_id")))
                    ->where("s.sku IN (?)", array_map(function($k) { return (string)$k; }, array_keys($productData)))
                    ->group("s.sku");
            $stockData = $db->fetchAssoc($select->__toString());

            $externalItemsQty = count($stockData);
            $actualUpdates = 0;
            Mage::helper($this->_defaultHelper)->log("Found {$externalItemsQty} products to potentially update.");

            foreach ($stockData as $sku => $stockRow) {
                if (!empty($productData[$sku])) {
                    if ($productData[$sku]["qty"] != $stockRow["external_qty"]) {
                        $productData[$sku]["external_qty"] = $stockRow["external_qty"];
                        $this->updateLocalStock($productData[$sku]);
                        $actualUpdates++;
                        usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                    }
                }
            }

            // Do reindexing after successfull product sync and/or successfull stock update
            if ($syncStatus || !empty($actualUpdates)) {
                Mage::helper($this->_defaultHelper)->log("Found {$actualUpdates} products to actually update.");
                $this->doReindex();
            }

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while syncing local stock!");
        }
    }

    public function updateProducts()
    {
        Mage::helper($this->_defaultHelper)->log("Updating products from table 'product_update'");

        $updates = [];
        $warehouseId = intval(Mage::getStoreConfig("integration/external_stock/warehouse_id"));
        $magentoId = intval(Mage::getStoreConfig("integration/general/magento_instance"));

        try {
            // Fetch all products tagged to be updated
            $sqlQuery1 = "
                SELECT
                    pu.product_id,
                    pu.sku,
                    SUM(s.qty) AS qty
                FROM product_update pu
                JOIN stock s ON s.sku = pu.sku AND s.warehouse_id = pu.warehouse_id AND (s.magento_id = 0 OR s.magento_id = pu.warehouse_id)
                WHERE pu.magento_id = :magentoId AND pu.warehouse_id = :warehouseId AND pu.tagged_at IS NOT NULL
                GROUP BY pu.sku
                ORDER BY pu.tagged_at
            ";
            $params1 = [
                "warehouseId" => $warehouseId,
                "magentoId" => $magentoId
            ];
            $externalStockRows = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchAll($sqlQuery1, $params1);

            $rowQty = count($externalStockRows);
            if (!$rowQty) {
                return;
            }

            // Ok, we got products in need of stock updates, now initialize queries and variables
            $sqlQuery2a = "
                SELECT
                    p.sku,
                    s.product_id,
                    min_qty,
                    s.qty
                FROM catalog_product_entity p
                JOIN cataloginventory_stock_item s ON s.product_id = p.entity_id
                WHERE
                    s.use_config_manage_stock = 0
                    AND s.manage_stock = 1
                    AND p.sku IN (%ph)
                GROUP BY p.sku
            ";
            $sqlQuery3a = "UPDATE product_update SET tagged_at = NULL WHERE warehouse_id = :warehouseId AND magento_id = :magentoId AND product_id IN (%ph)";
            $batch = 0;
            $batchSize = Awardit_Integration_Helper_Data::STOCK_UPDATE_BATCH_SIZE;
            $placeholders = [];
            $index = 0;

            // Do this until we processed all stock rows
            do {

                // Remember $stockRows index and its sku
                $placeholders[ $externalStockRows[$index]["sku"] ] = $index;
                $index++;

                // If we have reached batchSize OR if we have done all stock rows
                if ($index % $batchSize === 0 || $index >= $rowQty) {

                    // Prepare query to fetch product data from local Magento instance
                    $sqlQuery2b = str_replace ("%ph", ":ph" . implode(", :ph", $placeholders), $sqlQuery2a);
                    $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_read")->prepare($sqlQuery2b);
                    foreach ($placeholders as $sku => $idx) {
                        $stmt1->bindValue("ph{$idx}", $sku);
                    }
                    $stmt1->execute();
                    $result = $stmt1->fetchAll();
                    $updates = [];

                    // Parse result and update stock in local Magento instance
                    foreach ($result as $row) {

                        // Filter out famous "this should not happen" values
                        if (!array_key_exists($row["sku"], $placeholders) || !array_key_exists($placeholders[ $row["sku"] ], $externalStockRows)) {
                            continue;
                        }

                        // Update local stock
                        $status = $this->updateLocalStock([
                            "product_id" => $row["product_id"],
                            "external_qty" => $externalStockRows[ $placeholders[ $row["sku"] ] ]["qty"],
                            "min_qty" => $row["min_qty"],
                            "sku" => $row["sku"],
                            "qty" => $row["qty"],
                        ]);

                        if ($status) {

                            // If update was successfull, sleep for a while and remember product_id for table 'product_update'
                            usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                            $updates[] = $externalStockRows[ $placeholders[ $row["sku"] ] ]["product_id"];

                        }

                    }

                    // If we successfully updated local stock for any products, clear 'tagged_at' in table 'product_update' for those products.
                    if (!empty($updates)) {

                        $sqlQuery3b = str_replace ("%ph", ":ph" . implode(", :ph", array_keys($updates)), $sqlQuery3a);
                        $stmt2 = Mage::getSingleton("core/resource")->getConnection("stock_handler_write")->prepare($sqlQuery3b);
                        $stmt2->bindValue("warehouseId", $warehouseId);
                        $stmt2->bindValue("magentoId", $magentoId);
                        foreach ($updates as $idx => $productId) {
                            $stmt2->bindValue("ph{$idx}", intval($productId));
                        }
                        $stmt2->execute();
                        $affectedRows = $stmt2->rowCount();
                        if ($affectedRows > 0) {
                            usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                        }

                    }

                    // Increment and reset
                    $batch++;
                    $placeholders = [];

                }

                // Continue until we reached end of stock rows
            } while($index < $rowQty);

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating local stock for products tagged to be updated!");
        }

        // Update products set to not handle stock to always be in stock
        if (false): // Skip this stage for now
        try {

            $sqlUpdate4a = "
                UPDATE cataloginventory_stock_status css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = 1, css.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND css.stock_status = 0
            ";
            $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4a);
            $stmt1->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt1->execute();
            $updatedRows = $stmt1->rowCount();

            $sqlUpdate4b = "
                UPDATE cataloginventory_stock_status_idx css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = 1, css.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND css.stock_status = 0
            ";
            $stmt2 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4b);
            $stmt2->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt2->execute();
            $updatedRows += $stmt2->rowCount();

            $sqlUpdate4c = "
                UPDATE cataloginventory_stock_item csi
                SET csi.is_in_stock = 1, csi.qty = :defaultStockQty
                WHERE csi.manage_stock = 0 AND csi.is_in_stock = 0
            ";
            $stmt3 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlUpdate4c);
            $stmt3->bindValue("defaultStockQty", Awardit_Integration_Helper_Data::DEFAULT_STOCK_QTY);
            $stmt3->execute();
            $updatedRows += $stmt3->rowCount();

            if ($updatedRows > 0) {
                usleep(Awardit_Integration_Helper_Data::STOCK_UPDATE_SLEEP_TIME);
                Mage::helper($this->_defaultHelper)->log("Updated {$updatedRows} rows for products that were set to be out-of-stock even though they are set to not handle stock.");
            }

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating products to always be in stock when set to not handle stock!");
        }
        endif;

        // Do reindexing after successfull product sync and/or successfull stock update
        if (!empty($externalStockRows)) {
            $this->doReindex();
        }
    }

    public function updateLocalStock($stockRow)
    {
        try {
            $productId = $stockRow["product_id"];
            $isInStock = $stockRow["external_qty"] > $stockRow["min_qty"] ? Mage_CatalogInventory_Model_Stock::STOCK_IN_STOCK : Mage_CatalogInventory_Model_Stock::STOCK_OUT_OF_STOCK;

            $sqlQuery1 = "
                UPDATE cataloginventory_stock_status css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = :isInStock, css.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt1 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery1);
            $stmt1->bindValue("qty", $stockRow["external_qty"]);
            $stmt1->bindValue("isInStock", $isInStock);
            $stmt1->bindValue("productId", $productId);
            $stmt1->execute();
            $updatedRows = $stmt1->rowCount();

            $sqlQuery2 = "
                UPDATE cataloginventory_stock_status_idx css
                JOIN cataloginventory_stock_item csi ON csi.product_id = css.product_id
                SET css.stock_status = :isInStock, css.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt2 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery2);
            $stmt2->bindValue("qty", $stockRow["external_qty"]);
            $stmt2->bindValue("isInStock", $isInStock);
            $stmt2->bindValue("productId", $productId);
            $stmt2->execute();
            $updatedRows += $stmt2->rowCount();

            $sqlQuery3 = "
                UPDATE cataloginventory_stock_item csi
                SET csi.is_in_stock = :isInStock, csi.qty = :qty
                WHERE csi.product_id = :productId
            ";
            $stmt3 = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery3);
            $stmt3->bindValue("qty", $stockRow["external_qty"]);
            $stmt3->bindValue("isInStock", $isInStock);
            $stmt3->bindValue("productId", $productId);
            $stmt3->execute();
            $updatedRows += $stmt3->rowCount();

            if (!empty($updatedRows)) {
                Mage::helper($this->_defaultHelper)->log("Product [{$stockRow["sku"]}] was updated (updated rows: {$updatedRows}), changed local stock from {$stockRow["qty"]} to {$stockRow["external_qty"]}");
            }
            return true;

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception in syncIsInStock()!");
            return false;
        }
    }

    public function doReindex()
    {
        $indexCodes = [
            "cataloginventory_stock" => true,
            "catalog_product_price" => false,
            "catalog_product_flat" => false
        ];

        try {
            foreach ($indexCodes as $indexCode => $indexStatus) {
                if ($indexStatus) {
                    Mage::helper($this->_defaultHelper)->log("Running reindexing process for {$indexCode}");
                    $process = Mage::getModel("index/indexer")->getProcessByCode($indexCode);
                    $process->reindexAll();
                }
            }
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception in doReindex() where indexCode = '{$indexCode}'!");
            return false;
        }
    }

}
