<?php

class Awardit_Integration_Model_TimedPrices extends Mage_Core_Model_Abstract {

    /**
     * @var int
     */
    public static $DEFAULT_TIME_LIMIT = -1; // In months, range from -1 to -12

    /**
     * @var bool
     */
    protected $_doInside = true;

    /**
     * @var bool
     */
    protected $_doOutside = true;

    /**
     * @var string
     */
    protected $_dateAdjustment = 'now';

    /**
     * @param bool $doInside
     * @return void
     */
    public function setDoInside($doInside)
    {
        $this->_doInside = $doInside ? true : false;
    }

    /**
     * @param bool $doOutside
     * @return void
     */
    public function setDoOutside($doOutside)
    {
        $this->_doOutside = $doOutside ? true : false;
    }

    public function setDateAdjustment($adjustment)
    {
        $this->_dateAdjustment = $adjustment;
    }

    /**
     * @return void
     */
    public function processTimedPrices()
    {
        $vismaShopIds = Mage::helper("integration")->getInstanceUniqueVismaShopIds();
        if (!empty($vismaShopIds)) {
            $vismaShopIdString = implode(',', $vismaShopIds);
            $sqlQuery = "
                SELECT
                    t1.ProdNo AS sku,
                    t1.PurcPr AS purchase_price,
                    t1.PurcCur AS purchase_currency_id,
                    t1.VatNo AS tax_id,
                    t1.SugPr AS suggested_price,
                    t1.Cur AS currency_id,
                    t1.R1 AS shop_id,
                    t1.SalePr AS price,
                    t1.CustPrG2 AS customer_group_id,
                    t1.FrDt AS from_date,
                    t1.ToDt AS to_date,
                    CASE WHEN  t1.R1 > 0 AND t1.Cur > 0 THEN CONCAT(t1.CustPrG2, '-', t1.R1, '-', t1.Cur) ELSE NULL END AS price_index
                FROM PrDcMat t1
                WHERE
                    t1.ProdNo IN (SELECT t2.ProdNo FROM PrDcMat t2 WHERE t2.CustNo = 0 AND (t2.FrDt > 0 OR t2.ToDt > 0) AND t2.R1 IN ({$vismaShopIdString}) GROUP BY t2.ProdNo)
                    AND t1.CustNo = 0
                    AND t1.PurcCur = 0
                    AND t1.Cur > 0
                    AND t1.R1 > 0
                ORDER BY t1.ProdNo, t1.Cur, t1.R1, t1.CustPrG2, t1.FrDt, t1.ToDt
            ";
            try {
                $result = Mage::helper("integration")->getVismaDB()->fetchAll($sqlQuery);
            } catch (Exception $exception) {
                Mage::helper("integration")->logException($exception, "Exception while fetching timed prices!");
                return;
            }
            $priceData = [];
            foreach ($result as $row) {
                if (!array_key_exists($row['sku'], $priceData)) {
                    $priceData[$row['sku']] = [];
                }
                $priceData[$row['sku']][] = $row;
            }

            foreach (array_keys($priceData) as $sku) {
                $this->parseTimedPrices($sku, $priceData[$sku]);
                $this->setTimedPrices($sku, $priceData[$sku]);
            }
        }
    }

    /**
     * @param string $sku
     * @param array $priceData
     * @return void
     */
    public function setTimedPrices($sku, $priceData)
    {
        $productId = null;
        $localPrices = [];
        $groupPrices = [];

        foreach ($priceData as $priceRow) {
            if (!empty($priceRow['have_timed_price'])) {

                // Only load product if we actually have at least one timed price
                if (empty($productId)) {
                    $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
                    if (!empty($productId)) {
                        $product = Mage::getModel('catalog/product')->load($productId);
                        $productWebsiteIds = $product->getWebsiteIds();
                        sort($productWebsiteIds, SORT_NUMERIC);
                    } else {
                        // Unable to load product
                        break;
                    }
                }

                $priceRowWebsiteIds = Mage::helper("integration")->getMagentoWebsiteIds($priceRow['shop_id'], $priceRow['currency_id']);

                foreach ($priceRowWebsiteIds as $priceRowWebsiteId) {

                    // Only process products that previously have price for specified website.
                    if (in_array($priceRowWebsiteId, $productWebsiteIds)) {

                        // Prepare prices here
                        if (empty($priceRow['customer_group_id'])) {
                            $localPrices[$priceRowWebsiteId] = $priceRow;
                        } else {
                            $index = "{$priceRow['customer_group_id']}_{$priceRowWebsiteId}";
                            $groupPrices[$index] = $priceRow;
                        }
                    }
                }
            }
        }

        if (!empty($groupPrices)) {
            $this->_setGroupPrices($product, $groupPrices);
        }

        if (!empty($localPrices)) {
            $this->_setLocalPrices($product, $localPrices);
        }

    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @param array $groupPrices
     * @return void
     */
    protected function _setGroupPrices($product, $groupPrices)
    {
        $updateProduct = false;

        // Get current group prices for product
        $productGroupPrices = $product->getData('group_price');
        if (empty($productGroupPrices)) {
            $product->getGroupPrice();
            $productGroupPrices = $product->getData('group_price');
        }
        if (!is_array($productGroupPrices)) {
            $productGroupPrices = [];
        }

        // Only continue if product have group prices
        if (is_array($productGroupPrices) && !empty($productGroupPrices)) {
            foreach (array_keys($groupPrices) as $index) {
                $ids = explode('_', $index);
                $vismaGroupId = $ids[0];
                $websiteId = $ids[1];
                $magentoGroups = Mage::helper("integration")->getMagentoGroupIds($vismaGroupId);

                foreach (array_keys($productGroupPrices) as $key) {
                    // Only update group prices that previously exists
                    if ($productGroupPrices[$key]['website_id'] == $websiteId && in_array($productGroupPrices[$key]['cust_group'], $magentoGroups)) {
                        // Only update price if it is > 0 OR if zero price is allowed
                        if (!empty($groupPrices[$index]['price']) || in_array($product->getSku(), Awardit_Integration_Model_Product::$allowZeroPrice)) {
                            // And only update price if new price actually differs from current
                            if ($productGroupPrices[$key]['price'] != $groupPrices[$index]['price']) {
                                $productGroupPrices[$key]['price'] = $groupPrices[$index]['price'];
                                $updateProduct = true;
                            }
                        }
                    }
                }
            }
        }

        // Only update product if we actually changed any price
        if ($updateProduct) {
            $product->setData('group_price', $productGroupPrices);
            $product->save();
        }
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @param array $localPrices
     * @return void
     */
    protected function _setLocalPrices($product, $localPrices)
    {
        foreach (array_keys($localPrices) as $websiteId) {
            $storeIds = Mage::helper("integration")->getMagentoStoreIds($websiteId);
            foreach ($storeIds as $storeId) {
                $localProduct = Mage::getModel('catalog/product')->setStoreId($storeId)->load($product->getId());
                // Only update price if it is > 0 OR if zero price is allowed
                if (!empty($localPrices[$websiteId]['price']) || in_array($product->getSku(), Awardit_Integration_Model_Product::$allowZeroPrice)) {
                    // And only update price if new price actually differs from current
                    if ($localProduct->getPrice() != $localPrices[$websiteId]['price']) {
                        $localProduct->setPrice($localPrices[$websiteId]['price'])->getResource()->saveAttribute($localProduct, 'price');
                    }
                }
            }
        }
    }

    /**
     * @param string $sku
     * @param array $priceData
     * @return void
     *
     * Parses $priceData and overrides local prices if timed price exist and is within range, also removes all rows with timed prices.
     */
    public function parseTimedPrices($sku, &$priceData)
    {
        $priceIndexes = [];
        $timedPrices = [];

        // First, build indexes of pointers to rows cointaining normal prices and timed prices
        foreach (array_keys($priceData) as $rowIndex) {
            if (!empty($priceData[$rowIndex]['shop_id']) && !empty($priceData[$rowIndex]['currency_id']) && empty($priceData[$rowIndex]['purchase_price'])) {
                // Only process price rows containing local prices
                if (empty($priceData[$rowIndex]['from_date']) && empty($priceData[$rowIndex]['to_date'])) {
                    // Process price row NOT containing timed data
                    if (!array_key_exists($priceData[$rowIndex]['price_index'], $priceIndexes)) {
                        // Save pointer to this price row
                        $priceIndexes[$priceData[$rowIndex]['price_index']] = $rowIndex;
                    } elseif ($priceData[$priceIndexes[$priceData[$rowIndex]["price_index"]]]["price"] == $priceData[$rowIndex]["price"]) {
                        // Oops, more than one row of pricedata for same customer_group/shop_id/currency, but they have same price, continue
                        Mage::helper("integration")->log("Product [{$sku}] have multiple rows with same price for same customer_group/shop_id/currency!", LOG_ERR);
                        $priceIndexes[$priceData[$rowIndex]["price_index"]] = $rowIndex;
                    } else {
                        // Oops, more than one row of pricedata for same customer_group/shop_id/currency AND they have different price, stop it!
                        $subject = "Product [{$sku}] have multiple rows with different price for same customer_group/shop_id/currency!";
                        Mage::helper("integration")->log($subject, LOG_ERR);
                        Mage::helper("integration")->sendAdminEmail(
                            $subject,
                            "Look for rows where:\n"
                                . "customer_group = {$priceData[$rowIndex]["customer_group_id"]}\n"
                                . "shop_id = {$priceData[$rowIndex]["shop_id"]}\n"
                                . "currency = {$priceData[$rowIndex]["currency_id"]}\n\n"
                                . print_r($priceData, true),
                            ["it","products"]
                        );
                    }
                } else {
                    // Process price row containing timed data
                    if (!array_key_exists($priceData[$rowIndex]['price_index'], $timedPrices)) {
                        $timedPrices[$priceData[$rowIndex]['price_index']] = [];
                    }
                    // Save pricedata for this row
                    $timedPrices[$priceData[$rowIndex]['price_index']][] = $priceData[$rowIndex];
                    // Remove this row from pricedata
                    unset($priceData[$rowIndex]);
                }
            }
        }

        // Now parse rows containing timed prices and evaluate if timed price is active or not
        foreach (array_keys($timedPrices) as $priceIndex) {
            // Iterate over timed prices for current priceIndex
            foreach($timedPrices[$priceIndex] as $priceRow) {
                $status = $this->checkTimedDates($priceRow['from_date'], $priceRow['to_date']);
                // Check if we are inside range (true) or outside range (false)
                if ($status === true && $this->_doInside) {
                    // Only continue if price > 0 OR we allow zero price for this product
                    if (!empty($priceRow['price']) || Mage::helper("integration")->isZeroPriceProduct($sku)) {
                        if (array_key_exists($priceIndex, $priceIndexes)) {
                            // Override local price if we have it
                            Mage::helper("integration")->log("Overriding price ({$priceData[$priceIndexes[$priceIndex]]['price']}) with timed price ({$priceRow['price']}) using fromDate = {$priceRow['from_date']} and toDate = {$priceRow['to_date']}");
                            $priceData[$priceIndexes[$priceIndex]]['price'] = $priceRow['price'];
                            $priceData[$priceIndexes[$priceIndex]]['have_timed_price'] = true;
                        } elseif (!empty($priceData[$rowIndex]['from_date']) && empty($priceData[$rowIndex]['to_date'])) {
                            // It's ok to have timed price without local price as long as we have from_date and NO to_date.
                            // Need to add timed price when we are missing local price
                            Mage::helper("integration")->log("Adding price ({$priceRow['price']}) using fromDate = {$priceRow['from_date']} and toDate = {$priceRow['to_date']}");
                            $priceRow['have_timed_price'] = true;
                            $priceRow['from_date'] = 0;
                            $priceRow['to_date'] = 0;
                            $priceData[] = $priceRow;
                        } else {
                            // Ooops, we are missing local price, send email about this.
                            Mage::helper("integration")->log("Product [{$sku}] have timed price but is missing local price!", LOG_ERR);
                            Mage::helper("integration")->sendAdminEmail(
                                "Product [{$sku}] have timed price but is missing local price!",
                                "Look for rows where:\n"
                                    . "customer_group = {$priceRow['customer_group_id']}\n"
                                    . "shop_id = {$priceRow['shop_id']}\n"
                                    . "currency = {$priceRow['currency_id']}\n\n"
                                    . print_r($priceData, true),
                                ['it','products']
                            );
                        }
                    } else {
                        // DO NOT ALLOW PRICE TO BE ZERO! Skip overriding timed price and send email about this.
                        Mage::helper("integration")->log("Product [{$sku}] have empty timed price!", LOG_ERR);
                        Mage::helper("integration")->sendAdminEmail(
                            "Product [{$sku}] have empty timed price!",
                            "Look for rows where:\n"
                                . "customer_group = {$priceRow['customer_group_id']}\n"
                                . "shop_id = {$priceRow['shop_id']}\n"
                                . "currency = {$priceRow['currency_id']}\n\n"
                                . print_r($priceData, true),
                            ['it','products']
                        );
                    }
                } elseif ($status === false && $this->_doOutside) {
                    if (array_key_exists($priceIndex, $priceIndexes)) {
                        if (!empty($priceData[$priceIndexes[$priceIndex]]['price']) || Mage::helper("integration")->isZeroPriceProduct($sku)) {
                            // Since this is outside of range, revert back to normal price
                            Mage::helper("integration")->log("Reverting back to normal price ({$priceData[$priceIndexes[$priceIndex]]['price']}) using fromDate = {$priceRow['from_date']} and toDate = {$priceRow['to_date']}");
                            $priceData[$priceIndexes[$priceIndex]]['have_timed_price'] = true;
                        } else {
                            // DO NOT ALLOW PRICE TO BE ZERO! Skip overriding timed price and send email about this.
                            Mage::helper("integration")->log("Product [{$sku}] have empty normal price!", LOG_ERR);
                            Mage::helper("integration")->sendAdminEmail(
                                "Product [{$sku}] have empty normal price!",
                                "Look for rows where:\n"
                                    . "customer_group = {$priceRow['customer_group_id']}\n"
                                    . "shop_id = {$priceRow['shop_id']}\n"
                                    . "currency = {$priceRow['currency_id']}\n\n"
                                    . print_r($priceData, true),
                                ['it','products']
                            );
                        }
                    } else {
                        // Ooops, we are missing local price, send email about this.
                        Mage::helper("integration")->log("Product [{$sku}] have timed price but is missing local price!", LOG_ERR);
                        Mage::helper("integration")->sendAdminEmail(
                            "Product [{$sku}] have timed price but is missing local price!",
                            "Look for rows where:\n"
                                . "customer_group = {$priceRow['customer_group_id']}\n"
                                . "shop_id = {$priceRow['shop_id']}\n"
                                . "currency = {$priceRow['currency_id']}\n\n"
                                . print_r($priceData, true),
                            ['it','products']
                        );
                    }
                }
            }
        }
    }

    /**
     * @param string $fromDate
     * @param string $toDate
     * @return bool|null
     *
     * returns null if dates have syntax error, false if dates is outside range, true if dates is inside range
     */
    public function checkTimedDates($fromDate, $toDate)
    {
        $status = null;

        // Must convert dates from int because Visma does not use date-fields, stupid!!!
        $fromDateStr = "{$fromDate}";
        if (!empty($fromDate)) {
            if (strlen($fromDateStr) == 8) {
                $fromDate = substr($fromDateStr, 0, 4) . '-' . substr($fromDateStr, 4, 2) . '-' . substr($fromDateStr, 6, 2);
            }
        } else {
            $fromDate = date('Y-m-d', strtotime('yesterday'));
        }

        $toDateStr = "{$toDate}";
        if (!empty($toDate)) {
            if (strlen($toDateStr) == 8) {
                $toDate = substr($toDateStr, 0, 4) . '-' . substr($toDateStr, 4, 2) . '-' . substr($toDateStr, 6, 2);
            }
        } else {
            $toDate = date('Y-m-d', strtotime('tomorrow'));
        }

        if ($this->validateDate($fromDate) && $this->validateDate($toDate)) {
            $datetimeFrom = date_create($fromDate);
            $datetimeTo = date_create($toDate);
            if ($datetimeFrom <= $datetimeTo) {
                $datetimeNow = date_create(date('Y-m-d', strtotime($this->_dateAdjustment)));
                if ($datetimeNow >= $datetimeFrom && $datetimeNow <= $datetimeTo) {
                    $status = true;
                } else {
                    $status = false;
                }
            }
        }

        return $status;
    }

    /**
     * @param string $date
     * @return bool
     *
     * returns true if date is in correct format and valid, false otherwise.
     */
    public function validateDate($date)
    {
        $d = DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }

    /**
     * @return bool
     */
    public function doReindex()
    {
        $indexCode = '';
        $indexCodes = [
            'catalog_category_product' => true,
            'cataloginventory_stock' => true,
            'catalog_product_price' => true,
            'catalog_product_flat' => true,
            'catalog_product_attribute' => false,
            'catalogsearch_fulltext' => false,
            'catalog_category_flat' => false,
            'catalog_url' => false,
            'tag_summary' => false
        ];

        try {
            foreach ($indexCodes as $indexCode => $indexStatus) {
                if ($indexStatus) {
                    Mage::helper("integration")->log("Running reindexing process for {$indexCode}");
                    $process = Mage::getModel('index/indexer')->getProcessByCode($indexCode);
                    $process->reindexAll();
                }
            }
        } catch (Exception $exception) {
            Mage::helper("integration")->logException($exception, "Exception in doReindex() where indexCode = '{$indexCode}'!");
            Mage::helper("integration")->sendAdminEmail(
                "Exception in doReindex() where indexCode = '{$indexCode}'",
                Mage::registry('index-process-sql-query') . "\n\n" . $exception->getMessage() . "\n" . $exception->getTraceAsString(),
                ["it"]
            );
            return false;
        }

        return true;
    }

}
