<?php

class Awardit_Integration_Model_Cli_PointSales extends Awardit_Integration_Model_Cli {

    protected $_defaultHelper = "integration/PointsCore";

    const POINT_SALES_ORDER_PROCESSING_STATUS_NEW = 0;
    const POINT_SALES_ORDER_PROCESSING_STATUS_PREPARED = 1;
    const POINT_SALES_ORDER_PROCESSING_STATUS_TRANSFERRED = 2;
    const POINT_SALES_ORDER_PROCESSING_STATUS_SKIP = 8;
    const POINT_SALES_ORDER_PROCESSING_STATUS_ERROR = 9;

    const CSV_INCLUDE_HEADER = true;
    const CSV_DATA_DELIMITER = ";";
    const CSV_ROW_DELIMITER = "\n";
    const CSV_TEXT_ENCASEMENT = "";
    const CSV_DATE_FORMAT = "Ymd";

    const CSV_LOCAL_PATH = "pointsales/";
    const CSV_LOCAL_PATH_TEST = "test/";
    const CSV_REMOTE_PATH = "sas/";
    const CSV_REMOTE_PATH_TEST = "test/";

    // -m=PointSales -f=ScanForNewOrders
    public function CLI_ScanForNewOrders($param)
    {
        $this->scanForNewOrders();
    }

    // -m=PointSales -f=ProcessSingleOrder -p="orderId:<orderId>|incrementId:<incrementId>"
    public function CLI_ProcessSingleOrder($param)
    {
        $wantedParams = [
            "orderId" => "int",
            "incrementId" => "string"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $pointSalesOrders = [];
        if (!empty($extractedParams["orderId"])) {
            $pointSalesOrders[0] = $this->getPointSalesOrderByOrderId($extractedParams["orderId"]);
        } elseif (!empty($extractedParams["incrementId"])) {
            $pointSalesOrders[0] = $this->getPointSalesOrderByIncrementId($extractedParams["incrementId"]);
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
            return;
        }

        if (empty($pointSalesOrders)) {
            Mage::helper($this->_defaultHelper)->log("Unable to load point sales order!", LOG_ERR);
            return;
        }

        Mage::helper($this->_defaultHelper)->log("About to process point sales order {$pointSalesOrders[0]["increment_id"]}");
        $this->preparePointSalesOrders($pointSalesOrders);
        $this->transferPointSalesOrders($pointSalesOrders);
        $this->updatePointSalesOrders($pointSalesOrders, true);

        if ($pointSalesOrders[0]["processing_status"] == self::POINT_SALES_ORDER_PROCESSING_STATUS_TRANSFERRED) {
            Mage::helper($this->_defaultHelper)->log("Point sales order {$pointSalesOrders[0]["increment_id"]} was successfully processed");
        }

    }

    // -m=PointSales -f=ProcessAllOrders
    public function CLI_ProcessAllOrders($param)
    {
        $this->scanForNewOrders();

        $pointSalesOrders = $this->getPointSalesOrdersByProcessingStatus(self::POINT_SALES_ORDER_PROCESSING_STATUS_NEW);
        $qty = count($pointSalesOrders);
        if ($qty == 0) {
            Mage::helper($this->_defaultHelper)->log("Found no point sales orders to process.");
            return;
        } elseif ($qty == 1) {
            Mage::helper($this->_defaultHelper)->log("Found {$qty} point sales order to process.");
        } else {
            Mage::helper($this->_defaultHelper)->log("Found {$qty} point sales orders to process.");
        }

        $this->preparePointSalesOrders($pointSalesOrders);
        $this->transferPointSalesOrders($pointSalesOrders);
        $this->updatePointSalesOrders($pointSalesOrders);

        foreach ($pointSalesOrders as $pointSalesOrder) {
            if ($pointSalesOrder["processing_status"] == self::POINT_SALES_ORDER_PROCESSING_STATUS_TRANSFERRED) {
                Mage::helper($this->_defaultHelper)->log("Point sales order {$pointSalesOrder["increment_id"]} was successfully processed");
            }
        }

    }

    public function scanForNewOrders()
    {
        $configData = Mage::helper($this->_defaultHelper)->getConfigValueForStores("integration/orders/pointsales_processing", Mage_Eav_Model_Entity_Attribute_Source_Boolean::VALUE_YES);
        if (empty($configData)) {
            Mage::helper($this->_defaultHelper)->log("Found no stores configured to process point sales.");
            return;
        }

        $storeIds = [];
        foreach ($configData as $storeData) {
            $storeIds[] = intval($storeData["store_id"]);
        }
        if (empty($storeIds)) {
            Mage::helper($this->_defaultHelper)->log("Found no stores configured to process point sales.");
            return;
        }

        $idStr = implode(",", $storeIds);
        $statusNew = self::POINT_SALES_ORDER_PROCESSING_STATUS_NEW;
        $statusSkip = self::POINT_SALES_ORDER_PROCESSING_STATUS_SKIP;
        $sqlQuery = "
            INSERT IGNORE INTO awardit_pointsales (`order_id`, `increment_id`, `created_at` , `pointsales_value`, `processing_status`)
            SELECT
                sfo.entity_id,
                sfo.increment_id,
                sfo.created_at,
                sfo.grand_total,
                IF(sfo.grand_total > 0, {$statusNew}, $statusSkip) AS processing_status
            FROM sales_flat_order sfo
            LEFT JOIN awardit_pointsales cp ON cp.order_id = sfo.entity_id
            WHERE sfo.store_id IN ({$idStr}) AND cp.order_id IS NULL
            ORDER BY sfo.entity_id ASC
        ";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->execute();
        $addedOrdersQty = $stmt->rowCount();

        if ($addedOrdersQty == 0) {
            Mage::helper($this->_defaultHelper)->log("Added no new orders to point sales queue");
        } elseif ($addedOrdersQty == 1) {
            Mage::helper($this->_defaultHelper)->log("Added {$addedOrdersQty} new order to point sales queue");
        } else {
            Mage::helper($this->_defaultHelper)->log("Added {$addedOrdersQty} new orders to point sales queue");
        }

        $ordersInQueueQty = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne("SELECT COUNT(order_id) AS order_qty FROM awardit_pointsales WHERE processing_status = ?", self::POINT_SALES_ORDER_PROCESSING_STATUS_NEW);
        if ($ordersInQueueQty == 0) {
            Mage::helper($this->_defaultHelper)->log("Point sales processing queue is empty");
        } elseif ($ordersInQueueQty == 1) {
            Mage::helper($this->_defaultHelper)->log("Point sales processing queue contains {$ordersInQueueQty} order");
        } else {
            Mage::helper($this->_defaultHelper)->log("Point sales processing queue contains {$ordersInQueueQty} orders");
        }

    }

    public function getPointSalesOrderByOrderId($orderId)
    {
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow("SELECT * FROM awardit_pointsales WHERE order_id = :orderId", [ "orderId" => $orderId ]);
    }

    public function getPointSalesOrderByIncrementId($incrementId)
    {
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow("SELECT * FROM awardit_pointsales WHERE increment_id = :incrementId", [ "incrementId" => $incrementId ]);
    }

    public function getPointSalesOrdersByProcessingStatus($processingStatus)
    {
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll("SELECT * FROM awardit_pointsales WHERE processing_status = :processingStatus", [ "processingStatus" => $processingStatus ]);
    }

    public function getHeaderRow()
    {
        $header = "";
        if (self::CSV_INCLUDE_HEADER) {
            $columns = [
                "company",
                "inf",
                "ref",
                "date",
                "value",
                "currency"
            ];
            $header = implode(self::CSV_DATA_DELIMITER, $columns) . self::CSV_ROW_DELIMITER;
        }
        return $header;

    }

    public function getDataRow($pointSalesOrder)
    {
        $row = [
            self::CSV_TEXT_ENCASEMENT . $pointSalesOrder["company_db"] . self::CSV_TEXT_ENCASEMENT,
            self::CSV_TEXT_ENCASEMENT . $pointSalesOrder["increment_id"] . self::CSV_TEXT_ENCASEMENT,
            self::CSV_TEXT_ENCASEMENT . $pointSalesOrder["payment_ref"] . self::CSV_TEXT_ENCASEMENT,
            date(self::CSV_DATE_FORMAT, strtotime($pointSalesOrder["created_at"])),
            floatval($pointSalesOrder["pointsales_value"]),
            intval($pointSalesOrder["currency_id"])
        ];
        return implode(self::CSV_DATA_DELIMITER, $row). self::CSV_ROW_DELIMITER;

    }

    public function preparePointSalesOrders(&$pointSalesOrders)
    {
        foreach (array_keys($pointSalesOrders) as $rowIndex) {
            if ($pointSalesOrders[$rowIndex]["processing_status"] != self::POINT_SALES_ORDER_PROCESSING_STATUS_NEW) {
                continue;
            }

            $order = Mage::getModel("sales/order")->load($pointSalesOrders[$rowIndex]["order_id"]);
            if (empty($order) || !$order->getId()) {
                Mage::helper($this->_defaultHelper)->log("Unable to load order {$pointSalesOrders[$rowIndex]["increment_id"]}!", LOG_ERR);
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
                continue;
            }

            if (!Mage::getModel("integration/Cli_OrderExport")->invoiceOrder([ "increment_id" => $order->getIncrementId() ], $order)) {
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
                continue;
            }

            $payment = $order->getPayment();
            $additionalInformation = $payment->getAdditionalInformation();
            if (empty($additionalInformation["intent_id"])) {
                Mage::helper($this->_defaultHelper)->log("Got empty payment ref for point sales order {$pointSalesOrders[$rowIndex]["increment_id"]}!", LOG_ERR);
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
                continue;
            }

            $pointSalesOrders[$rowIndex]["payment_ref"] = $additionalInformation["intent_id"];
            $pointSalesOrders[$rowIndex]["currency_id"] = Mage::helper($this->_defaultHelper)->getVismaCurrencyId($order->getStoreCurrencyCode());
            $pointSalesOrders[$rowIndex]["company_db"] = Mage::helper($this->_defaultHelper)->getVismaOrderDB($order->getStoreId());
            $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_PREPARED;
        }
    }

    public function transferPointSalesOrders(&$pointSalesOrders)
    {
        $localPathAndFilename = Awardit_Integration_Helper_Data::$basePath;

        if ($this->getTestMode()) {
            $localPathAndFilename .= self::CSV_LOCAL_PATH_TEST;
            $remotePath = self::CSV_REMOTE_PATH_TEST;
        } else {
            $localPathAndFilename .= self::CSV_LOCAL_PATH;
            $remotePath = self::CSV_REMOTE_PATH;
        }

        $localPathAndFilename .= "pointsales_" . date("Y-m-d_Hi") . ".csv";
        $remoteFilename = basename($localPathAndFilename);

        $dir = dirname($localPathAndFilename);
        if (!is_dir($dir) && !mkdir($dir, 0777, true)) {
            Mage::helper($this->_defaultHelper)->log("Unable to create directory {$dir}", LOG_ERR);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Unable to create directory {$dir}",
                "Check directory permissions!",
                ["it"]
            );
            foreach (array_keys($pointSalesOrders) as $rowIndex) {
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
            }
            return;
        }

        $dataRows = "";
        foreach (array_keys($pointSalesOrders) as $rowIndex) {
            if ($pointSalesOrders[$rowIndex]["processing_status"] != self::POINT_SALES_ORDER_PROCESSING_STATUS_PREPARED) {
                continue;
            }
            $dataRows .= $this->getDataRow($pointSalesOrders[$rowIndex]);
        }

        if (empty($dataRows)) {
            return;
        }

        if (file_put_contents($localPathAndFilename, $this->getHeaderRow() . $dataRows) === false) {
            Mage::helper($this->_defaultHelper)->log("Unable to create file {$localPathAndFilename}", LOG_ERR);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Unable to create point sales file",
                "Was trying to create file {$localPathAndFilename}\nCheck directory permissions!",
                ["it"]
            );
            foreach (array_keys($pointSalesOrders) as $rowIndex) {
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
            }
            return;
        }

        $connection = Mage::helper($this->_defaultHelper)->openVismaFTP();
        if (!ftp_put($connection, $remotePath . $remoteFilename, $localPathAndFilename, FTP_ASCII)) {
            Mage::helper($this->_defaultHelper)->log("Unable to upload file {$localPathAndFilename}", LOG_ERR);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Unable to upload point sales order to Visma",
                "Was trying to upload {$localPathAndFilename} to {$remotePath}{$remoteFilename}\nCheck directory permissions!",
                ["it"]
            );
            foreach (array_keys($pointSalesOrders) as $rowIndex) {
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_ERROR;
            }
            unlink($localPathAndFilename);
            Mage::helper($this->_defaultHelper)->closeVismaFTP();
            return;
        }

        Mage::helper($this->_defaultHelper)->closeVismaFTP();

        foreach (array_keys($pointSalesOrders) as $rowIndex) {
            if ($pointSalesOrders[$rowIndex]["processing_status"] == self::POINT_SALES_ORDER_PROCESSING_STATUS_PREPARED) {
                $pointSalesOrders[$rowIndex]["processing_status"] = self::POINT_SALES_ORDER_PROCESSING_STATUS_TRANSFERRED;
            }
        }

    }

    public function updatePointSalesOrders($pointSalesOrders, $forceUpdate = false)
    {
        $pointSalesOrder = $pointSalesOrders[0]; // For try/catch if we get error before $pointSalesOrder is set.

        try {
            $sqlQuery = "
                UPDATE awardit_pointsales SET
                    `currency_id` = :currencyId,
                    `payment_ref` = :paymentRef,
                    `company_db` = :companyDb,
                    `processing_status` = :processingStatus,
                    `processing_date` = :processingDate
                WHERE
                    `order_id` = :orderId
            ";
            $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);

            foreach ($pointSalesOrders as $pointSalesOrder) {
                if (!$forceUpdate && $pointSalesOrder["processing_status"] == self::POINT_SALES_ORDER_PROCESSING_STATUS_NEW) {
                    continue;
                }
                $params = [
                    "currencyId" => intval($pointSalesOrder["currency_id"]),
                    "paymentRef" => $pointSalesOrder["payment_ref"],
                    "companyDb" => $pointSalesOrder["company_db"],
                    "processingStatus" => intval($pointSalesOrder["processing_status"]),
                    "processingDate" => date("Y-m-d H:i:s"),
                    "orderId" => intval($pointSalesOrder["order_id"])
                ];
                foreach ($params as $key => $val) {
                    $stmt->bindValue($key, $val);
                }
                $stmt->execute();
            }

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating point sales order {$pointSalesOrder["increment_id"]}!");
        }
    }
}
