<?php

class Awardit_Integration_Helper_Data extends Mage_Core_Helper_Abstract
{

    const ORDER_AGE_MIN               = "-5 minutes";      // Order must be at least this old to be exported
    const ORDER_AGE_MAX               = "-5 days";         // Time limit for checking for new orders
    const SHIPPING_EMAIL_TIME_LIMIT   = "-3 days";         // If shipping was done > this limit ago, do not send shipping email
    const PRODUCT_SAVE_SLEEP_TIME     = 100000;            // microseconds
    const STOCK_UPDATE_SLEEP_TIME     = 100000;            // microseconds
    const STOCK_UPDATE_BATCH_SIZE     = 100;
    const DEFAULT_STOCK_QTY           = 200000;
    const DEFAULT_LOG_FILENAME        = "integration.log";
    const DEFAULT_CURRENCY_ID         = 46;                // Sweden
    const DEFAULT_CURRENCY_CODE       = "SEK";             // Sweden
    const DEFAULT_COUNTRY_ID          = 46;                // Sweden
    const DEFAULT_VISMA_CUSTOMER_ID   = "100100";
    const DEFAULT_VISMA_EXCHANGE_RATE = 100;
    const DEFAULT_VISMA_ORDER_DB      = "F0001";           // Sweden

    const INTEGRATION_TABLE_ORDERS    = "awardit_integration_orders";
    const INTEGRATION_TABLE_PRODUCTS  = "awardit_integration_products";

    public static $basePath = null;
    public static $adminEmailRecipients = [
        "it" => ["it@crossroads.se"], // everything
        "kam" => ["kam@crossroads.se"], // prices
        "products" => ["products@crossroads.se"], // missing product data
        "stock" => ["jonas@crossroads.se"] // stock
    ];

    protected $_vismaDB = null;
    protected $_vismaFTPConnection = null;
    protected $_debugMode = false;
    protected $_instanceUniqueVismaShopIds = [];
    protected $_attributeIds = [];
    protected $_shopIdConversionTable = [];
    protected $_groupConversionTable = [];
    protected $_currencyConversionTable = [];
    protected $_zeroPriceProducts = [];
    protected $_systemAttributeOptions = [];
    protected $_vismaCurrencyExchangeRates = [];
    protected $_exceptionToException = [];
    protected $_instancesWithSaveToDB = [ 19 ];

    public function __construct()
    {
    }

    public function getInstanceId()
    {
        return intval(Mage::getStoreConfig("integration/general/magento_instance"));
    }

    public function getSaveXMLDataToDB()
    {
        return in_array($this->getInstanceId(), $this->_instancesWithSaveToDB);
    }

    public function setDebugMode($mode) {
        $this->_debugMode = $mode ? true : false;
    }

    public function getDebugMode() {
        return $this->_debugMode;
    }

    public function reconnectVismaDB() {
        if (!empty($this->_vismaDB)) {
            try {
                $this->_vismaDB->closeConnection();
            } catch (Exception $ex) {
                // Silently ignore exceptions here
            }
        }
        $this->_vismaDB = null;

        return $this->getVismaDB();
    }

    public function getVismaDB()
    {
        if (empty($this->_vismaDB)) {
            $config = [
                "host" => Mage::getStoreConfig("integration/general/visma_host"),
                "username" => Mage::getStoreConfig("integration/general/visma_db_user"),
                "password" => Mage::getStoreConfig("integration/general/visma_db_password"),
                "dbname" => self::DEFAULT_VISMA_ORDER_DB,
                "port" => "1433",
                "pdoType" => "dblib",
                "charset" => "UTF-8"
            ];

            try {
                $this->_vismaDB = new Zend_Db_Adapter_Pdo_Mssql($config);
            } catch (Exception $exception) {
                $this->logException($exception, "Exception while connecting to mssql server!");
            }
        }
        return $this->_vismaDB;
    }

    public function openVismaFTP()
    {
        try {
            if (empty($this->_vismaFTPConnection)) {
                $this->_vismaFTPConnection = ftp_connect(Mage::getStoreConfig('integration/general/visma_ftp_server'));
                if (ftp_login($this->_vismaFTPConnection, Mage::getStoreConfig('integration/general/visma_ftp_user'), Mage::getStoreConfig('integration/general/visma_ftp_password'))) {
                    ftp_pasv($this->_vismaFTPConnection, true);
                } else {
                    $this->log("Unable to login to FTP-server!", LOG_ERR);
                    ftp_close($this->_vismaFTPConnection);
                    $this->_vismaFTPConnection = null;
                }
            }
        } catch (Exception $exception) {
            $this->logException($exception, 'Exception while connecting to FTP server!');
            $this->closeVismaFTP();
            $this->_vismaFTPConnection = null;
        }

        return $this->_vismaFTPConnection;
    }

    public function closeVismaFTP()
    {
        if (!empty($this->_vismaFTPConnection)) {
            ftp_close($this->_vismaFTPConnection);
        }
        $this->_vismaFTPConnection = null;
    }

    public function getMagentoTaxIdFromVismaTaxId($vismaTaxId, $currencyId, $countryId = 0)
    {
        $sqlQuery = "SELECT magento_tax_id FROM tax_translations WHERE magento_instance = :instanceId AND swedish_visma_tax_id = :vismaTaxId AND (currency_id = :currencyId OR currency_id = 0) AND country_id = :countryId ORDER BY currency_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "vismaTaxId" => intval($vismaTaxId),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return intval(Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params));
        // If we don't get any value from query, intval() makes it a 0 => no vat, exactly what we want as default fallback value.

    }

    public function getTaxPercentFromMagentoTaxId($magentoTaxId, $currencyId, $countryId)
    {
        $sqlQuery = "SELECT tax_percent FROM tax_translations WHERE magento_instance = :instanceId AND magento_tax_id = :magentoTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
            "magentoTaxId" => intval($magentoTaxId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getTaxPercentFromVismaTaxId($vismaTaxId, $currencyId, $countryId)
    {
        $sqlQuery = "SELECT tax_percent FROM tax_translations WHERE magento_instance = :instanceId AND swedish_visma_tax_id = :vismaTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
            "vismaTaxId" => intval($vismaTaxId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getVismaTaxIdFromMagentoTaxId($magentoTaxId, $currencyId, $countryId)
    {
        // Always use local Visma tax id, based on currency and country.
        $sqlQuery = "SELECT local_visma_tax_id FROM tax_translations WHERE magento_instance = :instanceId AND magento_tax_id = :magentoTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "magentoTaxId" => intval($magentoTaxId),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getMaxTaxForCurrencyAndCountry($currencyId, $countryId)
    {
        $sqlQuery = "SELECT * FROM tax_translations WHERE magento_instance = :instanceId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, tax_percent DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchRow($sqlQuery, $params);
    }

    public function createExceptionToException()
    {
        $data = Mage::getStoreConfig('integration/products/exception_to_exception');
        if (!empty($data)) {
            $data = json_decode($data, true);
            if (is_array($data)) {
                $this->_exceptionToException = $data;
            }
        }
    }

    public function isExceptionToException($sku)
    {
        if (empty($this->_exceptionToException)) {
            $this->createExceptionToException();
        }

        return array_key_exists($sku, $this->_exceptionToException);
    }

    public function getMagentoGroupIds($vismaGroupId)
    {
        $groupIds = [];

        if (empty($this->_groupConversionTable)) {
            $this->createGroupConversionTable();
        }

        foreach ($this->_groupConversionTable as $row) {
            if ($row["visma_group_id"] == $vismaGroupId) {
                $groupIds[] = $row["customer_group_id"];
            }
        }

        // Normal instances needs empty array if no group id is found
        return !empty($groupIds) ? $groupIds : [];
    }

    public function getVismamGroupIds($magentoGroupId)
    {
        $groupIds = [];

        if (empty($this->_groupConversionTable)) {
            $this->createGroupConversionTable();
        }

        foreach ($this->_groupConversionTable as $row) {
            if ($row["customer_group_id"] == $magentoGroupId) {
                $groupIds[] = $row["visma_group_id"];
            }
        }

        return !empty($groupIds) ? $groupIds : [0];
    }

    public function createGroupConversionTable()
    {
        $sqlQuery = "SELECT visma_group_id, customer_group_id FROM customer_group WHERE visma_group_id > 0";
        $this->_groupConversionTable = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
    }

    public function getVismaProductStock($prodNo, $warehouses = [])
    {
        // If configured to use external stock:
        if (Mage::getStoreConfig("integration/external_stock/enabled")) {
            // Try to get stock qty from external stock
            $sqlQuery1 = "SELECT sku, SUM(qty) AS qty FROM stock WHERE sku = :sku AND warehouse_id = :warehouseId AND (magento_id = 0 OR magento_id = :magentoId) GROUP BY sku";
            $params1 = [
                "sku" => $prodNo,
                "warehouseId" => Mage::getStoreConfig("integration/external_stock/warehouse_id"),
                "magentoId" => $this->getInstanceId()
            ];
            $externalStockData = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchRow($sqlQuery1, $params1);
            if (!empty($externalStockData["sku"])) {
                return $externalStockData["qty"];
            }
        }

        // If no external stock was found, get stock qty from Visma
        if (empty($warehouses) || !is_array($warehouses)) {
            switch (Mage::getStoreConfig('integration/external_stock/warehouse_id')) {
                case 2:
                    $stcbal = "11,13";
                    break;
                case 1:
                default:
                    $stcbal = "1,3";
                    break;
            }
        } else {
            $stcbal = implode(",", $warehouses);
        }

        // Realiserat saldo + Orealiserad lagerökn. - Reserverad - Plockat utan reservering.
        // Bal + StcInc - (ShpRsv+ShpRsvIn) - PicNotR
        // ProdNo is specified as varchar and MSSQL freaks out if we ask for 12345 instead of '12345'
        $sqlQuery2 = "SELECT SUM(StcInc + Bal - ShpRsv - PicNotR) AS stock_balance FROM StcBal WHERE ProdNo = '{$prodNo}' AND StcNo IN ({$stcbal})";
        return $this->getVismaDB()->fetchOne($sqlQuery2);
    }

    public function getShopIdConversionTable()
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        return $this->_shopIdConversionTable;
    }

    public function getMagentoWebsiteIdsForProduct($priceData)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $productWebsites = [];

        foreach ($priceData as $prices) {
            if (!empty($prices["shop_id"]) && !empty($prices["currency_id"])) {
                foreach ($this->_shopIdConversionTable as $row) {
                    if ($row["currency_id"] == $prices["currency_id"] && $row["shop_id"] == $prices["shop_id"]) {
                        if (!in_array($row["website_id"], $productWebsites)) {
                            $productWebsites[] = $row["website_id"];
                        }
                    }
                }
            }
        }

        if (!empty($productWebsites)) {
            sort($productWebsites, SORT_NUMERIC);
        }

        return $productWebsites;
    }

    public function getMagentoWebsiteIds($shopId, $currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $productWebsites = [];

        foreach ($this->_shopIdConversionTable as $row) {
            if ($row["currency_id"] == $currencyId && $row["shop_id"] == $shopId) {
                if (!in_array($row["website_id"], $productWebsites)) {
                    $productWebsites[] = $row["website_id"];
                }
            }
        }

        return $productWebsites;
    }

    public function getMagentoStoreIds($websiteId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $storeIds = [];

        foreach ($this->_shopIdConversionTable as $row) {
            if ($row["website_id"] == $websiteId) {
                if (!in_array($row["store_id"], $storeIds)) {
                    $storeIds[] = $row["store_id"];
                }
            }
        }

        return $storeIds;
    }

    public function getInstanceUniqueVismaShopIds()
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        return $this->_instanceUniqueVismaShopIds;
    }

    public function checkIfShopIdExistsInThisMagentoInstance($shopId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        if (empty($shopId)) {
            return false;
        } else {
            return in_array($shopId, $this->_instanceUniqueVismaShopIds);
        }
    }

    public function havePriceRowForCurrencyShopIdGroupId($priceData, $currencyId, $shopId, $groupId)
    {
        foreach ($priceData as $priceRow) {
            if ($priceRow["currency_id"] == $currencyId && $priceRow["shop_id"] == $shopId && $priceRow["customer_group_id"] == $groupId) {
                return true;
            }
        }

        return false;
    }

    public function getAttributeIdByAttributeSetAndCode($attributeSetId, $attributeCode)
    {
        $key = "{$attributeSetId}:{$attributeCode}";

        if (!array_key_exists($key, $this->_attributeIds)) {
            $sqlQuery = "
                SELECT
                    a.attribute_id
                FROM eav_attribute_set s
                JOIN eav_entity_attribute ea ON ea.attribute_set_id = s.attribute_set_id
                JOIN eav_attribute a ON a.attribute_id = ea.attribute_id
                WHERE s.attribute_set_id = :attributeSetId AND a.attribute_code = :attributeCode
            ";
            $params = [
                "attributeSetId" => $attributeSetId,
                "attributeCode" => $attributeCode
            ];

            $this->_attributeIds[$key] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);
        }

        return $this->_attributeIds[$key];
    }

    public function getConfigValueForStores($path, $filter = null)
    {
        $baseQuery = "
            SELECT
                tmp1.website_id,
                tmp1.store_id,
                COALESCE(MAX(tmp1.value3), MAX(tmp1.value2), MAX(tmp1.value1)) AS config_value
            FROM (
                SELECT
                    cs.website_id,
                    cs.store_id,
                    cs.code,
                    IF (ccd1.scope = 'default', ccd1.value, NULL) AS value1,
                    IF (ccd1.scope = 'websites', ccd1.value, NULL) AS value2,
                    IF (ccd1.scope = 'stores', ccd1.value, NULL) AS value3
                FROM core_store cs
                JOIN core_config_data ccd1 ON ccd1.path = :path AND ((ccd1.scope_id = 0 AND ccd1.scope = 'default') OR (ccd1.scope_id = cs.website_id AND ccd1.scope = 'websites') OR (ccd1.scope_id = cs.store_id AND ccd1.scope = 'stores'))
                WHERE ccd1.value IS NOT NULL
            ) tmp1
            GROUP BY tmp1.website_id, tmp1.store_id
        ";
        if (empty($filter)) {
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($baseQuery, [ "path" => $path ]);
        }

        $wrappingQuery = "
            SELECT
                tmp2.*
            FROM (
                {$baseQuery}
            ) tmp2
            WHERE tmp2.config_value = :filterValue
        ";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($wrappingQuery, [ "path" => $path, "filterValue" => $filter ]);

    }

    public function getMagentoStoreIdsByVismaShopAndCurrency($vismaShopId, $currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $storeIds = [];
        foreach ($this->_shopIdConversionTable as $row) {
            if ($row["shop_id"] == $vismaShopId && $row["currency_id"] == $currencyId) {
                if (!in_array($row["store_id"], $storeIds)) {
                    $storeIds[] = $row["store_id"];
                }
            }
        }

        return $storeIds;
    }

    public function getVismaShopIdsByCurrencyId($currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $shopIds = [];
        foreach ($this->_shopIdConversionTable as $row) {
            if ($row["currency_id"] == $currencyId) {
                if (!in_array($row["shop_id"], $shopIds)) {
                    $shopIds[] = $row["shop_id"];
                }
            }
        }

        return $shopIds;

    }

    public function createShopIdConversionTable()
    {
        try {

            $sqlQuery = "
                SELECT
                    tmp.website_id,
                    tmp.store_id,
                    0 AS currency_id,
                    COALESCE(MAX(tmp.currency3), MAX(currency2), MAX(currency1)) AS currency_code,
                    COALESCE(MAX(tmp.shop2), MAX(shop1)) AS shop_id
                FROM (
                    SELECT
                        cs.website_id,
                        cs.store_id,
                        cs.code,
                        IF (ccd1.scope = 'default', ccd1.value, NULL) AS currency1,
                        IF (ccd1.scope = 'websites', ccd1.value, NULL) AS currency2,
                        IF (ccd1.scope = 'stores', ccd1.value, NULL) AS currency3,
                        IF (ccd1.scope = 'default', ccd2.value, NULL) AS shop1,
                        IF (ccd1.scope = 'websites', ccd2.value, NULL) AS shop2
                    FROM core_store cs
                    JOIN core_config_data ccd1 ON ccd1.path = 'currency/options/default' AND ((ccd1.scope_id = 0 AND ccd1.scope = 'default') OR (ccd1.scope_id = cs.website_id AND ccd1.scope = 'websites') OR (ccd1.scope_id = cs.store_id AND ccd1.scope = 'stores'))
                    JOIN core_config_data ccd2 ON ccd2.path = 'integration/general/visma_shop_id' AND ((ccd2.scope_id = 0 AND ccd2.scope = 'default') OR (ccd2.scope_id = cs.website_id AND ccd2.scope = 'websites') OR (ccd2.scope_id = cs.store_id AND ccd2.scope = 'stores'))
                    WHERE ccd2.value IS NOT NULL
                ) tmp
                GROUP BY tmp.website_id, tmp.store_id
            ";
            $this->_shopIdConversionTable = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
            $this->_instanceUniqueVismaShopIds = [];

            // Loop through all websites and get currency id
            foreach (array_keys($this->_shopIdConversionTable) as $rowIndex) {
                $this->_shopIdConversionTable[$rowIndex]["currency_id"] = $this->getVismaCurrencyId($this->_shopIdConversionTable[$rowIndex]["currency_code"]);
                if (!in_array($this->_shopIdConversionTable[$rowIndex]["shop_id"], $this->_instanceUniqueVismaShopIds)) {
                    $this->_instanceUniqueVismaShopIds[] = $this->_shopIdConversionTable[$rowIndex]["shop_id"];
                }
            }

            if (empty($this->_instanceUniqueVismaShopIds)) {
                $this->log("No Visma Shop Ids in this Magento instance!", LOG_ERR);
            }
        } catch (Exception $exception) {
            $this->logException($exception, "Exception while creating _shopIdConversionTable");
        }
    }

    public function getVismaCurrencyId($currencyCode)
    {
        if (empty($this->_currencyConversionTable)) {
            $this->createCurrencyCodeConversionTable();
        }

        foreach ($this->_currencyConversionTable as $row) {
            if ($row["currency_code"] == $currencyCode) {
                return $row["currency_id"];
            }
        }

        return self::DEFAULT_CURRENCY_ID;
    }

    public function getMagentoCurrencyCode($currencyId)
    {
        if (empty($this->_currencyConversionTable)) {
            $this->createCurrencyCodeConversionTable();
        }

        foreach ($this->_currencyConversionTable as $row) {
            if ($row["currency_id"] == $currencyId) {
                return $row["currency_code"];
            }
        }

        return self::DEFAULT_CURRENCY_CODE;
    }

    public function getVismaCurrencyExchangeRate($currencyId)
    {
        if (empty($this->_vismaCurrencyExchangeRates)) {
            $sqlQuery = "SELECT CurNo, SalesRt FROM Cur";
            $result = $this->getVismaDB()->fetchAll($sqlQuery);
            if (!empty($result)) {
                $this->_vismaCurrencyExchangeRates = [];
                foreach ($result as $row) {
                    $this->_vismaCurrencyExchangeRates[$row['CurNo']] = $row['SalesRt'];
                }
            }
        }

        if (array_key_exists($currencyId, $this->_vismaCurrencyExchangeRates)) {
            return $this->_vismaCurrencyExchangeRates[$currencyId];
        }

        return self::DEFAULT_VISMA_EXCHANGE_RATE;
    }

    public function getVismaCountryId($countryCode)
    {
        $sqlQuery = "SELECT country_id FROM country_codes WHERE country_code = ?";
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $countryCode);
    }

    public function createCurrencyCodeConversionTable()
    {
        // Get list of unique currency codes in this instance
        $sqlQuery1 = "
            SELECT GROUP_CONCAT(tmp.currency_code SEPARATOR \"','\") AS currency_codes FROM
            (SELECT DISTINCT(`value`) AS currency_code, 1 AS fake_group FROM core_config_data WHERE path = 'currency/options/default') tmp
            GROUP BY tmp.fake_group
        ";
        $currencyCodes = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery1);

        // Get list of currency ids from currency codes
        $sqlQuery2 = "SELECT CurNo AS currency_id, ISO AS currency_code FROM Cur WHERE iso IN ('{$currencyCodes}')";
        $this->_currencyConversionTable = $this->getVismaDB()->fetchAll($sqlQuery2);
    }

    public function getMagentoWebsiteIdByStoreId($storeId)
    {
        $sqlQuery = "SELECT website_id FROM core_store WHERE store_id = ?";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $storeId);
    }

    public function getVismaOrderDB($storeId)
    {
        return Mage::getStoreConfig('integration/general/visma_company_db', $storeId) ?: Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB;
    }

    public function getVismaOrderDBOverride($storeId, $currencyId, $countryId)
    {
        $rawOverrides = Mage::getStoreConfig("integration/general/db_override", $storeId);
        if (!empty($rawOverrides)) {
            $overrides = unserialize($rawOverrides);
            if (!empty($overrides) && is_array($overrides)) {
                foreach ($overrides as $override) {
                    if ($override["currency_id"] == $currencyId && $override["country_id"] == $countryId) {
                        return $override;
                    }
                }
            }
        }

        return [];
    }

    public function translateAttributeSetId($remoteInstance, $remoteAttributeSetId)
    {

        // Try to find the same attribute set here at local instance, as remote instance
        $sqlQuery = "
            SELECT
                T2.attribute_id
            FROM
                attribute_set_translations T1
            JOIN
                attribute_set_translations T2 ON T2.attribute_name = T1.attribute_name AND T2.instance_id = :localInstanceId
            WHERE
                T1.instance_id = :remoteInstanceId
                AND T1.attribute_id = :remoteAttributeSetId
        ";
        $params = [
            "localInstanceId" => $this->getInstanceId(),
            "remoteInstanceId" => $remoteInstance,
            "remoteAttributeSetId" => $remoteAttributeSetId
        ];
        $localAttributeSetId = Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);

        return $localAttributeSetId ?: Mage::getModel("catalog/product")->getDefaultAttributeSetId();
    }

    public function getManufacturerId($text)
    {
        return $this->getOptionId('manufacturer', $this->convertString($text));
    }

    public function getOptionId($attributeCode, $value, $createOnMissing = true)
    {

        try {
            $sqlQuery = "
                SELECT
                    eao.option_id
                FROM eav_attribute ea
                JOIN eav_attribute_option eao ON eao.attribute_id = ea.attribute_id
                JOIN eav_attribute_option_value eaov ON eaov.option_id = eao.option_id
                WHERE
                    ea.attribute_code = :attributeCode
                    AND eaov.value = :attributeValue
                    AND ea.entity_type_id = :typeId
            ";
            $params = [
                "attributeCode" => $attributeCode,
                "attributeValue" => $value,
                "typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()
            ];
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);

            // If no option was found, create one
            if ($createOnMissing && empty($data)) {
                return $this->createOption($attributeCode, $value);
            } else {
                return $data;
            }
        } catch (Exception $exception) {
            $this->logException($exception, "Exception while getting option_id for attribute '{$attributeCode}'!");
        }

        return null;
    }

    public function getSystemAttributeOptionValue($attributeCode, $optionText)
    {
        $optionValue = null;
        if (empty($this->_systemAttributeOptions)) {
            $this->_systemAttributeOptions = [
                "status" => Mage_Catalog_Model_Product_Status::getOptionArray(),
                "visibility" => Mage_Catalog_Model_Product_Visibility::getOptionArray()
            ];
        }

        if (array_key_exists($attributeCode, $this->_systemAttributeOptions)) {
            $optionValue = array_search($optionText, $this->_systemAttributeOptions[$attributeCode]);
        }

        return $optionValue;
    }

    public function createManufacturer($text)
    {
        return createOption("manufacturer", $text);
    }

    public function createOption($attributeCode, $value)
    {
        try {
            $sqlQuery = "SELECT ea.attribute_id FROM eav_attribute ea WHERE ea.attribute_code = :attributeCode AND ea.entity_type_id = :typeId";
            $params = [
                "attributeCode" => $attributeCode,
                "typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()
            ];
            $attributeId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);

            if (!empty($attributeId)) {
                $attribute_option = [
                    "option_id" => null,
                    "attribute_id" => $attributeId,
                    "sort_order" => 0
                ];

                Mage::getSingleton("core/resource")->getConnection("core_write")->insert("eav_attribute_option", $attribute_option);
                $option_id = Mage::getSingleton("core/resource")->getConnection("core_write")->lastInsertId();

                if (!empty($option_id)) {
                    $attribute_option_value = [
                        "value_id" => null,
                        "option_id" => $option_id,
                        "store_id" => 0,
                        "value" => $value
                    ];
                    Mage::getSingleton("core/resource")->getConnection("core_write")->insert("eav_attribute_option_value", $attribute_option_value);

                    return $option_id;
                } else {
                    $this->log("Unable to create new option '{$value}' for attribute '{$attributeCode}'");
                }
            } else {
                $this->log("Unable to find attribute_id for attribute '{$attributeCode}'");
            }
        } catch (Exception $exception) {
            $this->logException($exception, "Unable to create new option '{$value}' for attribute '{$attributeCode}'");
        }

        return null;
    }

    public function getAttributesForAttributeSet($attributeSetId)
    {

        if (empty($attributeSetId)) {
            $attributeSetId = Mage::getModel("catalog/product")->getDefaultAttributeSetId();
        }
        $attributesToExport = [];
        $sqlQuery = "
            SELECT
                a.attribute_code,
                a.frontend_input,
                a.source_model
            FROM eav_entity_attribute ea
            JOIN eav_attribute a ON a.attribute_id = ea.attribute_id
            JOIN eav_attribute_group g ON g.attribute_group_id = ea.attribute_group_id
            WHERE
                ea.attribute_set_id = ?
                AND g.attribute_group_name NOT IN ('Prices','Images')
                AND a.backend_type IN ('varchar','int','datetime','text','decimal')
                AND a.frontend_input IN ('text','weight','select','date','boolean','price','textarea')
                AND a.attribute_code NOT IN ('status')
            ORDER BY a.attribute_code
        ";

        $result = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $attributeSetId);
        if (!empty($result)) {
            foreach ($result as $row) {
                $attributesToExport[$row["attribute_code"]] = [
                    "frontend_input" => $row["frontend_input"],
                    "source_model" => $row["source_model"]
                ];
            }

            // Ugly fix for 'status' when it doesn't exist in current attribute set
            if (!array_key_exists("status", $attributesToExport)) {
                $attributesToExport["status"] = [
                    "frontend_input" => "select",
                    "source_model" => "catalog/product_status"
                ];
            }
        }

        return $attributesToExport;
    }

    public function deleteLocalAttributeValues($productId, $attributeCode)
    {
        // Fetch attribute info for wanted attribute code
        $sqlQuery1 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => $attributeCode
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery1, $params);

        // If no attribute was found, return
        if (empty($eava)) {
            return null;
        }

        // Find table name for matching backend type
        $table = "";
        switch ($eava["backend_type"]) {
            case "int":
                $table = "catalog_product_entity_int";
                break;

            case "datetime":
                $table = "catalog_product_entity_datetime";
                break;

            case "decimal":
                $table = "catalog_product_entity_decimal";
                break;

            case "text":
                $table = "catalog_product_entity_text";
                break;

            case "varchar":
                $table = "catalog_product_entity_varchar";
                break;

            default:
                break;
        }

        // If no matching table was found, return
        if (empty($table)) {
            return null;
        }

        // Now try to delete all attribute values except from default store
        $sqlQuery2 = "DELETE cpex.* FROM {$table} cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery2);
        $stmt->bindValue("typeId", $params["typeId"]);
        $stmt->bindValue("attributeId", intval($eava["attribute_id"]));
        $stmt->bindValue("productId", intval($productId));
        $stmt->execute();
        $affectedRows = $stmt->rowCount();
        if ($affectedRows > 0) {
            $this->log("Removed {$affectedRows} rows of local values for attribute '{$attributeCode}'");
        }

        return $affectedRows;

    }

    public function filterPhoneNumber($phonenumber, $countryCode = 'SE')
    {
        $filteredPhonenumber = $phonenumber;
        if (!empty($phonenumber)) {

            switch ($countryCode) {
                case 'NO':
                    $searchAndReplace = [
                        'search' => [
                            '/\(0\)/', '/[^\+\d]/', '/^0047/', '/^00/', '/^([^\+])/'
                        ],
                        'replace' => [
                            '', '', '+47', '+', '+47\1'
                        ]
                    ];
                    $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                    break;

                case 'DK':
                    $searchAndReplace = [
                        'search' => [
                            '/\(0\)/', '/[^\+\d]/', '/^0045/', '/^00/', '/^([^\+])/'
                        ],
                        'replace' => [
                            '', '', '+45', '+', '+45\1'
                        ]
                    ];
                    $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                    break;

                case 'FI':
                    $searchAndReplace = [
                        'search' => [
                            '/\(0\)/', '/[^\+\d]/', '/^00358/', '/^358/', '/^0/', '/^[^\+]/'
                        ],
                        'replace' => [
                            '', '', '+358', '+358', '+358', '+358'
                        ]
                    ];
                    $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                    break;

                case 'SE':
                    $searchAndReplace = [
                        'search' => [
                            '/\(0\)/', '/^\+46/', '/[^\+\d]/', '/^0046/', '/^46/', '/^(\+)?(0)*7/', '/^00/', '/^0123456$/'
                        ],
                        'replace' => [
                            '', '0', '', '0', '0', '07', '+', ''
                        ]
                    ];
                    $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                    break;

                default:
                    $searchAndReplace = [
                        'search' => [
                            '/\(0\)/', '/[^\+\d]/', '/^00/', '/^([^\+])/'
                        ],
                        'replace' => [
                            '', '', '+', '+\1'
                        ]
                    ];
                    $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                    break;
            }
        }

        return $filteredPhonenumber;
    }

    public function createZeroPriceProductsTable()
    {
        $data = Mage::getStoreConfig("integration/products/zero_price_products");
        if (!empty($data)) {
            $data = json_decode($data, true);
            if (is_array($data)) {
                $this->_zeroPriceProducts = $data;
            }
        }
    }

    public function isZeroPriceProduct($sku)
    {
        if (empty($this->_zeroPriceProducts)) {
            $this->createZeroPriceProductsTable();
        }
        return array_key_exists($sku, $this->_zeroPriceProducts);
    }

    public function isEmailBlacklisted($email, $storeId) {
        $rawBlacklist = Mage::getStoreConfig("integration/orders/email_blacklist", $storeId);
        if (!empty($rawBlacklist)) {
            $blacklist = unserialize($rawBlacklist);
            if (!empty($blacklist) && is_array($blacklist)) {
                foreach ($blacklist as $blacklistRow) {
                    if (!empty($blacklistRow["email_address"]) && !empty($email) && strtolower($blacklistRow["email_address"]) == strtolower($email)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    public function convertString($strParam, $mbCase = "none")
    {
        $isUTF8 = false;

        if (!empty($strParam)) {
            $strParam = trim($strParam);
        } else {
            return $strParam;
        }

        switch (mb_detect_encoding($strParam, "UTF-8, ISO-8859-1", true)) {
            case 'UTF-8':
                $isUTF8 = true;
                break;

            default:
                break;
        }

        if (!$isUTF8) {
            $strParam = utf8_encode($strParam);
        }

        switch ($mbCase) {
            case "upper":
                $strParam = mb_convert_case($strParam, MB_CASE_UPPER, "UTF-8");
                break;

            case "lower":
                $strParam = mb_convert_case($strParam, MB_CASE_LOWER, "UTF-8");
                break;

            case "title":
                $strParam = mb_convert_case($strParam, MB_CASE_TITLE, "UTF-8");
                break;

            case "none":
            default:
                break;
        }

        return $strParam;
    }

    public function isUTF8($strParam)
    {
        return mb_detect_encoding($strParam, "UTF-8, ISO-8859-1", true) === "UTF-8";
    }

    public function sendAdminEmail($subject, $body, $recipients = [])
    {
        if ($this->getDebugMode()) {
            // Always override email recipients in debug mode
            $recipients = ["it"];
        }

        $filteredRecipients = [];
        if (!is_array($recipients)) {
            if (!empty($recipients)) {
                $recipients = [$recipients];
            } else {
                $recipients = [];
            }
        }

        foreach($recipients as $recipient) {
            if (array_key_exists($recipient, self::$adminEmailRecipients)) {
                foreach (self::$adminEmailRecipients[$recipient] as $address) {
                    $filteredRecipients[] = $address;
                }
            } elseif(stripos($recipient, "@") !== false) {
                $filteredRecipients[] = $recipient;
            }
        }

        if (empty($filteredRecipients)) {
            // Default is to always send to IT
            foreach (self::$adminEmailRecipients['it'] as $address) {
                $filteredRecipients[] = $address;
            }
        }

        $recipientList = implode("', '", $filteredRecipients);
        if (count($filteredRecipients) == 1) {
            $body .= "\n\n(this email was ONLY sent to '{$recipientList}')";
        } else {
            $body .= "\n\n(this email was sent to '{$recipientList}')";
        }
        try {
            foreach ($filteredRecipients as $emailAddress) {
                mail($emailAddress, $subject, $body);
            }
        } catch (Exception $ex) {
            $this->logException($ex, "Exception sending email");
            $this->log($body, LOG_ERR);
        }
    }

    public function sendAdminHTMLEmail($subject, $body, $recipients = [])
    {
        if ($this->getDebugMode()) {
            // Always override email recipients in debug mode
            $recipients = ["it"];
        }

        $filteredRecipients = [];
        if (!is_array($recipients)) {
            if (!empty($recipients)) {
                $recipients = [$recipients];
            } else {
                $recipients = [];
            }
        }

        foreach($recipients as $recipient) {
            if (array_key_exists($recipient, self::$adminEmailRecipients)) {
                foreach (self::$adminEmailRecipients[$recipient] as $address) {
                    $filteredRecipients[] = $address;
                }
            } elseif(stripos($recipient, "@") !== false) {
                $filteredRecipients[] = $recipient;
            }
        }

        if (empty($filteredRecipients)) {
            // Default is to always send to IT
            foreach (self::$adminEmailRecipients['it'] as $address) {
                $filteredRecipients[] = $address;
            }
        }

        $message = "<html><head><title>{$subject}</title></head><body>";
        $message .= $body;

        $recipientList = implode("', '", array_keys($filteredRecipients));
        if (count($filteredRecipients) == 1) {
            $message .= "<p>(this email was ONLY sent to '{$recipientList}')</p>";
        } else {
            $message .= "<p>(this email was sent to '{$recipientList}')</p>";
        }

        $message .= "</body></html>";

        $headers[] = 'MIME-Version: 1.0';
        $headers[] = "Content-type: text/html; charset=utf-8";
        $headers[] = "Content-Transfer-Encoding: base64";

        try {
            mail(
                implode(", ", $filteredRecipients),
                "=?UTF-8?B?" . base64_encode($subject) . "?=",
                base64_encode($message),
                implode("\r\n", $headers)
            );
        } catch (Exception $ex) {
            $this->logException($ex, "Exception sending email");
            $this->log($body, LOG_ERR);
        }
    }

    public function log($message, $logType = LOG_DEBUG, $filename = self::DEFAULT_LOG_FILENAME)
    {
        if ($this->getDebugMode()) {
            echo "{$message}\n";
        }
        Mage::log("{$message}", $logType, $filename, true);
    }


    public function logException($exception, $messages = [])
    {
        if (!is_array($messages)) {
            $messages = [$messages];
        }

        $messages[] = (string)$exception;

        $this->log(trim(implode("\n", $messages), " \t\n\r"), LOG_ERR);
    }

    public function checkIP()
    {
        $IPs = [
            '127.0.0.1', // Localhost
            '5.150.194.196', // Crossroads firewall
            '77.72.102.100', // Awardit firewall
            '94.247.172.163', // api1
            '10.30.0.[0-9]{1,3}', // Internal network
        ];

        $remotes = ($_SERVER['HTTP_X_FORWARDED_FOR'] ?? '')
                 . ',' . ($_SERVER['REMOTE_ADDR'] ?? '');

        foreach ($IPs as $IP) {
            if (preg_match("/({$IP})/", $remotes, $m)) {
                return;
            }
        }

        $this->getResponse()->setHttpResponseCode(200);
        die("<p><a href=\"http://www.youtube.com/watch?v=RfiQYRn7fBg\">Ah ah ah, you didn't say the magic word...</a></p>");

    }

}

Awardit_Integration_Helper_Data::$basePath = Mage::getBaseDir("log").DS.'integration'.DS;
