<?php

class Awardit_Integration_Model_Ordermanager extends Mage_Core_Model_Abstract {

    /**
     * @return array
     */
    public function refundOrder($incrementId, $username = "unknown")
    {
        $order = Mage::getModel("sales/order");

        try {
            // Find entity id matching increment id
            $sqlQuery = "SELECT entity_id FROM sales_flat_order WHERE increment_id = ?";
            $orderId = intval(Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $incrementId));
            if (empty($orderId)) {
                return [
                    "status" => 0,
                    "message" => "Unable to find order {$incrementId}\n"
                ];
            }

            // Load order
            $order->load($orderId);
            if (!$order || $order->getId() != $orderId) {
                return [
                    "status" => 0,
                    "message" => "Unable to load order {$incrementId}\n",
                ];
            }

            // Check if credit memo can be created
            if (!$order->canCreditmemo()) {
                return [
                    "status" => 0,
                    "message" => "Unable to create creditmemo for order {$incrementId}\n",
                ];
            }

            $comment = "Order was refunded in Fraud Manager by {$username}.<br>";
            $order->getPayment()->setPreparedMessage($comment);

            // Start: ############################

            // Fetch collection of invoices
            $invoiceCollection = $order->getInvoiceCollection();
            foreach($invoiceCollection as $invoice) {

                // Check if invoice can be refunded
                if (!$invoice->canRefund()) {
                    continue;
                }

                // Load invoice
                $realInvoice = Mage::getModel('sales/order_invoice')
                ->load($invoice->getId())
                ->setOrder($order);

                // Prepare invoice for refund
                $data = [];
                $items = $realInvoice->getAllItems();
                foreach ($items as $item) {
                    $data['qtys'][$item->getOrderItemId()]["qty"] = $item->getQty();
                }

                $service = Mage::getModel('sales/service_order', $order);
                $creditmemo = $service->prepareInvoiceCreditmemo($realInvoice, $data);
                $creditmemo->setRefundRequested(true);
                $creditmemo->setOfflineRequested(false);
                $creditmemo->getOrder()->setCustomerNoteNotify(false);
                $creditmemo->register();

                // Create and save transaction.
                $transactionSave = Mage::getModel('core/resource_transaction')
                    ->addObject($creditmemo)
                    ->addObject($order)
                    ->addObject($realInvoice);
                $transactionSave->save();

            }

            // End: ############################

            return [
                "status" => 1,
                "message" => "successfully refunded online\n",
                "result" => [
                    "incrementId" => $incrementId,
                    "text" => "Order was successfully refunded online.",
                ]
            ];

        } catch (Exception $ex) {
            Mage::logException($ex);

            // If order was loaded, create history for this refund attempt
            if ($order && $order->getId()) {
                $history = $order->addStatusHistoryComment("Refund initiated by {$username} from Fraud Manager.<br>{$ex->getMessage()}");
                $history->setIsCustomerNotified(Mage_Sales_Model_Order_Status_History::CUSTOMER_NOTIFICATION_NOT_APPLICABLE);
                $order->save();
            }

            return [
                "status" => 1,
                "message" => $ex->getMessage() . "\n",
                "result" => [
                    "incrementId" => $incrementId,
                    "text" => $ex->getMessage(),
                ],
                "debug" => [
                    "trace" => $ex->getTraceAsString()
                ]
            ];
        }
    }
}