<?php

class Awardit_Integration_Model_Cli_OrderExport extends Awardit_Integration_Model_Cli {

    const EXPORT_STATUS_NEW          =  1; // New
    const EXPORT_STATUS_PREPARED     =  2; // Prepared (XML-file created and saved)
    const EXPORT_STATUS_TRANSFERRED1 =  3; // Transferred Step 1 (XML-file transferred to Visma)
    const EXPORT_STATUS_TRANSFERRED2 =  4; // Transferred Step 2 (XML-file transferred to Visma)
    const EXPORT_STATUS_VERIFIED     =  5; // Verified (Have visma_order_id and visma_status)
    const EXPORT_STATUS_SHIPPED      =  6; // Shipped (Order status (4) in Visma indicate order has shipped)
    const EXPORT_STATUS_COMPLETE     =  7; // Order complete
    const EXPORT_STATUS_CANCELED     =  8; // Order has been canceled (Order status (6) in Visma)
    const EXPORT_STATUS_ERROR        =  9; // Order have some kind of error
    const EXPORT_STATUS_TEST         = 10; // Test
    const EXPORT_STATUS_BLACKLIST    = 11; // Order placed by blacklisted user
    const EXPORT_STATUS_ARCHIVED     = 99; // Archived order

    const VISMA_SEND_TO_WAREHOUSE = 0;
    const VISMA_DO_NOT_SEND_TO_WAREHOUSE = 1;

    const VISMA_FAULTY_LOCK_STATES = [1,3];

    const VISMA_ORDER_STATUS_ORDERED   = 1;
    const VISMA_ORDER_STATUS_HOLD      = 2;
    const VISMA_ORDER_STATUS_PICKING   = 3;
    const VISMA_ORDER_STATUS_DELIVERED = 4;
    const VISMA_ORDER_STATUS_CANCELED  = 5;

    protected $_wasLastOrderUpdated = false;

    // -m=OrderExport -f=scanForNewOrders
    public function CLI_scanForNewOrders($param)
    {
        $this->scanForNewOrders();
    }

    // -m=OrderExport -f=tagSingleOrder -p="incrementId:<incrementId>"
    public function CLI_tagSingleOrder($param)
    {
        $wantedParams = [
            "incrementId" => "string"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["incrementId"])) {
            $this->tagSingleOrder($extractedParams["incrementId"]);
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
        }
    }

    // -m=OrderExport -f=prepareSingleOrder -p="integrationId:<integrationId>|incrementId:<incrementId>"
    public function CLI_prepareSingleOrder($param)
    {
        $wantedParams = [
            "integrationId" => "int",
            "incrementId" => "string",
            "skipConfigCheck" => "bool"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $integrationOrder = null;
        if (!empty($extractedParams["id"])) {
            $integrationOrder = $this->getOrderByIntegrationId($extractedParams["id"]);
        } elseif (!empty($extractedParams["incrementId"])) {
            $integrationOrder = $this->getOrderByIncrementId($extractedParams["incrementId"]);
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
            return;
        }

        if (empty($integrationOrder)) {
            Mage::helper($this->_defaultHelper)->log("Unable to load integration order!", LOG_ERR);
            return;
        }

        $this->prepareSingleOrder($integrationOrder);
    }

    // -m=OrderExport -f=prepareAllOrders
    public function CLI_prepareAllOrders($param)
    {
        $integrationOrders = $this->getAllOrdersToExport();

        if (!empty($integrationOrders)) {
            $numOrders = count($integrationOrders);
            $preparedOrders = 0;

            Mage::helper($this->_defaultHelper)->log("Starting to prepare all (max {$this->getLimit()}) orders. Found {$numOrders} in queue.");

            foreach ($integrationOrders as $integrationOrder) {
                $this->prepareSingleOrder($integrationOrder);

                $preparedOrders++;
                if ($this->getLimit() > 0) {
                    if ($preparedOrders >= $this->getLimit()) {
                        break;
                    }
                }

            }

            Mage::helper($this->_defaultHelper)->log("Prepared {$preparedOrders} orders.");

        } else {
            Mage::helper($this->_defaultHelper)->log("Found no orders to prepare.");
        }
    }

    // -m=OrderExport -f=updateSingleOrder -p="integrationId:<integrationId>|incrementId:<incrementId>"
    public function CLI_updateSingleOrder($param)
    {
        $wantedParams = [
            "integrationId" => "int",
            "incrementId" => "string"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $integrationOrder = null;
        if (!empty($extractedParams["id"])) {
            $integrationOrder = $this->getOrderByIntegrationId($extractedParams["id"]);
        } elseif (!empty($extractedParams["incrementId"])) {
            $integrationOrder = $this->getOrderByIncrementId($extractedParams["incrementId"]);
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
            return;
        }

        if (empty($integrationOrder)) {
            Mage::helper($this->_defaultHelper)->log("Unable to load integration order!", LOG_ERR);
            return;
        }

        $this->updateSingleOrder($integrationOrder);
    }

    // -m=OrderExport -f=updateAllOrders [ -p="fromStatus:<status>,toStatus:<status>" | -p="status=<status>" ]
    public function CLI_updateAllOrders($param)
    {
        $wantedParams = [
            "fromStatus" => "int",
            "toStatus" => "int",
            "status" => "int"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        if (!empty($extractedParams["fromStatus"])) {
            $this->setExportStatusFrom($extractedParams["fromStatus"]);
        }
        if (!empty($extractedParams["toStatus"])) {
            $this->setExportStatusTo($extractedParams["toStatus"]);
        }
        if (!empty($extractedParams["status"])) {
            $this->setExportStatusFrom($extractedParams["status"]);
            $this->setExportStatusTo($extractedParams["status"]);
        }

        $ordersToUpdate = $this->getAllOrdersToUpdate();
        if (!empty($ordersToUpdate)) {
            $maxQty = count($ordersToUpdate);

            $iterations = 0;
            $updatedOrders = 0;
            Mage::helper($this->_defaultHelper)->log("Starting to update orders (max {$this->getLimit()}). Found {$maxQty} in queue.");

            foreach($ordersToUpdate as $integrationOrder) {
                $this->updateSingleOrder($integrationOrder);
                $updatedOrders += $this->getWasLastOrderUpdated() ? 1 : 0;
                $iterations++;
                if ($this->getLimit() > 0) {
                    if ($iterations >= $this->getLimit()) {
                        break;
                    }
                }
            }

            Mage::helper($this->_defaultHelper)->log("Checked {$iterations} and updated {$updatedOrders} orders.");
        }
    }

    // -m=OrderExport -f=testXML -p="integrationId:<integrationId>|incrementId:<incrementId>"
    public function CLI_testXML($param)
    {
        $this->setTestMode(true);

        $wantedParams = [
            "integrationId" => "int",
            "incrementId" => "string",
            "skipConfigCheck" => "bool"
        ];
        $extractedParams = $this->extractParams($param, $wantedParams);

        $integrationOrder = null;
        if (!empty($extractedParams["id"])) {
            $integrationOrder = $this->getOrderByIntegrationId($extractedParams["id"]);
        } elseif (!empty($extractedParams["incrementId"])) {
            $integrationOrder = $this->getOrderByIncrementId($extractedParams["incrementId"]);
        } else {
            Mage::helper($this->_defaultHelper)->log("Missing parameter!", LOG_ERR);
            return;
        }

        if (empty($integrationOrder)) {
            Mage::helper($this->_defaultHelper)->log("Unable to load integration order!", LOG_ERR);
            return;
        }

        $orderData = $this->getOrderData($integrationOrder);
        if ($orderData === false) {
            return;
        }

        // Check if order contains any items that's not supposed to be exported
        foreach (array_keys($orderData["xmlData"]["items"]) as $itemId) {
            if (Mage::helper($this->_defaultHelper)->isItemBlacklisted($orderData["magento_store_id"], $orderData["xmlData"]["items"][$itemId]["OrdLn_ProdNo"])) {
                // If so, delete that order item row
                unset($orderData["xmlData"]["items"][$itemId]);
            }
        }

        $xmlData = $this->createXmlData($orderData);

        $newXMLObj = simplexml_load_string($xmlData->asXML(), null, LIBXML_NOCDATA);
        if (!empty($newXMLObj)) {
            $dom = new DOMDocument("1.0", "UTF-8");
            $dom->formatOutput = true;
            $dom->loadXML(utf8_encode($newXMLObj->asXML()));
            $dom->save("php://output");
        } else {
            Mage::helper($this->_defaultHelper)->log("Unable to generate XML data!", LOG_ERR);
            echo print_r($orderData, true);
        }
    }

    // -m=OrderExport -f=copyXMLtoDB
    public function CLI_copyXMLtoDB($param)
    {
        $this->setTestMode(true);
        $basePath = Awardit_Integration_Helper_Data::$basePath . "transferred" . DS;
        $dirs = [];

        $handle = opendir($basePath);
        if ($handle) {
            while (false !== ($entry = readdir($handle))) {
                $parts = explode("-", $entry);
                if (empty($parts[1])) {
                    continue;
                }

                if (intval($parts[0]) < 2022) {
                    // Module Crossroads/Integration was replaced by Awardit/Integration 2022
                    continue;
                }

                $dirs[] = $entry;
            }
            closedir($handle);
        }
        printf("Found %d directories to parse.\n", count($dirs));

        if (empty($dirs)) {
            return;
        }

        $sqlQuery = "UPDATE awardit_integration_orders SET xml_data = :xml WHERE increment_id = :id AND xml_data IS NULL";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);

        foreach($dirs as $subDir) {
            $subPath = $basePath.$subDir.DS;

            $handle = opendir($subPath);
            if ($handle) {

                echo "Checking directory: $subPath\n";
                $fileQty = 0;

                while (false !== ($entry = readdir($handle))) {

                    $filename = $subPath.$entry;
                    echo "$entry";
                    if (!is_file($filename)) {
                        echo " - skipping.\n";
                        continue;
                    }

                    $fileParts = explode(".", $entry);
                    if (empty($fileParts[0])) {
                        echo " - no incrementId.\n";
                        continue;
                    }

                    $incrementId = $fileParts[0];
                    $xmlData = file_get_contents($filename);

                    if (empty($xmlData)) {
                        echo " - no xml-data.\n";
                        continue;
                    }

                    $stmt->bindValue("xml", $xmlData);
                    $stmt->bindValue("id", $incrementId);

                    try {
                        $stmt->execute();
                        $affectedRows = $stmt->rowCount();
                        if ($affectedRows > 0) {
                            echo " - copied.\n";
                            $fileQty += 1;
                        } else {
                            echo " - no rows updated.\n";
                        }
                    } catch (Exception $ex) {
                        echo "\n\nException!\n";
                        echo $ex->getMessage()."\n";
                        echo $ex->getTraceAsString()."\n";
                        return;
                    }
                }

                closedir($handle);
                echo "---------------------------------------\n";
                printf("Copied data for %d orders.\n", $fileQty);
            }
        }
    }

    public function scanForNewOrders()
    {
        Mage::helper($this->_defaultHelper)->log("Scanning for new orders.");

        $sqlQuery = "
            INSERT INTO " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . "
                (`increment_id`,`magento_order_id`,`magento_status`,`order_category`,`export_status`)
                SELECT
                    sfo.increment_id,
                    sfo.entity_id AS magento_order_id,
                    sfo.`status` AS magento_status,
                    :orderCategory AS order_category,
                    1 AS export_status
                FROM sales_flat_order sfo
                LEFT JOIN " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " cio ON cio.magento_order_id = sfo.entity_id
                WHERE
                    sfo.created_at > :longTimeAgo
                    AND sfo.created_at < :someTimeAgo
                    AND cio.id IS NULL
        ";
        $params = [
            "orderCategory" => intval(Mage::getStoreConfig("integration/orders/visma_order_category")),
            "longTimeAgo" => date("Y-m-d H:i:s", strtotime(Awardit_Integration_Helper_Data::ORDER_AGE_MAX)),
            "someTimeAgo" => date("Y-m-d H:i:s", strtotime(Awardit_Integration_Helper_Data::ORDER_AGE_MIN))
        ];
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        foreach ($params as $key => $val) {
            $stmt->bindValue($key, $val);
        }

        try {
            $stmt->execute();
            $affectedRows = $stmt->rowCount();
            Mage::helper($this->_defaultHelper)->log("Tagged {$affectedRows} orders as new.");
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while scanning for new orders!");
            return false;
        }

        return true;
    }

    public function tagSingleOrder($incrementId)
    {
        Mage::helper($this->_defaultHelper)->log("Tagging order {$incrementId} for export.");

        $sqlQuery = "
            INSERT INTO " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . "
                (`increment_id`,`magento_order_id`,`magento_status`,`order_category`,`export_status`)
                SELECT
                    sfo.`increment_id`,
                    sfo.`entity_id` AS magento_order_id,
                    sfo.`status` AS magento_status,
                    :orderCategory AS order_category,
                    1 AS export_status
                FROM sales_flat_order sfo
                WHERE sfo.increment_id = :incrementId
            ON DUPLICATE KEY UPDATE `order_category` = VALUES(`order_category`), `export_status` = VALUES(`export_status`), `visma_order_id` = NULL, `visma_db` = NULL, `visma_status` = 0
        ";
        $params = [
            "orderCategory" => intval(Mage::getStoreConfig("integration/orders/visma_order_category")),
            "incrementId" => $incrementId,
        ];
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        foreach ($params as $key => $val) {
            $stmt->bindValue($key, $val);
        }

        try {
            $stmt->execute();
            Mage::helper($this->_defaultHelper)->log("Tagged {$incrementId} as new.");
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while tagging order!");
            return false;
        }

        return true;
    }

    public function getWasLastOrderUpdated()
    {
        return $this->_wasLastOrderUpdated;
    }

    public function getSingleOrderToExport()
    {
        return $this->getOrdersByAttribute("export_status", self::EXPORT_STATUS_NEW, true);
    }

    public function getAllOrdersToExport()
    {
        return $this->getOrdersByAttribute("export_status", self::EXPORT_STATUS_NEW, false);
    }

    public function getSingleOrderByStatus($status)
    {
        return $this->getOrdersByAttribute("export_status", $status, false);
    }

    public function getAllOrdersByStatus($status)
    {
        return $this->getOrdersByAttribute("export_status", $status, false);
    }

    public function getOrderByIntegrationId($id)
    {
        return $this->getOrdersByAttribute("id", $id, true);
    }

    public function getOrderByIncrementId($incrementId)
    {
        return $this->getOrdersByAttribute("increment_id", $incrementId, true);
    }

    public function getOrdersByAttribute($attribute, $value, $single = false)
    {
        switch ($attribute) {
            case "id":
            case "increment_id":
            case "magento_order_id":
            case "magento_status":
            case "visma_order_id":
            case "visma_status":
            case "export_status":
                $sqlQuery = "SELECT * FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " WHERE {$attribute} = ? ORDER BY id DESC";
                if ($single) {
                    $sqlQuery .= " LIMIT 1";
                    return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, $value);
                } else {
                    return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $value);
                }
                break;

            default:
                Mage::helper($this->_defaultHelper)->log("getOrdersByAttribute: Unknown attribute '{$attribute}'", LOG_ERR);
                break;
        }
        return [];

    }

    public function getSingleOrderToUpdate()
    {
        return $this->_getOrdersToUpdate(true);
    }

    public function getAllOrdersToUpdate()
    {
        return $this->_getOrdersToUpdate(false);
    }

    private function _getOrdersToUpdate($single = false)
    {
        $fromStatus = $this->getExportStatusFrom();
        $toStatus = $this->getExportStatusTo();

        if ($fromStatus > $toStatus) {
            $toStatus = $this->getExportStatusFrom();
            $fromStatus = $this->getExportStatusTo();
        }

        /*
         * Order first by export status, then by random.
         * This is done since order update only runs a couple of times each day.
         * We don't want to always get the same list of orders to check, each run.
         */
        $sqlQuery = "
            SELECT
                i.*,
                DATEDIFF(NOW(), o.created_at) AS days_old
            FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " i
            JOIN sales_flat_order o ON o.entity_id = i.magento_order_id
            WHERE i.export_status BETWEEN {$fromStatus} AND {$toStatus}
            ORDER BY i.export_status, rand()
        ";

        if ($single) {
            $sqlQuery .= " LIMIT 1";
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery);
        } else {
            $limit = $this->getLimit() ?: $this->getDefaultLimit();
            $sqlQuery .= " LIMIT {$limit}";
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
        }
    }

    public function getMaxQtyToExport()
    {
        $status = self::EXPORT_STATUS_NEW;
        $sqlQuery = "SELECT count(id) AS max_qty FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " WHERE export_status = {$status}";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery);
    }

    public function getMaxQtyToUpdate()
    {
        $fromStatus = $this->getExportStatusFrom();
        $toStatus = $this->getExportStatusTo();

        if ($fromStatus > $toStatus) {
            $toStatus = $this->getExportStatusFrom();
            $fromStatus = $this->getExportStatusTo();
        }

        $sqlQuery = "SELECT count(id) AS max_qty FROM " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " WHERE export_status BETWEEN {$fromStatus} AND {$toStatus}";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery);
    }

    public function updateExportStatusById($id, $status)
    {
        $data = [
            "id" => $id,
            "export_status" => $status
        ];
        return $this->updateIntegrationOrder($data);
    }

    public function updateExportStatusByIncrementId($incrementId, $status)
    {
        $data = [
            "increment_id" => $incrementId,
            "export_status" => $status
        ];
        return $this->updateIntegrationOrder($data);
    }

    public function createIntegrationOrder($integrationOrder)
    {
        $values = [];

        foreach ($integrationOrder as $key => $val) {
            switch ($key) {
                case "increment_id":
                case "magento_order_id":
                case "magento_status":
                case "order_category":
                case "visma_order_id":
                case "visma_db":
                case "visma_status":
                case "export_status":
                case "xml_data":
                    $values[$key] = $val;
                    break;
                default:
                    break;
            }
        }

        if (empty($values["increment_id"])) {
            return false;
        }

        $sqlQuery = sprintf(
                "INSERT INTO %s (`%s`) VALUES (:%s)",
                Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS,
                implode("`,`", array_keys($values)),
                implode(", :", array_keys($values))
        );

        try {
            $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
            foreach ($values as $key => $val) {
                $stmt->bindValue($key, $val);
            }
            $stmt->execute();

            $id = Mage::getSingleton("core/resource")->getConnection("core_write")->lastInsertId();
            if (empty($id)) {
                Mage::helper($this->_defaultHelper)->log("Got no id from creating integration order!", LOG_ERR);
                return false;
            }

            return true;

        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while creating integration order!");
            return false;
        }
    }

    public function updateIntegrationOrder($integrationOrder, $storeId = 0)
    {
        $values = [];
        $status = false;

        foreach ($integrationOrder as $key => $val) {
            switch ($key) {
                case "id":
                case "increment_id":
                case "magento_order_id":
                case "magento_status":
                case "order_category":
                case "visma_order_id":
                case "visma_db":
                case "visma_status":
                case "export_status":
                case "xml_data":
                    $values[$key] = $val;
                    break;
                default:
                    break;
            }
        }

        if (!empty($values["id"]) || !empty($values["increment_id"])) {
            $sqlQuery = "UPDATE " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " SET ";

            if (!empty($values["id"])) {
                $where = " WHERE id = :id";
            } elseif (!empty($values["increment_id"])) {
                $where = " WHERE increment_id = :increment_id";
            }

            $updates = [];
            foreach(array_keys($values) as $key) {
                // If `id` is supplied, use that for $where and update `increment_id` (if supplied)
                // If only `increment_id` is supplied, use it for $where and skip its update
                if ($key == "id" || ($key == "increment_id" && empty($values["id"]))) {
                    continue;
                }
                $updates[] = "{$key} = :{$key}";
            }
            $sqlQuery .= implode(", ", $updates);
            $sqlQuery .= $where;

            try {
                $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
                foreach ($values as $key => $val) {
                    $stmt->bindValue($key, $val);
                }
                $stmt->execute();
                $status = true;

            } catch (Exception $exception) {
                Mage::helper($this->_defaultHelper)->logException($exception, "Exception while updating integration order!");
                $status = false;
            }
        }

        return $status;
    }

    public function updateIntegrationOrderStatusSpecial()
    {
        $exportStatusFrom = self::EXPORT_STATUS_TRANSFERRED1;
        $exportStatusTo = self::EXPORT_STATUS_TRANSFERRED2;

        $sqlQuery = "UPDATE " . Awardit_Integration_Helper_Data::INTEGRATION_TABLE_ORDERS . " SET export_status = :exportStatusTo WHERE export_status = :exportStatusFrom";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->bindValue("exportStatusFrom", $exportStatusFrom);
        $stmt->bindValue("exportStatusTo", $exportStatusTo);
        $stmt->execute();
    }

    public function prepareSingleOrder($integrationOrder)
    {
        $incrementId = $integrationOrder["increment_id"];
        $updateData = [];
        $orderData = [];

        // First, check if order changed in Magento since it was placed in the integration-table
        $newMagentoData = $this->getMagentoOrderData($integrationOrder["magento_order_id"]);
        if (!empty($newMagentoData["status"]) && $newMagentoData["status"] != $integrationOrder["magento_status"]) {
            // If the status has changed, prepare to update
            $updateData["magento_status"] = $newMagentoData["status"];
            $integrationOrder["magento_status"] = $newMagentoData["status"];
        }

        // Depending on status, do different things
        switch ($integrationOrder["magento_status"]) {
            // These should never be exported to Visma
            case "canceled":
            case "closed":
                $updateData["export_status"] = self::EXPORT_STATUS_CANCELED;
                break;

            // Don't process these right now, just wait.
            case "holded":
            case "pending":
            case "pending_payment":
            case "payment_review":
            case "klarna_pending":
                break;

            // These should be exported
            case "complete":
            case "processing":
            case "klarna_accepted":
                $orderData = $this->getOrderData($integrationOrder);
                if ($orderData === false) {
                    $orderData["export_status"] = self::EXPORT_STATUS_ERROR;
                    $updateData["export_status"] = $orderData["export_status"];
                    break;
                }

                $orderData["export_status"] = $this->handleBlacklist($orderData);
                $updateData["export_status"] = $orderData["export_status"];
                if ($orderData["export_status"] !== self::EXPORT_STATUS_PREPARED) {
                    break;
                }

                if ($orderData["visma_order_db"] !== $integrationOrder["visma_db"]) {
                    $updateData["visma_db"] = $orderData["visma_order_db"];
                }

                $XMLData = $this->createXmlData($orderData);
                $cleanXMLData = simplexml_load_string($XMLData->asXML(), null, LIBXML_NOCDATA);
                $updateData["xml_data"] = $cleanXMLData->asXML();
                break;

            // Unknown status, just log it and do not export
            default:
                Mage::helper($this->_defaultHelper)->log("Unknown order status '{$integrationOrder["magento_status"]}' for order {$incrementId}", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Unknown order status '{$integrationOrder["magento_status"]}' for order {$incrementId}",
                    print_r($integrationOrder, true),
                    ["it"]
                );
                $updateData["export_status"] = self::EXPORT_STATUS_ERROR;
                break;
        }

        if (!empty($updateData)) {
            $updateData["id"] = $integrationOrder["id"];

            if (!empty($orderData["xmlData"]["items"])) {

                // Check if order contains any items that's not supposed to be exported
                foreach (array_keys($orderData["xmlData"]["items"]) as $itemId) {
                    if (Mage::helper($this->_defaultHelper)->isItemBlacklisted($newMagentoData["store_id"], $orderData["xmlData"]["items"][$itemId]["OrdLn_ProdNo"])) {
                        // If so, delete that order item row
                        unset($orderData["xmlData"]["items"][$itemId]);
                    }
                }

                // If order only contained blacklisted items, no more order item rows should exist
                if (empty($orderData["xmlData"]["items"])) {
                    // If so, mark order as complete
                    $updateData["export_status"] = self::EXPORT_STATUS_COMPLETE;
                }
            }

            $this->updateIntegrationOrder($updateData);
        }

    }

    public function getMagentoOrderData($id)
    {
        $sqlQuery = "SELECT entity_id, increment_id, `status`, store_id  FROM sales_flat_order WHERE entity_id = ?";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, $id);
    }

    public function getVismaOrderData($incrementId, $vismaDbPrefix)
    {
        if (empty($incrementId) || empty($vismaDbPrefix)) {
            return [];
        }

        $sqlQuery = "
            SELECT
                OrdNo,
                Inf,
                Gr,
                WebPg,
                InvoNo,
                LckSt,
                CASE WHEN RlDelDt > 0 THEN
                    CONCAT(FORMAT(RlDelDt, '0000-00-00'), ' ', FORMAT(ConCnt, '00:00'),':00')
                ELSE
                    null
                END AS release_iso,
                FORMAT(GETDATE(),'yyyy-MM-dd HH:mm:ss') AS now_iso
            FROM {$vismaDbPrefix}.dbo.Ord
            WHERE Inf = ?
        ";
        $tryAgain = false;

        do {
            try {
                $data = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchRow($sqlQuery, $incrementId);
                if (!empty($data)) {
                    $retArr = [
                        "OrdNo" => empty($data["OrdNo"]) ? 0 : intval($data["OrdNo"]),     // Ordernummer
                        "Inf" => empty($data["Inf"]) ? null : trim($data["Inf"]),          // increment_id
                        "Gr" => empty($data["Gr"]) ? 0 : intval($data["Gr"]),              // Orderstatus
                        "WebPg" => empty($data["WebPg"]) ? null : trim($data["WebPg"]),    // Trackingnummer
                        "InvoNo" => empty($data["InvoNo"]) ? null : trim($data["InvoNo"]), // Fakturanummer
                        "LckSt" => empty($data["LckSt"]) ? 0 : intval($data["LckSt"]),     // Lock state
                        "safe_to_complete" => false,
                        "safe_to_send_email" => false,
                        "times" => [
                            "now_timestamp" => strtotime($data["now_iso"]),
                            "now_iso" => $data["now_iso"],
                            "release_timestamp" => 0,
                            "release_iso" => $data["release_iso"],
                            "check_timestamp" => 0,
                            "check_iso" => "",
                            "safe_email_timestamp" => 0,
                            "safe_email_iso" => ""
                        ]
                    ];
                    $retArr["times"]["check_timestamp"] = strtotime("-30 minutes", $retArr["times"]["now_timestamp"]);
                    $retArr["times"]["check_iso"] = date("Y-m-d H:i:s", $retArr["times"]["check_timestamp"]);

                    if (!empty($retArr["times"]["release_iso"])) {
                        $retArr["times"]["release_timestamp"] = strtotime($retArr["times"]["release_iso"]);
                    } else {

                        // Orders without release date and/or time may be considered released anyway.
                        // Orders with only virtual products have this
                        // But this is only vaild for orders with status 4 from Visma!
                        if ($retArr["Gr"] == self::VISMA_ORDER_STATUS_DELIVERED) {

                            // Fake the timestamp to make order safe to make complete
                            $retArr["times"]["release_timestamp"] = $retArr["times"]["check_timestamp"] - 1;
                        } else {

                            // Fake the timestamp to make order NOT safe to make complete
                            $retArr["times"]["release_timestamp"] = $retArr["times"]["check_timestamp"] + 1;
                        }
                    }

                    // Set bool to reflect if order is safe to complete
                    $retArr["safe_to_complete"] = ($retArr["times"]["release_timestamp"] < $retArr["times"]["check_timestamp"]);

                    if ($retArr["safe_to_complete"]) {

                        // Disable shipping email if order was completed over 3 days ago.
                        $retArr["times"]["safe_email_timestamp"] = strtotime(Awardit_Integration_Helper_Data::SHIPPING_EMAIL_TIME_LIMIT, $retArr["times"]["now_timestamp"]);
                        $retArr["times"]["safe_email_iso"] = date("Y-m-d H:i:s", $retArr["times"]["safe_email_timestamp"]);
                        $retArr["safe_to_send_email"] = ($retArr["times"]["release_timestamp"] > $retArr["times"]["safe_email_timestamp"]);
                    }

                    return $retArr;
                }

                return [];

            } catch (Exception $exception) {
                if (!$tryAgain) {
                    Mage::helper($this->_defaultHelper)->logException($exception, "Unable to get Visma order data for order {$incrementId}, trying to reconnect.");
                    Mage::helper($this->_defaultHelper)->reconnectVismaDB();
                } else {
                    Mage::helper($this->_defaultHelper)->logException($exception, "Unable to get Visma order data for order {$incrementId}, giving up.");
                }
                $tryAgain = !$tryAgain;
            }
        } while ($tryAgain);

        return [];
    }

    public function getOrderData($integrationOrder)
    {
        if (empty($integrationOrder["magento_order_id"])) {
            Mage::helper($this->_defaultHelper)->log("magento_order_id is empty!", LOG_ERR);
            return false;
        }

        // Try to load order
        $order = Mage::getModel("sales/order")->load($integrationOrder["magento_order_id"]);
        $magentoOrderId = $order->getId();

        if (empty($magentoOrderId)) {
            Mage::helper($this->_defaultHelper)->log("Unable to load order!", LOG_ERR);
            return false;
        }

        if ($magentoOrderId != $integrationOrder["magento_order_id"]) {
            Mage::helper($this->_defaultHelper)->log("Wanted order and loaded order does not match!", LOG_ERR);
            return false;
        }

        // Yes, we have successfully loaded order
        $magentoStoreId = intval($order->getStoreId());
        $vismaStoreId = intval(Mage::getStoreConfig("integration/general/visma_shop_id", $magentoStoreId));
        $incrementId = $order->getIncrementId();

        // Check to see if we are supposed to export orders for this store
        if (!Mage::getStoreConfig("integration/orders/export", $magentoStoreId) && !$this->getExtraConfig("skipConfigCheck")) {
            Mage::helper($this->_defaultHelper)->log("We are not supposed to export orders from this Magento store, #{$magentoStoreId}.");
            return false;
        }

        // Check to see if we got visma_shop_id
        if (empty($vismaStoreId)) {
            Mage::helper($this->_defaultHelper)->log("Missing visma_shop_id for order {$incrementId}.", LOG_ERR);
            return false;
        }

        // Fetch currency and country ids
        $currencyId = Mage::helper($this->_defaultHelper)->getVismaCurrencyId($order->getStoreCurrencyCode());
        $addressId = $order->getShippingAddressId() ?: $order->getBillingAddressId() ?: null;
        $countryId = null;
        if (!empty($addressId)) {
            $address = Mage::getModel("sales/order_address")->load($addressId);
            $countryId = Mage::helper($this->_defaultHelper)->getVismaCountryId($address->getCountryId());
        }
        if (empty($countryId)) {
            $countryId = $currencyId;
        }

        // Create data array
        $orderData = [
            "xmlData" => [
                "order_head" => [
                    "Ord_Inf" => $order->getIncrementId(), // Order identifier
                    "Ord_Inf3" => "", // Payment reference / transaction id
                    "Ord_Inf4" => "", // Payment method
                    "Ord_Inf5" => $order->getAppliedRuleIds(), // Discount rule ids
                    "Ord_Gr" => self::VISMA_ORDER_STATUS_ORDERED,
                    "Ord_Gr6" => $integrationOrder["order_category"], // Order category
                    "Ord_Gr11" => $order->getIsVirtual() ? self::VISMA_DO_NOT_SEND_TO_WAREHOUSE : self::VISMA_SEND_TO_WAREHOUSE, // Do not send virtual orders to warehouse
                    "Ord_R1" => $vismaStoreId, // Visma store id
                    "Ord_R5" => $vismaStoreId, // Unknown, same as R1 (Visma store id)
                    "Ord_CustNo" => Mage::getStoreConfig("integration/orders/visma_customer_id", $magentoStoreId) ?: Awardit_Integration_Helper_Data::DEFAULT_VISMA_CUSTOMER_ID, // Visma customer id
                    "Ord_Cur" => $currencyId, // Currency id
                    "Ord_ExRt" => Mage::helper($this->_defaultHelper)->getVismaCurrencyExchangeRate($currencyId), // Currency exchange rate (percent)
                    "Ord_OrdDt" => date("Ymd", strtotime($order->getCreatedAt())), // Order creation date
                    "Ord_CardNm" => $order->getCreatedAt() // Order creation timestamp
                ],
                "items" => [],
                "shipping" => [],
                "fees" => [],
                "discount" => []
            ],
            "struct_items" => [], // Special Visma structure products,
            "visma_order_db" => Mage::helper($this->_defaultHelper)->getVismaOrderDB($magentoStoreId), // Visma order db
            "visma_store_id" => $vismaStoreId, // Visma store id
            "magento_store_id" => $magentoStoreId, // Magento store id
            "magento_store_code" => $order->getStore()->getCode(), // Magento store code
            "magento_order_id" => $magentoOrderId, // Magento order id
            "increment_id" => $order->getIncrementId(), // Order identifier
            "magento_customer" => null, // If logged in customer, set this later
            "magento_customer_id" => $order->getCustomerId(), // Magento customer id (0 = Not logged in / Guest)
            "magento_customer_group_id" => 0, // 0 = Not logged in, 1 = Default
            "order_currency_id" => $currencyId,
            "order_country_id" => $countryId,
            "customer_email" => $order->getCustomerEmail()
        ];

        // If customer was logged in, try to load customer
        if (!empty($orderData["magento_customer_id"])) {
            $orderData["magento_customer_group_id"] = 1; // Fallback: default group
            $customer = Mage::getModel("customer/customer")->load($orderData["magento_customer_id"]);
            if ($customer && $customer->getId() == $orderData["magento_customer_id"]) {
                $orderData["magento_customer"] = $customer;
                $orderData["magento_customer_group_id"] = $customer->getGroupId();
            }
        }

        // Check if we are supposed to override Visma order DB
        $override = Mage::helper($this->_defaultHelper)->getVismaOrderDBOverride($magentoStoreId, $currencyId, $countryId);
        if (!empty($override["visma_order_db"]) && $orderData["visma_order_db"] != $override["visma_order_db"]) {
            $orderData["visma_order_db"] = $override["visma_order_db"];
        }

        // Tell Visma to transfer order back to default order db if current order db is something else than default
        // But not if it only contains dropship products
        if ($orderData["visma_order_db"] != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {
            if (!Mage::helper($this->_defaultHelper)->isOnlyDropship($order)) {
                $orderData["xmlData"]["order_head"]["Ord_Gr5"] = 1;
            }
        }

        if (!$this->_addPaymentData($orderData, $order)) {
            return false;
        }

        if (!$this->_addAddressData($orderData, $order)) {
            return false;
        }

        if (!$this->_addPriceData($orderData, $order)) {
            return false;
        }

        if ($this->getDebugMode()) {
            Mage::log(print_r($orderData, true), LOG_DEBUG, "order-export-debug.log", true);
        }

        // We should now have all data needed to create xml

        if (!Mage::getStoreConfig("integration/orders/exclude_shipping", $orderData["magento_store_id"])) {
            if (!$this->_prepareShipment($orderData, $order)) {
                return false;
            }
        }

        if (!Mage::getStoreConfig("integration/orders/exclude_fees", $orderData["magento_store_id"])) {
            if (!$this->_prepareFees($orderData, $order)) {
                return false;
            }
        }

        if (!$this->_prepareDiscounts($orderData, $order)) {
            return false;
        }

        if (!$this->_prepareItems($orderData, $order)) {
            return false;
        }

        // Process local instance changes to export data here
        if (!$this->prepareLocalOrderUpdates($orderData, $order)) {
            return false;
        }

        // Expand some Visma structure products
        if (!$this->expandStruct($orderData, $order)) {
            return false;
        }

        return $orderData;
    }

    protected function _addPaymentData(&$orderData, $order)
    {

        $payment = $order->getPayment();

        switch ($payment->getMethod()) {
            case "DibsD2":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "DIBS";

                // Set transaction id (Ord_Inf3)
                try {
                    $transaction = Mage::helper("DibsD2")->getLastTransaction($payment);
                    $additionalInformation = $transaction->getAdditionalInformation();
                    $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation[Crossroads_DibsD2_Helper_Data::FIELD_TRANSACT] ?? "saknas") ?: "saknas";
                } catch (Exception $exception) {
                    // Just silently catch error
                    $orderData["xmlData"]["order_head"]["Ord_Inf3"] = "saknas";
                }
                break;

            case "Dibspw":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "DIBS";

                // Set transaction id (Ord_Inf3)
                $sqlQuery = "SELECT * FROM dibs_pw_results WHERE orderid = ?";
                $additionalInformation = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery, $orderData["increment_id"]);
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation["transaction"] ?? "saknas") ?: "saknas";
                break;

            case "Crossroads_Stripe_PaymentIntents":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Stripe";

                // Set transaction id (Ord_Inf3)
                $additionalInformation = $payment->getAdditionalInformation();
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation["intent_id"] ?? "saknas") ?: "saknas";
                break;

            case "Crossroads_Klarna_Invoice":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Klarna";

                // Set transaction id (Ord_Inf3)
                $sqlQuery = "SELECT additional_information FROM sales_flat_order_payment WHERE parent_id = ?";
                $additionalInformation = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $orderData["magento_order_id"]);
                if (!empty($additionalInformation)) {
                    $additionalInformation = unserialize($additionalInformation);
                }
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation[Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID] ?? "saknas") ?: "saknas";
                break;

            case "Crossroads_Collector":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Collector Bank";

                // Set transaction id (Ord_Inf3)
                $additionalInformation = $order->getPayment()->getAdditionalInformation();
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation[Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID] ?? "saknas") ?: "saknas";
                if (!empty($additionalInformation[Crossroads_Collector_Helper_Data::FIELD_USING_FALLBACK])) {
                    // We are using a fallback, add that info to the order export
                    $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Collector Bank (norisk)";
                }
                break;

            case "Crossroads_CollectorCheckout":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Walley";

                // Set transaction id (Ord_Inf3)
                $additionalInformation = $order->getPayment()->getAdditionalInformation();
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = ($additionalInformation[Crossroads_Collector_Helper_Data::FIELD_INVOICE_ID] ?? "saknas") ?: "saknas";
                break;

            case "altapay":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Altapay";

                // Set transaction id (Ord_Inf3), use publicPaymentId from additional_information
                $paymentId = $order->getPayment()->getAdditionalInformation("payment.gateway.publicPaymentId");
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = $paymentId ?: "saknas";
                break;

            case "crossroads_payex_masterpass":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Payex";

                // Set transaction id (Ord_Inf3), for payex: same as increment id
                $orderData["xmlData"]["order_head"]["Ord_Inf3"] = $orderData["xmlData"]["order_head"]["Ord_Inf"];
                break;

            case "free":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Gratis";
                break;

            case "customercredit":
            case "moviestars_points":
            case "crossroads_resurs":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Poäng";
                break;

            case "crossroads_vouchers_giftshop":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Klarna";
                break;

            case "crossroads_benifybio":
                // This payment method uses special customer id and different name for payment method
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Självfaktura";
                $orderData["xmlData"]["order_head"]["Ord_CustNo"] = "101000";
                break;

            case "julklappsvalet_giftshop":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Självfaktura";
                $orderData["xmlData"]["order_head"]["Ord_OrdPref"] = "8708"; // Samlingsfaktura
                break;

            case "awardit_token":
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Faktura";
                $orderData["xmlData"]["order_head"]["Ord_OrdPref"] = "8708"; // Samlingsfaktura
                break;

            case "checkmo":
                if (!Mage::getStoreConfig("payment/checkmo/active", $orderData["magento_store_id"])) {
                    Mage::helper($this->_defaultHelper)->log("Checkmo not active for this store!", LOG_ERR);
                    Mage::helper($this->_defaultHelper)->sendAdminEmail(
                        "Checkmo not active for this store: #{$orderData["magento_store_id"]}!",
                        print_r($orderData, true),
                        ["it"]
                    );
                    return false;
                }
                $orderData["xmlData"]["order_head"]["Ord_Inf4"] = "Faktura";
                $orderData["xmlData"]["order_head"]["Ord_OrdPref"] = "8708"; // Samlingsfaktura
                break;

            default:
                // Unknown payment method. skipping order!
                Mage::helper($this->_defaultHelper)->log("Unknown payment method: {$payment->getMethod()}", LOG_ERR);
                Mage::helper($this->_defaultHelper)->sendAdminEmail(
                    "Unknown payment method for order {$orderData["increment_id"]}",
                    print_r($orderData, true),
                    ["it"]
                );
                return false;
        }
        $orderData["amount_paid"] = $payment->getBaseAmountPaid() ?: $payment->getBaseAmountAuthorized();

        return true;
    }

    protected function _addAddressData(&$orderData, $order)
    {
        $returnStatus = true;
        $errorMessage = "";
        $billingAddressId = $order->getBillingAddressId();
        $shippingAddressId = $order->getShippingAddressId();

        if (empty($billingAddressId) && empty($shippingAddressId)) {
            // Allow order without shipping AND billing address? Don't think so!
            $errorMessage .= "Order is missing billing AND shipping address!\n";
            $returnStatus = false;
        } elseif (empty($billingAddressId) && !empty($shippingAddressId)) {
            // Free order get same billing address as shipping address.
            $shippingAddress = Mage::getModel("sales/order_address")->load($shippingAddressId);
            $billingAddress = $shippingAddress;
        } elseif (!empty($billingAddressId) && empty($shippingAddressId)) {
            // Virtual order get same shipping address as billing address.
            $billingAddress = Mage::getModel("sales/order_address")->load($billingAddressId);
            $shippingAddress = $billingAddress;
        } else {
            // Normal order with both billing and shipping address.
            $billingAddress = Mage::getModel("sales/order_address")->load($billingAddressId);
            $shippingAddress = Mage::getModel("sales/order_address")->load($shippingAddressId);
        }

        if ($returnStatus) {
            // Fetch phone numbers
            $bPhone = $billingAddress->getTelephone();
            $sPhone = $shippingAddress->getTelephone();
            if (empty($bPhone) && empty($sPhone)) {
                // Order is missing phonenumbers!
                $errorMessage .= "Order is missing phonenumbers!\n";
            } elseif (empty($bPhone) && !empty($sPhone)) {
                // Order is missing billing phonenumber, copy shipping phonenumber.
                $bPhone = $sPhone;
            } elseif (!empty($bPhone) && empty($sPhone)) {
                // Order is missing shipping phonenumber, copy billing phonenumber.
                $sPhone = $bPhone;
            }

            $customerEmail = $order->getCustomerEmail();
            if (empty($customerEmail) && !empty($orderData["price_data"]["contains_virtual"])) {
                // Order contains virtual products and is missing email address!
                $errorMessage .= "Order contains virtual products and is missing email address!\n";
            }

            $bCompany = $billingAddress->getCompany();
            $bFirstName = $billingAddress->getFirstname();
            $bLastName = $billingAddress->getLastname();
            $bName = "{$bFirstName} {$bLastName}";
            $bAddress = $billingAddress->getStreet();
            if (!is_array($bAddress)) { // Sometimes the street seems to be a string, not an array (should have 2 rows)
                $bAddress = [$bAddress];
            }
            $orderData["xmlData"]["order_head"]["Ord_Nm"] = (empty($bCompany) ? "" : "{$bCompany}, ") . $bName;

            // If special Visma customer id is used, we usually don't need to include invoice (billing) address.
            if (!Mage::getStoreConfig("integration/orders/exclude_invoice_address", $orderData["magento_store_id"])) {
                $orderData["xmlData"]["order_head"]["Ord_Ad1"] = $bAddress[0];
                $orderData["xmlData"]["order_head"]["Ord_Ad2"] = empty($bAddress[1]) ? "" : $bAddress[1];
                $orderData["xmlData"]["order_head"]["Ord_PNo"] = $billingAddress->getPostcode();
                $orderData["xmlData"]["order_head"]["Ord_PArea"] = $billingAddress->getCity();
                $orderData["xmlData"]["order_head"]["Ord_Ctry"] = Mage::helper($this->_defaultHelper)->getVismaCountryId($billingAddress->getCountryId());
            }

            $sCompany = $shippingAddress->getCompany();
            $sFirstName = $shippingAddress->getFirstname();
            $sLastName = $shippingAddress->getLastname();
            $sName = "{$sFirstName} {$sLastName}";
            $sAddress = $shippingAddress->getStreet();
            if (!is_array($sAddress)) { // Sometimes the street seems to be a string, not an array (should have 2 rows)
                $sAddress = [$sAddress];
            }
            $orderData["xmlData"]["order_head"]["Ord_DelNm"] = (empty($sCompany) ? "" : "{$sCompany}, ") . $sName;
            $orderData["xmlData"]["order_head"]["Ord_DelAd1"] = $sAddress[0];
            $orderData["xmlData"]["order_head"]["Ord_DelAd2"] = empty($sAddress[1]) ? "-" : $sAddress[1];
            $orderData["xmlData"]["order_head"]["Ord_DelPNo"] = $shippingAddress->getPostcode();
            $orderData["xmlData"]["order_head"]["Ord_DelPArea"] = $shippingAddress->getCity();
            $orderData["xmlData"]["order_head"]["Ord_DelCtry"] = Mage::helper($this->_defaultHelper)->getVismaCountryId($shippingAddress->getCountryId());
            $orderData["xmlData"]["order_head"]["Ord_Ad3"] = Mage::helper($this->_defaultHelper)->filterPhoneNumber($sPhone, $shippingAddress->getCountryId());
            $orderData["xmlData"]["order_head"]["Ord_Ad4"] = $order->getCustomerEmail();
        }

        // If we have errors, send email to admin about it
        if (!empty($errorMessage)) {
            $errorMessage .= print_r($orderData, true);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                "Order {$orderData["increment_id"]} have problems.",
                $errorMessage
                ["it"]
            );
        }

        return $returnStatus;
    }

    protected function _addPriceData(&$orderData, $order)
    {
        $items = $order->getItemsCollection()->getItems();

        // Initialize price data
        $orderData["price_data"] = [
            "items" => [],
            "taxes" => [
                "all" => [
                    "sum" => 0,
                    "qty" => 0,
                    "tax" => [], // [<tax%>] => [ "sum" => 0, "ratio" => 0 ]
                ],
                "physical" => [
                    "sum" => 0,
                    "qty" => 0,
                    "tax" => [], // [<tax%>] => [ "sum" => 0, "ratio" => 0 ]
                ],
                "virtual" => [
                    "sum" => 0,
                    "qty" => 0,
                    "tax" => [], // [<tax%>] => [ "sum" => 0, "ratio" => 0 ]
                ]
            ],
            "discounts" => [
                "rows" => [],
                "sum" => 0
            ],
            "shipping" => [],
            "fees" => [],
            "debug" => []
        ];
        $taxPart = [
            "sum" => 0,
            "ratio" => 0,
            "qty" => 0,
            "magento_tax_class_id" => 0,
            "visma_tax_class_id" => 0
        ];

        // Get item data
        $haveChildren = false;
        foreach ($items as $item) {
            $itemId = intval($item->getData("item_id"));
            $product = Mage::getModel("catalog/product")->setStoreId($order->getStoreId())->load($item->getProductId());
            $taxPercent = Mage::helper($this->_defaultHelper)->getTaxPercentFromMagentoTaxId($product->getTaxClassId(), $orderData["order_currency_id"], $orderData["order_country_id"]);
            $parentItemId = $item->getData("parent_item_id");
            $qty = intval($item->getQtyOrdered());
            $sku = $item->getSku();

            // Expand "fake" bundles. SKU is formatted as <sku>x<qty>, i.e. 21092x3
            // Only expand simple and virtual products NOT in a bundle or config.
            if (in_array($item->getProductType(), [ Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL, Mage_Catalog_Model_Product_Type::TYPE_SIMPLE ]) && empty($parentItemId)) {
                $matches = [];
                if (preg_match('/^([0-9]+)x([0-9]+)$/i', $sku, $matches)) {
                    if (!empty($matches[1]) && !empty($matches[2])) {
                        $sku = $matches[1];
                        $qty = $qty * (intval($matches[2]) ?: 1);
                    }
                }
            }

            $orderData["price_data"]["items"][$itemId]["product_id"] = $item->getData("product_id");
            $orderData["price_data"]["items"][$itemId]["sku"] = $sku;
            $orderData["price_data"]["items"][$itemId]["name"] = $product->getName();
            $orderData["price_data"]["items"][$itemId]["type"] = $item->getProductType();
            $orderData["price_data"]["items"][$itemId]["parent"] = $parentItemId;
            $orderData["price_data"]["items"][$itemId]["qty"] = $qty;
            $orderData["price_data"]["items"][$itemId]["tax_percent"] = $taxPercent;
            $orderData["price_data"]["items"][$itemId]["tax_factor"] = (100 + $taxPercent) / 100;
            $orderData["price_data"]["items"][$itemId]["magento_tax_class_id"] = $product->getTaxClassId();
            $orderData["price_data"]["items"][$itemId]["visma_tax_class_id"] = Mage::helper($this->_defaultHelper)->getVismaTaxIdFromMagentoTaxId($product->getTaxClassId(), $orderData["order_currency_id"], $orderData["order_country_id"]);
            $orderData["price_data"]["items"][$itemId]["original_price"] = $item->getData("base_original_price") * $orderData["price_data"]["items"][$itemId]["qty"];
            $orderData["price_data"]["items"][$itemId]["row_total_money"] = $item->getBaseRowTotal() + $item->getBaseTaxAmount() + $item->getBaseHiddenTaxAmount();
            $orderData["price_data"]["items"][$itemId]["discount"] = $item->getBaseDiscountAmount();
            $orderData["price_data"]["discounts"]["sum"] += $orderData["price_data"]["items"][$itemId]["discount"];

            // Add product data specific to this Magento instance
            $this->getLocalProductExtraData($item, $product, $orderData);

            // If item have parent, calculate child data
            $parentId = intval($item->getData("parent_item_id"));
            if ($parentId > 0) {
                $haveChildren = true;
                if (empty($orderData["price_data"]["items"][$parentId]["children"])) {
                    $orderData["price_data"]["items"][$parentId]["children"] = [];
                    $orderData["price_data"]["items"][$parentId]["child_qty"] = 0;
                    $orderData["price_data"]["items"][$parentId]["child_money_sum"] = 0;
                }
                $orderData["price_data"]["items"][$parentId]["children"][] = $itemId;
                $orderData["price_data"]["items"][$parentId]["child_qty"] += $item->getQtyOrdered();
                $orderData["price_data"]["items"][$parentId]["child_money_sum"] += ($item->getQtyOrdered() * $orderData["price_data"]["items"][$itemId]["original_price"]);
            }

        }

        // If there are children, calculate prices for them and remove parent
        if ($haveChildren) {
            foreach (array_keys($orderData["price_data"]["items"]) as $itemId) {
                if (!empty($orderData["price_data"]["items"][$itemId]["children"])) {
                    $childQty = count($orderData["price_data"]["items"][$itemId]["children"]);
                    foreach ($orderData["price_data"]["items"][$itemId]["children"] as $childId) {

                        if ($childQty == 1) {
                            // Need to copy original price from parent
                            $orderData["price_data"]["items"][$childId]["original_price"] = $orderData["price_data"]["items"][$itemId]["original_price"];
                        } else {
                            // Bundle with more than 1 item, just fake it and divide by child qty
                            // TODO: Get correct prices some other way
                            $orderData["price_data"]["items"][$childId]["original_price"] = $orderData["price_data"]["items"][$itemId]["original_price"] / $childQty;
                        }

                        // If sum of child value > 0, use price to calculate ratio
                        if ($orderData["price_data"]["items"][$itemId]["child_money_sum"] > 0) {
                            $orderData["price_data"]["items"][$childId]["child_rate"] = $orderData["price_data"]["items"][$childId]["original_price"] / $orderData["price_data"]["items"][$itemId]["child_money_sum"];

                        // If no sum > 0 was found, just calculate ratio based on number of children
                        } else {
                            $orderData["price_data"]["items"][$childId]["child_rate"] = $orderData["price_data"]["items"][$childId]["qty"] / $orderData["price_data"]["items"][$itemId]["child_qty"];
                        }

                        $orderData["price_data"]["items"][$childId]["row_total_money"] = $orderData["price_data"]["items"][$itemId]["row_total_money"] * $orderData["price_data"]["items"][$childId]["child_rate"];
                        $orderData["price_data"]["items"][$childId]["discount"] = $orderData["price_data"]["items"][$itemId]["discount"] * $orderData["price_data"]["items"][$childId]["child_rate"];
                    }

                    // Remove parent
                    unset($orderData["price_data"]["items"][$itemId]);
                }
            }
        }

        foreach (array_keys($orderData["price_data"]["items"]) as $itemId) {

            $taxPercent = $orderData["price_data"]["items"][$itemId]["tax_percent"];

            // Calculate tax totals (all products)
            $orderData["price_data"]["taxes"]["all"]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
            $orderData["price_data"]["taxes"]["all"]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];
            if (!array_key_exists($taxPercent, $orderData["price_data"]["taxes"]["all"]["tax"])) {
                $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent] = $taxPart;
                $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["magento_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["magento_tax_class_id"];
                $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["visma_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["visma_tax_class_id"];
            }
            $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
            $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];

            if ($orderData["price_data"]["items"][$itemId]["type"] == Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL) {

                // Calculate tax totals (virtual products)
                $orderData["price_data"]["taxes"]["virtual"]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
                $orderData["price_data"]["taxes"]["virtual"]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];
                if (!array_key_exists($taxPercent, $orderData["price_data"]["taxes"]["virtual"]["tax"])) {
                    $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent] = $taxPart;
                    $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["magento_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["magento_tax_class_id"];
                    $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["visma_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["visma_tax_class_id"];
                }
                $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
                $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];

            } else {

                // Calculate tax totals (physical products)
                $orderData["price_data"]["taxes"]["physical"]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
                $orderData["price_data"]["taxes"]["physical"]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];
                if (!array_key_exists($taxPercent, $orderData["price_data"]["taxes"]["physical"]["tax"])) {
                    $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent] = $taxPart;
                    $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["magento_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["magento_tax_class_id"];
                    $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["visma_tax_class_id"] = $orderData["price_data"]["items"][$itemId]["visma_tax_class_id"];
                }
                $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["sum"] += $orderData["price_data"]["items"][$itemId]["original_price"];
                $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["qty"] += $orderData["price_data"]["items"][$itemId]["qty"];

            }
        }

        // Calculate tax ratios (all products)
        foreach (array_keys($orderData["price_data"]["taxes"]["all"]["tax"]) as $taxPercent) {
            if ($orderData["price_data"]["taxes"]["all"]["sum"] > 0) {
                $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["ratio"] = $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["sum"] / $orderData["price_data"]["taxes"]["all"]["sum"];
            }
        }

        // Calculate tax ratios (virtual products)
        foreach (array_keys($orderData["price_data"]["taxes"]["virtual"]["tax"]) as $taxPercent) {
            if ($orderData["price_data"]["taxes"]["virtual"]["sum"] > 0) {
                $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["ratio"] = $orderData["price_data"]["taxes"]["virtual"]["tax"][$taxPercent]["sum"] / $orderData["price_data"]["taxes"]["virtual"]["sum"];
            }
        }

        // Calculate tax ratios (physical products)
        foreach (array_keys($orderData["price_data"]["taxes"]["physical"]["tax"]) as $taxPercent) {
            if ($orderData["price_data"]["taxes"]["physical"]["sum"] > 0) {
                $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"] = $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["sum"] / $orderData["price_data"]["taxes"]["physical"]["sum"];
            }
        }

        // Calculate and add Retain24 voucher as a discount
        if ($order->getData("retain24_amount") != 0) {
            $orderData["price_data"]["discounts"]["rows"][] = [
                "product_id" => null,
                "sku" => "RabattZupergift",
                "name" => "Zupergift Presentkort",
                "qty" => 1,
                "tax_factor" => 1,
                "tax_percent" => 0,
                "magento_tax_class_id" => 0,
                "visma_tax_class_id" => Mage::helper($this->_defaultHelper)->getVismaTaxIdFromMagentoTaxId(0, $orderData["order_currency_id"], $orderData["order_country_id"]),
                "row_total_money" => $order->getData("retain24_amount"),
                "original_price" => $order->getBaseRetain24Amount()
            ];

            // Use voucher code as discount id
            $orderData["xmlData"]["order_head"]["Ord_Inf5"] = $order->getRetain24Code();
        }

        // Calculate and add shipping
        if ($order->getData("shipping_incl_tax") > 0) {
            $originalMoney = $order->getData("shipping_incl_tax");
            $moneyPaid = $order->getData("base_shipping_incl_tax");
            $orderData["price_data"]["discounts"]["sum"] += $order->getBaseShippingDiscountAmount(); // TODO: Check if we need to add $order->getBaseShippingHiddenTaxAmnt()?

            if ($orderData["visma_order_db"] == Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {

                // If order have Sweden as Visma company db, distribute shipping fee propotionally for each tax used for physical products
                foreach (array_keys($orderData["price_data"]["taxes"]["physical"]["tax"]) as $taxPercent) {
                    $orderData["price_data"]["shipping"][$taxPercent] = [
                        "product_id" => null,
                        "sku" => "Frakt",
                        "name" => $order->getShippingDescription() . " ({$taxPercent}% moms)",
                        "qty" => 1,
                        "tax_factor" => (100 + $taxPercent) / 100,
                        "tax_percent" => $taxPercent,
                        "magento_tax_class_id" => $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["magento_tax_class_id"],
                        "visma_tax_class_id" => $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["visma_tax_class_id"],
                        "original_price" =>  $originalMoney * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"],
                        "row_total_money" => $moneyPaid * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"],
                        "discount" => $order->getBaseShippingDiscountAmount() * $orderData["price_data"]["taxes"]["physical"]["tax"][$taxPercent]["ratio"],
                    ];
                }

            } else {

                // If order does NOT have Sweden as Visma company db, use max tax for shipping fee
                $taxData = Mage::helper($this->_defaultHelper)->getMaxTaxForCurrencyAndCountry($orderData["order_currency_id"], $orderData["order_country_id"]);
                $orderData["price_data"]["shipping"][ $taxData["tax_percent"] ] = [
                    "product_id" => null,
                    "sku" => "Frakt",
                    "name" => $order->getShippingDescription(),
                    "qty" => 1,
                    "tax_factor" => (100 + $taxData["tax_percent"]) / 100,
                    "tax_percent" => $taxData["tax_percent"],
                    "magento_tax_class_id" => $taxData["magento_tax_id"],
                    "original_price" =>  $originalMoney,
                    "visma_tax_class_id" => $taxData["local_visma_tax_id"],
                    "row_total_money" => $moneyPaid,
                    "discount" => $order->getBaseShippingDiscountAmount(),
                ];
            }
        }

        // Calculate and add payment fee
        if ($order->getData("base_payment_fee_amount") > 0) {
            $moneyPaid = $order->getData("base_payment_fee_amount") + $order->getData("base_payment_fee_tax_amount");
            foreach (array_keys($orderData["price_data"]["taxes"]["all"]["tax"]) as $taxPercent) {
                $orderData["price_data"]["fees"][$taxPercent] = [
                    "product_id" => null,
                    "sku" => "Admin",
                    "name" => $order->getData("payment_fee_title") . " ({$taxPercent}% moms)",
                    "qty" => 1,
                    "tax_factor" => (100 + $taxPercent) / 100,
                    "tax_percent" => $taxPercent,
                    "magento_tax_class_id" => $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["magento_tax_class_id"],
                    "visma_tax_class_id" => $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["visma_tax_class_id"],
                    "row_total_money" => $moneyPaid * $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["ratio"],
                ];
            }
        }

        // Calculate and add discount
        if ($order->getDiscountAmount() < 0) {
            foreach (array_keys($orderData["price_data"]["taxes"]["all"]["tax"]) as $taxPercent) {
                $adjustedValue = $order->getDiscountAmount() * ($orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["ratio"] ?: 1);
                $vismaTaxClassId = $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["visma_tax_class_id"];

                // Set special Visma tax class id for discounts when tax is 0% and in company Awardit CLS AB
                if ($taxPercent == 0 && $orderData["visma_order_db"] == "F0001") {
                    $vismaTaxClassId = 29;
                }

                $orderData["price_data"]["discounts"]["rows"][] = [
                    "product_id" => null,
                    "sku" => "Rabatt",
                    "name" => "Rabatt ({$taxPercent}% moms)",
                    "qty" => 1,
                    "tax_factor" => (100 + $taxPercent) / 100,
                    "tax_percent" => $taxPercent,
                    "magento_tax_class_id" => $orderData["price_data"]["taxes"]["all"]["tax"][$taxPercent]["magento_tax_class_id"],
                    "visma_tax_class_id" => $vismaTaxClassId,
                    "row_total_money" => $adjustedValue,
                    "original_price" => $adjustedValue,
                ];
            }
        }

        return true;
    }

    protected function _prepareShipment(&$orderData, $order)
    {
        if (empty($orderData["price_data"]["shipping"])) {
            return true;
        }

        // Check if we got override value for transGr2
        if (!empty($orderData["overrides"]["OrdLn_TransGr2"])) {
            // Yes we do, set it
            $transGr2 = $orderData["overrides"]["OrdLn_TransGr2"];
        } else {
            // No we don't, just go with defaults
            switch ($order->getShippingMethod()) {
                case "freeshipping_freeshipping":
                case "flatrate_flatrate":
                case "tablerate_bestway":
                case "amtable_amtable3":
                case "amtable_amtable5":
                case "amtable_amtable10":
                    $transGr2 = 7; // Schenker
                    break;

                case "amtable_amtable7":
                    $transGr2 = 8; // Schenker
                    break;

                case "amtable_amtable4":
                case "amtable_amtable8":
                case "amtable_amtable9":
                    $transGr2 = 9; // Bring Pickup-parcel
                    break;

                case "amtable_amtable6":
                case "amtable_amtable11":
                    $transGr2 = 11; // UPS Express Saver
                    break;

                default:
                    // This is an unknown shipping method!
                    Mage::helper($this->_defaultHelper)->log("Unknown shipping method for {$orderData["increment_id"]}: '{$order->getShippingMethod()}'.", LOG_ERR);
                    return false;
            }
        }

        foreach ($orderData["price_data"]["shipping"] as $index => $shipping) {
            $item = [
                "OrdLn_TransGr2" => $transGr2,
                "OrdLn_ProdNo"   => $shipping["sku"],
                "OrdLn_Descr"    => $shipping["name"],
                "OrdLn_Price"    => $shipping["original_price"] / $shipping["tax_factor"],
                "OrdLn_VatRt"    => $shipping["tax_percent"],
                "OrdLn_VatNo"    => $shipping["visma_tax_class_id"],
                "OrdLn_ExRt"     => $orderData["xmlData"]["order_head"]["Ord_ExRt"],
                "OrdLn_NoInvoAb" => $shipping["qty"],
                "OrdLn_ProdTp"   => 1,
            ];

            if ($orderData["amount_paid"] > 0 && $shipping["row_total_money"] > 0) {
                $item["OrdLn_Free3"] = $shipping["row_total_money"];
            }

            $orderData["xmlData"]["shipping"][$index] = $item;
        }

        return true;
    }

    protected function _prepareFees(&$orderData, $order)
    {
        if (empty($orderData["price_data"]["fees"])) {
            return true;
        }

        foreach ($orderData["price_data"]["fees"] as $index => $fee) {
            $item = [
                "OrdLn_ProdNo"   => $fee["sku"],
                "OrdLn_Descr"    => $fee["name"],
                "OrdLn_Price"    => $fee["row_total_money"] / $fee["tax_factor"],
                "OrdLn_VatRt"    => $fee["tax_percent"],
                "OrdLn_VatNo"    => $fee["visma_tax_class_id"],
                "OrdLn_ExRt"     => $orderData["xmlData"]["order_head"]["Ord_ExRt"],
                "OrdLn_NoInvoAb" => $fee["qty"],
                "OrdLn_ProdTp"   => 1,
            ];

            if ($orderData["amount_paid"] > 0 && $fee["row_total_money"] > 0) {
                $item["OrdLn_Free3"] = $fee["row_total_money"];
            }

            $orderData["xmlData"]["fees"][$index] = $item;
        }

        return true;
    }

    protected function _prepareDiscounts(&$orderData, $order)
    {
        if (!empty($orderData["price_data"]["discounts"]["rows"])) {

            foreach ($orderData["price_data"]["discounts"]["rows"] as $index => $discount) {
                $orderData["xmlData"]["discount"][$index] = [
                    "OrdLn_ProdNo"   => $discount["sku"],
                    "OrdLn_Descr"    => $discount["name"],
                    "OrdLn_Price"    => $discount["original_price"] / $discount["tax_factor"],
                    "OrdLn_VatRt"    => $discount["tax_percent"],
                    "OrdLn_VatNo"    => $discount["visma_tax_class_id"],
                    "OrdLn_NoInvoAb" => $discount["qty"],
                    "OrdLn_ProdTp"   => 1,
                ];

                if ($discount["sku"] == "RabattZupergift") {
                    $orderData["xmlData"]["discount"][$index]["OrdLn_ProdTp"] = 2;
                }

                if ($orderData["amount_paid"] > 0 && $discount["row_total_money"] < 0) {
                    $orderData["xmlData"]["discount"][$index]["OrdLn_Free3"] = $discount["row_total_money"];
                }
            }
        }

        return true;
    }

    protected function _prepareItems(&$orderData, $order)
    {
        foreach ($orderData["price_data"]["items"] as $index => $item) {

            $itemData = [
                "OrdLn_ProdNo" => $item["sku"],
                "OrdLn_R10" => $item["sku"],
                "OrdLn_NoInvoAb" => $item["qty"],
                "OrdLn_Price" => ($item["original_price"] / $item["tax_factor"]) / ($item["qty"] ?: 1),
                "OrdLn_VatRt" => $item["tax_percent"],
                "OrdLn_VatNo" => $item["visma_tax_class_id"],
                "OrdLn_ExRt" => $orderData["xmlData"]["order_head"]["Ord_ExRt"]
            ];

            if ($orderData["visma_order_db"] != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {
                 // Vissa produkter skall INTE undantas lagerhantering i andra bolag än Sverige
                if (!Mage::helper($this->_defaultHelper)->isExceptionToException($itemData["OrdLn_ProdNo"])) {
                    $itemData["OrdLn_ExcPrint"] = 16384; // Undantas lagerhantering i andra bolag än Sverige
                }
            }

            if ($orderData["amount_paid"] > 0 && $item["row_total_money"] > 0) {
                $itemData["OrdLn_Free3"] = $item["row_total_money"];
            }

            $orderData["xmlData"]["items"][$index] = $itemData;
        }

        return true;
    }

    // Expand (some) products with SKU ending in "-strukt".
    public function expandStruct(&$orderData, $order)
    {
        foreach (array_keys($orderData["xmlData"]["items"]) as $rowIndex) {
            if ($this->isStructToExpand($orderData["xmlData"]["items"][$rowIndex]["OrdLn_ProdNo"])) {
                $structChildren = $this->getStructChildren(
                    $orderData["xmlData"]["items"][$rowIndex]["OrdLn_ProdNo"],
                    $orderData["xmlData"]["order_head"]["Ord_R1"],
                    $orderData["xmlData"]["order_head"]["Ord_Cur"]
                );
                if (!empty($structChildren)) {
                    $orderData["struct_items"][$rowIndex] = [
                        "visma_children" => $structChildren,
                        "order_children" => [],
                        "sum" => 0,
                    ];
                }
            }
        }

        if (empty($orderData["struct_items"])) {
            return true;
        }

        $orderCustomerGroupId = $order->getCustomerGroupId();
        $vismaCustomerGroupId = null;
        if (!empty($orderCustomerGroupId) && $orderCustomerGroupId > 1) {
            $vismaCustomerGroupId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne("SELECT visma_group_id FROM customer_group WHERE customer_group_id = ?", $orderCustomerGroupId);
        }

        foreach (array_keys($orderData["struct_items"]) as $rowIndex) {

            // Prepare for calculation of ratio for price

            // Loop children to calculate prices.
            foreach (array_keys($orderData["struct_items"][$rowIndex]["visma_children"]) as $sku) {
                // Prices from table is excluding VAT.
                // Prices for calculations needs to be including VAT

                // Get VAT percentage for child.
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"] = Mage::helper($this->_defaultHelper)->getTaxPercentFromVismaTaxId($orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_id"], $orderData["order_currency_id"], $orderData["order_country_id"]);
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"] = (100 + $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"]) / 100;

                // Get price for 1 unit of this child
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"] * ((!empty($vismaCustomerGroupId) && array_key_exists($vismaCustomerGroupId, $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"])) ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"][$vismaCustomerGroupId] : $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["prices"][0]);

                // Get total price for all units of this child
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["total"] = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["per_unit"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];

                // Add to total price for all units of all children
                $orderData["struct_items"][$rowIndex]["sum"] += $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["total"];
            }

            // Loop children again to create new item(s) for order
            foreach (array_keys($orderData["struct_items"][$rowIndex]["visma_children"]) as $sku) {

                // NoInvoAb is total units therefore we must multiply units of parent with units of child.
                $totalQty = $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"] * $orderData["xmlData"]["items"][$rowIndex]["OrdLn_NoInvoAb"];

                // Calculate ratio by dividing (total child price) by (total price of all children).
                $ratio = $orderData["struct_items"][$rowIndex]["sum"] > 0 ? $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["total"] / $orderData["struct_items"][$rowIndex]["sum"] : 1;
                $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["ratio"] = $ratio;

                // Calculation of price must be done including VAT, to get price correct if child have different VAT than parent.
                $totalPriceInc = $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Price"] * ((100 + $orderData["xmlData"]["items"][$rowIndex]["OrdLn_VatRt"]) / 100);

                // Multiply by ratio to get correct price for this child
                $priceInc = $totalPriceInc * $ratio;

                // Calculate price excluding VAT using factor from child
                $priceEx = $priceInc / $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_factor"];

                // Price from parent is per unit of parent. We need to divide by number of units per child.
                $price = $priceEx / $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["qty"];

                $newItem = [
                    "OrdLn_ProdNo" => $sku,
                    "OrdLn_NoInvoAb" => $totalQty,
                    "OrdLn_Price" => $price, // Price per unit !
                    "OrdLn_VatRt" => $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_percent"],
                    "OrdLn_VatNo" => $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["tax_id"],
                    "OrdLn_ExRt" => $orderData["xmlData"]["order_head"]["Ord_ExRt"],
                ];

                // If needed, split cash paid from parent to child
                if (array_key_exists("OrdLn_Free3", $orderData["xmlData"]["items"][$rowIndex]) && $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"] > 0) {
                    $newItem["OrdLn_Free3"] = $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"] * $orderData["struct_items"][$rowIndex]["visma_children"][$sku]["ratio"];
                }

                if ($orderData["visma_order_db"] != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {
                     // Vissa produkter skall INTE undantas lagerhantering i andra bolag än Sverige
                    if (!Mage::helper($this->_defaultHelper)->isExceptionToException($newItem["OrdLn_ProdNo"])) {
                        $newItem["OrdLn_ExcPrint"] = 16384; // Undantas lagerhantering i andra bolag Ã¤n Sverige
                    }
                }

                // Add child to order items
                $orderData["struct_items"][$rowIndex]["order_children"][$sku] = $newItem;
            }

            // Reset values in parent
            $orderData["xmlData"]["items"][$rowIndex]["OrdLn_Price"] = 0;
            if (array_key_exists("OrdLn_Free3", $orderData["xmlData"]["items"][$rowIndex])) {
                unset($orderData["xmlData"]["items"][$rowIndex]["OrdLn_Free3"]);
            }
        }

        return true;
    }

    public function isStructToExpand($sku)
    {
        if (stripos($sku, "strukt") > 0) {
            $ExpStr = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchOne("SELECT Prod.ExpStr FROM Prod WHERE Prod.ProdNo = ?", $sku);
            if ($ExpStr !== null && (intval($ExpStr) & 1) === 0) {
                return true;
            }
        }
        return false;
    }

    public function getStructChildren($prodNo, $shopId, $currency)
    {
        $childProducts = [];
        $sqlQuery = "
            SELECT
                Struct.ProdNo,
                Struct.SubProd,
                Struct.NoPerStr AS qty,
                Prod.ProdTp AS product_type,
                PrDcMat.VatNo AS tax_id,
                PrDcMat.SugPr AS suggested_price,
                PrDcMat.Cur AS currency_code,
                PrDcMat.R1 AS shop_id,
                PrDcMat.SalePr AS price,
                PrDcMat.CustPrG2 AS customer_group_id,
                PrDcMat.FrDt AS from_date,
                PrDcMat.ToDt AS to_date
            FROM Struct
            JOIN Prod ON Prod.ProdNo = Struct.SubProd
            LEFT JOIN PrDcMat ON PrDcMat.ProdNo = Struct.SubProd AND PrDcMat.Cur = :currency AND (PrDcMat.R1 = :shopId OR (PrDcMat.R1 = 0 AND PrDcMat.VatNo > 0))
            WHERE Struct.ProdNo = :prodNo
            ORDER BY Struct.ProdNo, Struct.SubProd, PrDcMat.R1, PrDcMat.CustPrG2
        ";
        $params = [
            "prodNo" => $prodNo,
            "shopId" => $shopId,
            "currency" => $currency
        ];
        $result = Mage::helper($this->_defaultHelper)->getVismaDB()->fetchAll($sqlQuery, $params);
        $containsDatePrices = false;
        foreach ($result as $priceRow) {
            if ($priceRow["tax_id"] > 0) {
                $childProducts[$priceRow["SubProd"]]["tax_id"] = $priceRow["tax_id"];
                $childProducts[$priceRow["SubProd"]]["product_type"] = $priceRow["product_type"];
                $childProducts[$priceRow["SubProd"]]["qty"] = $priceRow["qty"];
                continue; // row with tax_id does not have any prices
            }
            if (empty($priceRow["from_date"]) && empty($priceRow["to_date"])) {
                $childProducts[$priceRow["SubProd"]]["prices"][$priceRow["customer_group_id"]] = $priceRow["price"];
            } else {
                $containsDatePrices = true;
            }
        }
        if ($containsDatePrices) {
            foreach ($result as $priceRow) {
                if (!empty($priceRow["from_date"]) || !empty($priceRow["to_date"])) {
                    if ($this->getTimedPricesModel()->checkTimedDates($priceRow["from_date"], $priceRow["to_date"]) === true) {
                        $childProducts[$priceRow["SubProd"]]["prices"][$priceRow["customer_group_id"]] = $priceRow["price"];
                    }
                }
            }
        }

        return $childProducts;
    }

    public function fixRoundingError(&$orderData)
    {
        // Last minute rounding of OrdLn_Free3 based on money paid.
        if (empty($orderData["amount_paid"]) || !($orderData["amount_paid"] > 0)) {
            return;
        }

        // Try to find highest amount of money used
        $orderSum = 0.0;
        $maxValue = 0.0;
        $maxItemIndex = null;
        $maxChildIndex = null;
        $maxShippingIndex = null;
        $maxFeeIndex = null;

        // Search items and child items first
        foreach ($orderData["xmlData"]["items"] as $rowIndex => $item) {
            if (!empty($item["OrdLn_Free3"]) && $item["OrdLn_Free3"] > 0) {
                $item["OrdLn_Free3"] = round($item["OrdLn_Free3"], 2);
                $orderSum += $item["OrdLn_Free3"];
                if ($item["OrdLn_Free3"] > $maxValue) {
                    $maxValue = $item["OrdLn_Free3"];
                    $maxItemIndex = $rowIndex;
                    $maxChildIndex = null;
                }
            }
            if (!empty($orderData["struct_items"][$rowIndex]["order_children"])) {
                foreach ($orderData["struct_items"][$rowIndex]["order_children"] as $sku => $childItem) {
                    if (!empty($childItem["OrdLn_Free3"]) && $childItem["OrdLn_Free3"] > 0) {
                        $childItem["OrdLn_Free3"] = round($childItem["OrdLn_Free3"], 2);
                        $orderSum += $childItem["OrdLn_Free3"];
                        if ($childItem["OrdLn_Free3"] > $maxValue) {
                            $maxValue = $childItem["OrdLn_Free3"];
                            $maxItemIndex = $rowIndex;
                            $maxChildIndex = $sku;
                        }
                    }
                }
            }
        }

        // Then search shipping
        foreach ($orderData["xmlData"]["shipping"] as $index => $shipping) {
            if (!empty($shipping["OrdLn_Free3"]) && $shipping["OrdLn_Free3"] > 0) {
                $shipping["OrdLn_Free3"] = round($shipping["OrdLn_Free3"], 2);
                $orderSum += $shipping["OrdLn_Free3"];
                if ($shipping["OrdLn_Free3"] > $maxValue) {
                    $maxValue = $shipping["OrdLn_Free3"];
                    $maxItemIndex = null;
                    $maxChildIndex = null;
                    $maxShippingIndex = $index;
                }
            }
        }

        // And finally search fees
        foreach ($orderData["xmlData"]["fees"] as $index => $fee) {
            if (!empty($fee["OrdLn_Free3"]) && $fee["OrdLn_Free3"] > 0) {
                $fee["OrdLn_Free3"] = round($fee["OrdLn_Free3"], 2);
                $orderSum += $fee["OrdLn_Free3"];
                if ($fee["OrdLn_Free3"] > $maxValue) {
                    $maxValue = $fee["OrdLn_Free3"];
                    $maxItemIndex = null;
                    $maxChildIndex = null;
                    $maxShippingIndex = null;
                    $maxFeeIndex = $index;
                }
            }
        }

        // Convert to öre/cent/whatever/least-monetary-unit and use that when comparing
        $amountPaid = intval(round($orderData["amount_paid"], 2) * 100);
        $orderSum = intval(round($orderSum, 2) * 100);
        if ($orderSum != $amountPaid) {
            $diff = ($orderSum - $amountPaid) / 100; // diff = amount missing or amount too much on order sum
            if (!empty($maxItemIndex)) {
                if (!empty($maxChildIndex)) {
                    Mage::helper($this->_defaultHelper)->log("Rounding error {$diff}, correcting on child item [{$orderData["struct_items"][$maxItemIndex]["order_children"][$maxChildIndex]["OrdLn_ProdNo"]}]");
                    $orderData["struct_items"][$maxItemIndex]["order_children"][$maxChildIndex]["OrdLn_Free3"] -= $diff;
                } else {
                    Mage::helper($this->_defaultHelper)->log("Rounding error {$diff}, correcting on item [{$orderData["xmlData"]["items"][$maxItemIndex]["OrdLn_ProdNo"]}]");
                    $orderData["xmlData"]["items"][$maxItemIndex]["OrdLn_Free3"] -= $diff;
                }
            } elseif (!empty($maxShippingIndex)) {
                Mage::helper($this->_defaultHelper)->log("Rounding error {$diff}, correcting on shipping \"{$orderData["xmlData"]["shipping"][$maxShippingIndex]["OrdLn_Descr"]}]\"");
                $orderData["xmlData"]["shipping"][$maxShippingIndex]["OrdLn_Free3"] -= $diff;
            } elseif (!empty($maxFeeIndex)) {
                Mage::helper($this->_defaultHelper)->log("Rounding error {$diff}, correcting on fee \"{$orderData["xmlData"]["fees"][$maxFeeIndex]["OrdLn_Descr"]}]\"");
                $orderData["xmlData"]["fees"][$maxFeeIndex]["OrdLn_Free3"] -= $diff;
            } else {
                Mage::helper($this->_defaultHelper)->log("Rounding error {$diff}, found nowhere to place correction", LOG_ERR);
            }
        }

    }

    public function createXmlData($orderData)
    {
        $XMLOrder = new SimpleXMLExtended('<?xml version="1.0" encoding="windows-1252"?><ORDERS format_xml="2.0"></ORDERS>');

        $XMLOrder->addArray("OrderHead", $orderData["xmlData"]["order_head"]);

        $this->fixRoundingError($orderData);

        foreach ($orderData["xmlData"]["items"] as $rowIndex => $item) {
            $XMLOrder->addArray("OrderLine", $item);
            if (!empty($orderData["struct_items"][$rowIndex]["order_children"])) {
                foreach ($orderData["struct_items"][$rowIndex]["order_children"] as $childItem) {
                    $XMLOrder->addArray("OrderLine", $childItem);
                }
            }
        }

        if (!empty($orderData["xmlData"]["fees"])) {
            foreach ($orderData["xmlData"]["fees"] as $item) {
                $XMLOrder->addArray("OrderLine", $item);
            }
        }

        if (!empty($orderData["xmlData"]["shipping"])) {
            foreach ($orderData["xmlData"]["shipping"] as $shipping) {
                $XMLOrder->addArray("OrderLine", $shipping);
            }
        }

        if (!empty($orderData["xmlData"]["discount"])) {
            foreach ($orderData["xmlData"]["discount"] as $discount) {
                $XMLOrder->addArray("OrderLine", $discount);
            }
        }

        return $XMLOrder;
    }

    public function handleBlacklist($orderData)
    {
        $isEmailBlacklisted = Mage::helper($this->_defaultHelper)->isEmailBlacklisted($orderData["customer_email"], $orderData["magento_store_id"]);

        if ($isEmailBlacklisted) {
            $exportStatus = self::EXPORT_STATUS_BLACKLIST;
        } elseif ($this->getTestMode()) {
            $exportStatus = self::EXPORT_STATUS_TEST;
        } else {
            $exportStatus = self::EXPORT_STATUS_PREPARED;
        }

        // Order contains blacklisted email address
        if ($isEmailBlacklisted) {
            $order = Mage::getModel("sales/order")->load($orderData["magento_order_id"]);
            $msg = "Order {$order->getIncrementId()} was put on hold due to blacklisted customer email address.";
            $order->setState(Mage_Sales_Model_Order::STATE_HOLDED, true, $msg);
            $order->save();
            Mage::helper($this->_defaultHelper)->log($msg);
            Mage::helper($this->_defaultHelper)->sendAdminEmail(
                $msg,
                "Matching email address: '{$orderData["customer_email"]}'\n",
                ["it"]
            );
        }

        return $exportStatus;
    }

    public function prepareLocalOrderUpdates(&$orderData, $order)
    {
        return true;
    }

    public function updateSingleOrder($integrationOrder)
    {
        $this->_wasLastOrderUpdated = false;

        if (empty($integrationOrder["id"])) {
            return false;
        }

        $incrementId = $integrationOrder["increment_id"];
        $makeCompleteInMagento = false;
        $cancelInMagento = false;

        // Fetch Magento data to se if order changed.
        $newMagentoData = $this->getMagentoOrderData($integrationOrder["magento_order_id"]);
        if (empty($newMagentoData)) {
            Mage::helper($this->_defaultHelper)->log("Unable to get data from Magento for order {$incrementId}", LOG_ERR);
            return false;
        }

        do {
            // If Magento status has changed, add this to things to update
            if ($newMagentoData["status"] != $integrationOrder["magento_status"]) {
                $integrationOrder["magento_status"] = $newMagentoData["status"];
            }
            if ($integrationOrder["magento_status"] == "canceled") {
                $integrationOrder["export_status"] = self::EXPORT_STATUS_ERROR;
                break;
            }

            // We have some kind of race condition with newly exported orders.
            // If order was just exported, wait to next iteration of order export/update
            if ($integrationOrder["export_status"] == self::EXPORT_STATUS_TRANSFERRED1) {
                $updateData["export_status"] = self::EXPORT_STATUS_TRANSFERRED2;
                break;
            }

            // Fetch Visma data to se what needs to be updated
            $newVismaData = $this->getVismaOrderData($incrementId, Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB);
            if (empty($newVismaData)) {

                // Double check if order is stuck in other company DB
                if ($integrationOrder["visma_db"] != Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB) {

                    // Try to get data for order from other company/country
                    $newVismaData = $this->getVismaOrderData($incrementId, $integrationOrder["visma_db"]);

                    // Check to se if we got any data
                    if (empty($newVismaData)) {

                        // Nope, no order there, just skip further processing for now
                        if ($this->getDebugMode()) {
                            Mage::helper($this->_defaultHelper)->log("Unable to fetch data for order {$incrementId} from Visma. Skip for now.");
                        }
                        break;
                    } else {

                        // Yes! we got data from other company/country. Use that data instead.
                        if (empty($integrationOrder["visma_order_id"])) {
                            Mage::helper($this->_defaultHelper)->log("Found order {$incrementId} in Visma DB '{$integrationOrder["visma_db"]}'.");
                        }
                    }
                } else {

                    // If not, we don't need to check further
                    if ($this->getDebugMode()) {
                        Mage::helper($this->_defaultHelper)->log("Unable to fetch data for order {$incrementId} from Visma. Skip for now.");
                    }
                    break;
                }
            }

            $this->localOrderUpdate($integrationOrder, $newMagentoData, $newVismaData);

            // We have problems with Visma not completing order creation and leaving orders in LckSt = 1 (or 3)
            // When we find one, skip further processing, revert export status back to new and send email about it!
            if (in_array($newVismaData["LckSt"], self::VISMA_FAULTY_LOCK_STATES)) {
                $updateData["export_status"] = self::EXPORT_STATUS_NEW;
                $subject = "Problem with order {$incrementId} in Visma";
                $body = "An order with id #{$newVismaData["OrdNo"]} was created in Visma, but it is broken (with LckSt = 1).\nThis usually means something went wrong when importing order.\n\n";
                $body .= print_r($newVismaData, true);
                Mage::helper($this->_defaultHelper)->sendAdminEmail($subject, $body, ["it"]);
                break;
            }

            // If integration table doesn't contain visma_order_id, but we got it now, add this to things to update
            // Sometimes we get visma_order_id from other company/country, but later we get it from default company/country
            if ($integrationOrder["visma_order_id"] != $newVismaData["OrdNo"]) {
                $integrationOrder["visma_order_id"] = $newVismaData["OrdNo"];
                $updateData["visma_order_id"] = $newVismaData["OrdNo"];
            }

            // If visma_status has changed, add this to things to update
            if ($integrationOrder["visma_status"] != $newVismaData["Gr"]) {
                $integrationOrder["visma_status"] = $newVismaData["Gr"];
                $updateData["visma_status"] = $newVismaData["Gr"];

                // Check if order was canceled in Visma
                if ($updateData["visma_status"] == self::VISMA_ORDER_STATUS_CANCELED) {
                    if ($newMagentoData["status"] != "canceled") {
                        $integrationOrder["export_status"] = self::EXPORT_STATUS_CANCELED;
                        $updateData["export_status"] = self::EXPORT_STATUS_CANCELED;
                        $cancelInMagento = true;
                    }
                }
            }

            // Check if order was exported and that we have visma order id
            if (in_array($integrationOrder["export_status"], [ self::EXPORT_STATUS_TRANSFERRED1, self::EXPORT_STATUS_TRANSFERRED2 ]) && !empty($integrationOrder["visma_order_id"])) {
                $integrationOrder["export_status"] = self::EXPORT_STATUS_VERIFIED;
                $updateData["export_status"] = self::EXPORT_STATUS_VERIFIED;
            }

            // Check if order was verified and delivered
            if ($integrationOrder["export_status"] == self::EXPORT_STATUS_VERIFIED && $integrationOrder["visma_status"] == self::VISMA_ORDER_STATUS_DELIVERED) {
                $integrationOrder["export_status"] = self::EXPORT_STATUS_SHIPPED;
                $updateData["export_status"] = self::EXPORT_STATUS_SHIPPED;
                $makeCompleteInMagento = true;
            }

            // If export_status show that order is shipped and visma_status show that order is complete and we have invoce id from Visma:
            if ($integrationOrder["export_status"] == self::EXPORT_STATUS_SHIPPED && $integrationOrder["visma_status"] == self::VISMA_ORDER_STATUS_DELIVERED && !empty($newVismaData["InvoNo"])) {
                $integrationOrder["export_status"] = self::EXPORT_STATUS_COMPLETE;
                $updateData["export_status"] = self::EXPORT_STATUS_COMPLETE;
            }

            // Create shipping and invoce
            if ($makeCompleteInMagento) {

                // Check if it is safe to release order.
                // This is because it takes some time from order completion to tracking number showing up.
                // Actual calculation is done in function getVismaOrderData()
                if ($newVismaData["safe_to_complete"]) {

                    // Order is released and at least 30 minutes old, it should be safe to complete it
                    if ($this->invoiceOrder($integrationOrder)) {
                        if ($this->_shipOrder($integrationOrder, $newVismaData)) {
                            Mage::helper($this->_defaultHelper)->log("Order {$incrementId} now shipped and invoiced!");
                        } else {
                            Mage::helper($this->_defaultHelper)->log("Unable create shipment for order {$incrementId}", LOG_ERR);
                            $integrationOrder["export_status"] = self::EXPORT_STATUS_ERROR;
                            $updateData["export_status"] = self::EXPORT_STATUS_ERROR;
                        }
                    } else {
                        Mage::helper($this->_defaultHelper)->log("Unable to invoice order {$incrementId}", LOG_ERR);
                        $integrationOrder["export_status"] = self::EXPORT_STATUS_ERROR;
                        $updateData["export_status"] = self::EXPORT_STATUS_ERROR;
                    }
                } else {

                    // Order is released but we need to wait for tracking number from shipping
                    // Revert export_status to reflect that we need to wait.
                    $integrationOrder["export_status"] = self::EXPORT_STATUS_VERIFIED;
                    $updateData["export_status"] = self::EXPORT_STATUS_VERIFIED;
                }
            } elseif ($cancelInMagento) {
                try {
                    $order = Mage::getModel("sales/order")->load($integrationOrder["magento_order_id"]);
                    if ($order->canCancel()) {

                        // Please note that this triggers all cancel-events in Magento
                        $order->cancel();
                        $history = $order->addStatusHistoryComment("Order canceled by order export script.");
                        $history->setIsCustomerNotified(Mage_Sales_Model_Order_Status_History::CUSTOMER_NOTIFICATION_NOT_APPLICABLE);
                        $order->save();

                        $integrationOrder["export_status"] = self::EXPORT_STATUS_CANCELED;
                        $updateData["export_status"] = self::EXPORT_STATUS_CANCELED;
                        $updateData["magento_status"] = "canceled";
                    } else {
                        Mage::helper($this->_defaultHelper)->log("Unable to cancel order {$incrementId}!");
                        $integrationOrder["export_status"] = self::EXPORT_STATUS_ERROR;
                        $updateData["export_status"] = self::EXPORT_STATUS_ERROR;
                    }
                } catch (Exception $exception) {
                    Mage::helper($this->_defaultHelper)->logException($exception, "Exception while canceling order {$incrementId}!");
                    $integrationOrder["export_status"] = self::EXPORT_STATUS_ERROR;
                    $updateData["export_status"] = self::EXPORT_STATUS_ERROR;
                }
            }
        } while(false); // Only do loop once. Needed ability to "break".

        // Finally, update integration table
        if (!empty($updateData)) {
            $updateData["id"] = $integrationOrder["id"];
            $this->updateIntegrationOrder($updateData, intval($newMagentoData["store_id"]));
            Mage::helper($this->_defaultHelper)->log("Order {$incrementId} was updated.");
            $this->_wasLastOrderUpdated = true;
        }

        return $integrationOrder["export_status"] == self::EXPORT_STATUS_ERROR ? false : true;
    }

    public function localOrderUpdate($integrationOrder, $newMagentoData, $newVismaData)
    {
        return;
    }

    public function invoiceOrder($integrationOrder, $order = null)
    {
        try {
            if ($order === null) {
                $order = Mage::getModel("sales/order")->load($integrationOrder["magento_order_id"]);
            }

            if (!$order->getId()) {
                Mage::helper($this->_defaultHelper)->log("Unable to load order {$integrationOrder["increment_id"]}!", LOG_ERR);
                return false;
            }

            if ($order->canInvoice()) {
                $itemsQty = [];

                foreach ($order->getAllItems() as $_eachItem) {
                    $qtyToShip = $_eachItem->getQtyToShip();
                    if (!empty($qtyToShip)) {
                        $itemsQty[$_eachItem->getId()] = $qtyToShip;
                    }
                }

                if (!empty($itemsQty)) {
                    $invoice = Mage::getModel("sales/service_order", $order)->prepareInvoice($itemsQty);
                    if ($invoice->getTotalQty()) {
                        // TODO: CAPTURE_OFFLINE or CAPTURE_ONLINE?
                        $invoice->setRequestedCaptureCase(Mage_Sales_Model_Order_Invoice::CAPTURE_ONLINE);
                        $invoice->addComment("Invoice generated by order export script.");
                        $invoice->register();
                        $transactionSave = Mage::getModel("core/resource_transaction");
                        $transactionSave->addObject($invoice)->addObject($order);
                        $transactionSave->save();
                        return true;
                    } else {
                        Mage::helper($this->_defaultHelper)->log("Error invoicing order {$integrationOrder["increment_id"]} (no products found in order)", LOG_ERR);
                    }
                } else {
                    Mage::helper($this->_defaultHelper)->log("Error invoicing order {$integrationOrder["increment_id"]} (no products found in order)", LOG_ERR);
                }
            } else if ($order->hasInvoices()) {
                Mage::helper($this->_defaultHelper)->log("Order {$integrationOrder["increment_id"]} already invoiced");
                return true;
            } else {
                Mage::helper($this->_defaultHelper)->log("Unable to invoice order {$integrationOrder["increment_id"]}. canInvoice() and hasInvoices() is both false.", LOG_ERR);
            }
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while creating invoice for order {$integrationOrder["increment_id"]}");
        }

        return false;
    }

    protected function _shipOrder($integrationOrder, $vismaData)
    {
        $initialEnvironmentInfo = null;

        try {
            $order = Mage::getModel("sales/order")->load($integrationOrder["magento_order_id"]);

            if (!$order->getId()) {
                Mage::helper($this->_defaultHelper)->log("Unable to load order {$integrationOrder["increment_id"]}!", LOG_ERR);
                return false;
            }

            $order->setData("state", Mage_Sales_Model_Order::STATE_COMPLETE);
            $order->setData("status", Mage_Sales_Model_Order::STATE_COMPLETE);

            if ($order->canShip()) {
                $itemsQty = [];

                foreach ($order->getAllItems() as $_eachItem) {
                    $qtyToShip = $_eachItem->getQtyToShip();
                    $sku = $_eachItem->getSku();
                    if (
                        !empty($qtyToShip)
                        && !$_eachItem->getIsVirtual()
                        && !$_eachItem->getAwarditDropship() // no dropship
                        && stripos($sku, "awd_") === false // no dropship
                    ) {
                        $itemsQty[$_eachItem->getId()] = $qtyToShip;
                    }
                }

                if (!empty($itemsQty)) {
                    $shipment = Mage::getModel("sales/service_order", $order)->prepareShipment($itemsQty);

                    // Begin: Emulate store context for order
                    $storeId = $order->getStoreId();
                    $appEmulation = Mage::getSingleton("core/app_emulation");
                    $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($storeId);

                    if (!empty($vismaData["WebPg"])) {
                        $trackingNumbers = explode(",", $vismaData["WebPg"]);
                        if (!empty($trackingNumbers)) {
                            foreach ($trackingNumbers as $trackingNumber) {
                                $trackingNumber = trim($trackingNumber);
                                if (!empty($trackingNumber)) {
                                    $arrTracking = [
                                        "carrier_code" => $order->getShippingCarrier()->getCarrierCode(),
                                        "title" => $order->getShippingCarrier()->getConfigData("title"),
                                        "track_number" => $trackingNumber
                                    ];
                                    $track = Mage::getModel("sales/order_shipment_track")->addData($arrTracking);
                                    $shipment->addTrack($track);
                                    Mage::helper($this->_defaultHelper)->log("Added tracking to order {$integrationOrder["increment_id"]}: " . json_encode($arrTracking));
                                }
                            }
                        }
                    } else {
                        Mage::helper($this->_defaultHelper)->log("Order {$integrationOrder["increment_id"]} is missing tracking numbers.");
                    }

                    // Register Shipment
                    $shipment->register();

                    // Save the Shipment
                    $shipment->getOrder()->setIsInProcess(true);
                    $transactionSave = Mage::getModel("core/resource_transaction");
                    $transactionSave->addObject($shipment)->addObject($order);
                    $transactionSave->save();

                    if ($vismaData["safe_to_send_email"]) {
                        $emailSentStatus = $shipment->getData("email_sent");
                        if (!$emailSentStatus) {
                            $shipment->sendEmail(true, "");
                            $shipment->setEmailSent(true);
                            $shipment->save();
                        }
                    } else {
                        Mage::helper($this->_defaultHelper)->log("Order {{$integrationOrder["increment_id"]}} was released {$vismaData["times"]["release_iso"]}, do not send shipping email.");
                    }

                    $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
                    // End: Emulation

                    // Add order comment about shipment
                    $history = $order->addStatusHistoryComment("Order shipped by order export script.");
                    $history->setIsCustomerNotified(Mage_Sales_Model_Order_Status_History::CUSTOMER_NOTIFICATION_NOT_APPLICABLE);
                }
            }

            // Save the Order
            $order->save();
        } catch (Exception $exception) {
            Mage::helper($this->_defaultHelper)->logException($exception, "Exception while creating shipment for order {$integrationOrder["increment_id"]}!");
            if (!empty($initialEnvironmentInfo)) {
                    $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo); // End: Emulation
            }
            return false;
        }

        return true;
    }

    public function getLocalProductExtraData($item, $product, &$orderData)
    {
        return;
    }
}

class SimpleXMLExtended extends SimpleXMLElement {

    static $roundKeys = [
        "OrdLn_NoInvoAb" => 0, // units (qty)
        "OrdLn_Price" => 4, // Price / unit
        "OrdLn_Free1" => 2, // points / unit
        "OrdLn_Free3" => 4, // cash / unit
    ];

    public function addCData($key, $cdata)
    {
        if (!empty(self::$roundKeys[$key]) && is_numeric($cdata)) {
            $cdata = round($cdata, self::$roundKeys[$key]);
        }
        $node = dom_import_simplexml($this);
        $no = $node->ownerDocument;
        $node->appendChild($no->createCDATASection(trim($cdata)));
    }

    public function addArray($xmlKey, $xmlData)
    {
        $newObj = $this->addChild($xmlKey);

        if (is_array($xmlData)) {
            foreach ($xmlData as $key => $data) {
                $newChild = $newObj->addChild($key);
                $newChild->addCData($key, $data);
            }
        } else {
            $newObj->addCData($xmlKey, $xmlData);
        }

        return $newObj;
    }

}
