<?php

class Awardit_Integration_Helper_Data extends Mage_Core_Helper_Abstract
{

    const ORDER_AGE_MIN               = "-5 minutes";      // Order must be at least this old to be exported
    const ORDER_AGE_MAX               = "-5 days";         // Time limit for checking for new orders
    const SHIPPING_EMAIL_TIME_LIMIT   = "-3 days";         // If shipping was done > this limit ago, do not send shipping email
    const PRODUCT_SAVE_SLEEP_TIME     = 100000;            // microseconds
    const STOCK_UPDATE_SLEEP_TIME     = 100000;            // microseconds
    const STOCK_UPDATE_BATCH_SIZE     = 100;
    const DEFAULT_STOCK_QTY           = 200000;
    const DEFAULT_LOG_FILENAME        = "integration.log";
    const DEFAULT_CURRENCY_ID         = 46;                // Sweden
    const DEFAULT_CURRENCY_CODE       = "SEK";             // Sweden
    const DEFAULT_COUNTRY_ID          = 46;                // Sweden
    const DEFAULT_VISMA_CUSTOMER_ID   = "100100";
    const DEFAULT_VISMA_EXCHANGE_RATE = 100;
    const DEFAULT_VISMA_ORDER_DB      = "F0001";           // Sweden

    const INTEGRATION_TABLE_ORDERS       = "awardit_integration_orders";
    const INTEGRATION_TABLE_PRODUCTS     = "awardit_integration_products";
    const INTEGRATION_TABLE_BONUS_QUEUE  = "awardit_integration_bonus_queue";

    const BONUS_QUEUE_ACL_ORDER_INFO = "admin/sales/bonus_queue_info";
    const BONUS_QUEUE_PROCESSED_STATUS_WAITING = 0;
    const BONUS_QUEUE_PROCESSED_STATUS_DONE = 1;
    const BONUS_QUEUE_PROCESSED_STATUS_CANCELED = 8;
    const BONUS_QUEUE_PROCESSED_STATUS_ERROR = 9;
    const BONUS_QUEUE_PROCESSING_TRIES_MAX = 5;

    public static $basePath = null;
    public static $adminEmailRecipients = [
        "it" => ["it@awardit.com"], // everything
        "kam" => ["kam@awardit.com"], // prices
        "products" => ["products@awardit.com"], // missing product data
        "stock" => ["jonas.sjogren@awardit.com"] // stock
    ];

    protected $_vismaDB = null;
    protected $_vismaFTPConnection = null;
    protected $_debugMode = false;
    protected $_instanceUniqueVismaShopIds = [];
    protected $_attributeIds = [];
    protected $_shopIdConversionTable = [];
    protected $_groupConversionTable = [];
    protected $_currencyConversionTable = [];
    protected $_zeroPriceProducts = [];
    protected $_systemAttributeOptions = [];
    protected $_vismaCurrencyExchangeRates = [];
    protected $_exceptionToException = [];
    protected $_itemBlacklist = [];

    public function __construct()
    {
    }

    public function getInstanceId()
    {
        return intval(Mage::getStoreConfig("integration/general/magento_instance"));
    }

    public function setDebugMode($mode)
    {
        $this->_debugMode = $mode ? true : false;
    }

    public function getDebugMode()
    {
        return $this->_debugMode;
    }

    public function reconnectVismaDB()
    {
        if (!empty($this->_vismaDB)) {
            try {
                $this->_vismaDB->closeConnection();
            } catch (Exception $ex) {
                // Silently ignore exceptions here
            }
        }
        $this->_vismaDB = null;

        return $this->getVismaDB();
    }

    public function getVismaDB()
    {
        if (empty($this->_vismaDB)) {
            $config = [
                "host" => Mage::getStoreConfig("integration/general/visma_host"),
                "username" => Mage::getStoreConfig("integration/general/visma_db_user"),
                "password" => Mage::getStoreConfig("integration/general/visma_db_password"),
                "dbname" => self::DEFAULT_VISMA_ORDER_DB,
                "port" => "1433",
                "pdoType" => "dblib",
                "charset" => "UTF-8"
            ];

            try {
                $this->_vismaDB = new Zend_Db_Adapter_Pdo_Mssql($config);
            } catch (Exception $exception) {
                $this->logException($exception, "Exception while connecting to mssql server!");
            }
        }

        return $this->_vismaDB;
    }

    public function openVismaFTP()
    {
        if (!empty($this->_vismaFTPConnection)) {
            return $this->_vismaFTPConnection;
        }

        try {
            $this->_vismaFTPConnection = ftp_connect(Mage::getStoreConfig('integration/general/visma_ftp_server'));
            if (
                ftp_login(
                    $this->_vismaFTPConnection,
                    Mage::getStoreConfig('integration/general/visma_ftp_user'),
                    Mage::getStoreConfig('integration/general/visma_ftp_password')
                )
            ) {
                ftp_pasv($this->_vismaFTPConnection, true);
            } else {
                $this->log("Unable to login to FTP-server!", LOG_ERR);
                ftp_close($this->_vismaFTPConnection);
                $this->_vismaFTPConnection = null;
            }
        } catch (Exception $exception) {
            $this->logException($exception, 'Exception while connecting to FTP server!');
            $this->closeVismaFTP();
            $this->_vismaFTPConnection = null;
        }

        return $this->_vismaFTPConnection;
    }

    public function closeVismaFTP()
    {
        if (!empty($this->_vismaFTPConnection)) {
            ftp_close($this->_vismaFTPConnection);
        }

        $this->_vismaFTPConnection = null;
    }

    public function getMagentoTaxIdFromVismaTaxId($vismaTaxId, $currencyId, $countryId = 0)
    {
        $sqlQuery = "SELECT magento_tax_id FROM tax_translations WHERE magento_instance = :instanceId AND swedish_visma_tax_id = :vismaTaxId AND (currency_id = :currencyId OR currency_id = 0) AND country_id = :countryId ORDER BY currency_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "vismaTaxId" => intval($vismaTaxId),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return intval(Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params));
        // If we don't get any value from query, intval() makes it a 0 => no vat, exactly what we want as default fallback value.

    }

    public function getTaxPercentFromMagentoTaxId($magentoTaxId, $currencyId, $countryId)
    {
        $sqlQuery = "SELECT tax_percent FROM tax_translations WHERE magento_instance = :instanceId AND magento_tax_id = :magentoTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
            "magentoTaxId" => intval($magentoTaxId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getTaxPercentFromVismaTaxId($vismaTaxId, $currencyId, $countryId)
    {
        $sqlQuery = "SELECT tax_percent FROM tax_translations WHERE magento_instance = :instanceId AND swedish_visma_tax_id = :vismaTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
            "vismaTaxId" => intval($vismaTaxId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getVismaTaxIdFromMagentoTaxId($magentoTaxId, $currencyId, $countryId)
    {
        // Always use local Visma tax id, based on currency and country.
        $sqlQuery = "SELECT local_visma_tax_id FROM tax_translations WHERE magento_instance = :instanceId AND magento_tax_id = :magentoTaxId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, country_id DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "magentoTaxId" => intval($magentoTaxId),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);
    }

    public function getMaxTaxForCurrencyAndCountry($currencyId, $countryId)
    {
        $sqlQuery = "SELECT * FROM tax_translations WHERE magento_instance = :instanceId AND (currency_id = :currencyId OR currency_id = 0) AND (country_id = :countryId OR country_id = 0) ORDER BY currency_id DESC, tax_percent DESC LIMIT 1";
        $params = [
            "instanceId" => intval($this->getInstanceId()),
            "currencyId" => intval($currencyId),
            "countryId" => intval($countryId),
        ];
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchRow($sqlQuery, $params);
    }

    public function createExceptionToException()
    {
        $data = Mage::getStoreConfig('integration/products/exception_to_exception');
        if (!empty($data)) {
            $data = json_decode($data, true);
            if (is_array($data)) {
                $this->_exceptionToException = $data;
            }
        }
    }

    public function isExceptionToException($sku)
    {
        if (empty($this->_exceptionToException)) {
            $this->createExceptionToException();
        }

        return array_key_exists($sku, $this->_exceptionToException);
    }

    public function isOnlyDropship($order) {
        foreach ($order->getAllItems() as $_eachItem) {
            $sku = $_eachItem->getSku();
            if (
                !$_eachItem->getAwarditDropship()
                    &&
                stripos($sku, "awd_") === false
            ) {
                return false;
            }
        }
        return true;
    }

    public function containsDropship($order) {
        foreach ($order->getAllItems() as $_eachItem) {
            $sku = $_eachItem->getSku();
            if (
                $_eachItem->getAwarditDropship()
                    ||
                stripos($sku, "awd_") !== false
            ) {
                return true;
            }
        }
        return false;
    }

    public function getMagentoGroupIds($vismaGroupId)
    {
        $groupIds = [];

        if (empty($this->_groupConversionTable)) {
            $this->createGroupConversionTable();
        }

        foreach ($this->_groupConversionTable as $row) {
            if ($row["visma_group_id"] == $vismaGroupId) {
                $groupIds[] = $row["customer_group_id"];
            }
        }

        // Normal instances needs empty array if no group id is found
        return !empty($groupIds) ? $groupIds : [];
    }

    public function getVismamGroupIds($magentoGroupId)
    {
        $groupIds = [];

        if (empty($this->_groupConversionTable)) {
            $this->createGroupConversionTable();
        }

        foreach ($this->_groupConversionTable as $row) {
            if ($row["customer_group_id"] == $magentoGroupId) {
                $groupIds[] = $row["visma_group_id"];
            }
        }

        return !empty($groupIds) ? $groupIds : [0];
    }

    public function createGroupConversionTable()
    {
        $sqlQuery = "SELECT visma_group_id, customer_group_id FROM customer_group WHERE visma_group_id > 0";
        $this->_groupConversionTable = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
    }

    public function getVismaProductStock($prodNo, $warehouses = [])
    {
        // If configured to use external stock:
        if (Mage::getStoreConfig("integration/external_stock/enabled")) {
            // Try to get stock qty from external stock
            $sqlQuery1 = "SELECT sku, SUM(qty) AS qty FROM stock WHERE sku = :sku AND warehouse_id = :warehouseId AND (magento_id = 0 OR magento_id = :magentoId) GROUP BY sku";
            $params1 = [
                "sku" => $prodNo,
                "warehouseId" => Mage::getStoreConfig("integration/external_stock/warehouse_id"),
                "magentoId" => $this->getInstanceId()
            ];
            $externalStockData = Mage::getSingleton("core/resource")->getConnection("stock_handler_read")->fetchRow($sqlQuery1, $params1);
            if (!empty($externalStockData["sku"])) {
                return $externalStockData["qty"];
            }
        }

        // If no external stock was found, get stock qty from Visma
        if (
            empty($warehouses)
                ||
            !is_array($warehouses)
        ) {
            switch (Mage::getStoreConfig('integration/external_stock/warehouse_id')) {
                case 2:
                    $stcbal = "11,13";
                    break;
                case 1:
                default:
                    $stcbal = "1,3";
                    break;
            }
        } else {
            $stcbal = implode(",", $warehouses);
        }

        // Realiserat saldo + Orealiserad lagerökn. - Reserverad - Plockat utan reservering.
        // Bal + StcInc - (ShpRsv+ShpRsvIn) - PicNotR
        // ProdNo is specified as varchar and MSSQL freaks out if we ask for 12345 instead of '12345'
        $sqlQuery2 = "SELECT SUM(StcInc + Bal - ShpRsv - PicNotR) AS stock_balance FROM StcBal WHERE ProdNo = '{$prodNo}' AND StcNo IN ({$stcbal})";
        return $this->getVismaDB()->fetchOne($sqlQuery2);
    }

    public function getShopIdConversionTable()
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        return $this->_shopIdConversionTable;
    }

    public function getMagentoWebsiteIdsForProduct($priceData)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $productWebsites = [];

        foreach ($priceData as $prices) {
            if (
                !empty($prices["shop_id"])
                    &&
                !empty($prices["currency_id"])
            ) {
                foreach ($this->_shopIdConversionTable as $row) {
                    if (
                        $row["currency_id"] == $prices["currency_id"]
                            && 
                        $row["shop_id"] == $prices["shop_id"]
                            &&
                        !in_array($row["website_id"], $productWebsites)
                    ) {
                        $productWebsites[] = $row["website_id"];
                    }
                }
            }
        }

        if (!empty($productWebsites)) {
            sort($productWebsites, SORT_NUMERIC);
        }

        return $productWebsites;
    }

    public function getMagentoWebsiteIds($shopId, $currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $productWebsites = [];

        foreach ($this->_shopIdConversionTable as $row) {
            if (
                $row["currency_id"] == $currencyId
                    &&
                $row["shop_id"] == $shopId
                    &&
                !in_array($row["website_id"], $productWebsites)
            ) {
                $productWebsites[] = $row["website_id"];
            }
        }

        return $productWebsites;
    }

    public function getMagentoStoreIds($websiteId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $storeIds = [];

        foreach ($this->_shopIdConversionTable as $row) {
            if (
                $row["website_id"] == $websiteId
                    &&
                !in_array($row["store_id"], $storeIds)
            ) {
                $storeIds[] = $row["store_id"];
            }
        }

        return $storeIds;
    }

    public function getInstanceUniqueVismaShopIds()
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        return $this->_instanceUniqueVismaShopIds;
    }

    public function checkIfShopIdExistsInThisMagentoInstance($shopId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        if (empty($shopId)) {
            return false;
        } else {
            return in_array($shopId, $this->_instanceUniqueVismaShopIds);
        }
    }

    public function havePriceRowForCurrencyShopIdGroupId($priceData, $currencyId, $shopId, $groupId)
    {
        foreach ($priceData as $priceRow) {
            if (
                $priceRow["currency_id"] == $currencyId
                    &&
                $priceRow["shop_id"] == $shopId
                    &&
                $priceRow["customer_group_id"] == $groupId
            ) {
                return true;
            }
        }

        return false;
    }

    public function getAttributeIdByAttributeSetAndCode($attributeSetId, $attributeCode)
    {
        $key = "{$attributeSetId}:{$attributeCode}";

        if (array_key_exists($key, $this->_attributeIds)) {
            return $this->_attributeIds[$key];
        }

        $sqlQuery = "
            SELECT
                a.attribute_id
            FROM eav_attribute_set s
            JOIN eav_entity_attribute ea ON ea.attribute_set_id = s.attribute_set_id
            JOIN eav_attribute a ON a.attribute_id = ea.attribute_id
            WHERE s.attribute_set_id = :attributeSetId AND a.attribute_code = :attributeCode
        ";
        $params = [
            "attributeSetId" => $attributeSetId,
            "attributeCode" => $attributeCode
        ];
        $this->_attributeIds[$key] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);

        return $this->_attributeIds[$key];
    }

    public function getConfigValueForStores($path, $filter = null)
    {
        $baseQuery = "
            SELECT
                tmp1.website_id,
                tmp1.store_id,
                COALESCE(MAX(tmp1.value3), MAX(tmp1.value2), MAX(tmp1.value1)) AS config_value
            FROM (
                SELECT
                    cs.website_id,
                    cs.store_id,
                    cs.code,
                    IF (ccd1.scope = 'default', ccd1.value, NULL) AS value1,
                    IF (ccd1.scope = 'websites', ccd1.value, NULL) AS value2,
                    IF (ccd1.scope = 'stores', ccd1.value, NULL) AS value3
                FROM core_store cs
                JOIN core_config_data ccd1 ON ccd1.path = :path AND ((ccd1.scope_id = 0 AND ccd1.scope = 'default') OR (ccd1.scope_id = cs.website_id AND ccd1.scope = 'websites') OR (ccd1.scope_id = cs.store_id AND ccd1.scope = 'stores'))
                WHERE ccd1.value IS NOT NULL
            ) tmp1
            GROUP BY tmp1.website_id, tmp1.store_id
        ";
        if (empty($filter)) {
            return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($baseQuery, [ "path" => $path ]);
        }

        $wrappingQuery = "SELECT tmp2.* FROM ({$baseQuery}) tmp2 WHERE tmp2.config_value = :filterValue";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($wrappingQuery, [ "path" => $path, "filterValue" => $filter ]);
    }

    public function getMagentoStoreIdsByVismaShopAndCurrency($vismaShopId, $currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $storeIds = [];
        foreach ($this->_shopIdConversionTable as $row) {
            if (
                $row["shop_id"] == $vismaShopId
                    &&
                $row["currency_id"] == $currencyId
                    &&
                !in_array($row["store_id"], $storeIds)
            ) {
                $storeIds[] = $row["store_id"];
            }
        }

        return $storeIds;
    }

    public function getVismaShopIdsByCurrencyId($currencyId)
    {
        if (empty($this->_shopIdConversionTable)) {
            $this->createShopIdConversionTable();
        }

        $shopIds = [];
        foreach ($this->_shopIdConversionTable as $row) {
            if (
                $row["currency_id"] == $currencyId
                    &&
                !in_array($row["shop_id"], $shopIds)
            ) {
                $shopIds[] = $row["shop_id"];
            }
        }

        return $shopIds;

    }

    public function createShopIdConversionTable()
    {
        try {

            $sqlQuery = "
                SELECT
                    tmp.website_id,
                    tmp.store_id,
                    0 AS currency_id,
                    COALESCE(MAX(tmp.currency3), MAX(currency2), MAX(currency1)) AS currency_code,
                    COALESCE(MAX(tmp.shop2), MAX(shop1)) AS shop_id
                FROM (
                    SELECT
                        cs.website_id,
                        cs.store_id,
                        cs.code,
                        IF (ccd1.scope = 'default', ccd1.value, NULL) AS currency1,
                        IF (ccd1.scope = 'websites', ccd1.value, NULL) AS currency2,
                        IF (ccd1.scope = 'stores', ccd1.value, NULL) AS currency3,
                        IF (ccd1.scope = 'default', ccd2.value, NULL) AS shop1,
                        IF (ccd1.scope = 'websites', ccd2.value, NULL) AS shop2
                    FROM core_store cs
                    JOIN core_config_data ccd1 ON ccd1.path = 'currency/options/default' AND ((ccd1.scope_id = 0 AND ccd1.scope = 'default') OR (ccd1.scope_id = cs.website_id AND ccd1.scope = 'websites') OR (ccd1.scope_id = cs.store_id AND ccd1.scope = 'stores'))
                    JOIN core_config_data ccd2 ON ccd2.path = 'integration/general/visma_shop_id' AND ((ccd2.scope_id = 0 AND ccd2.scope = 'default') OR (ccd2.scope_id = cs.website_id AND ccd2.scope = 'websites') OR (ccd2.scope_id = cs.store_id AND ccd2.scope = 'stores'))
                    WHERE ccd2.value IS NOT NULL
                ) tmp
                GROUP BY tmp.website_id, tmp.store_id
            ";
            $this->_shopIdConversionTable = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
            $this->_instanceUniqueVismaShopIds = [];

            // Loop through all websites and get currency id
            foreach (array_keys($this->_shopIdConversionTable) as $rowIndex) {
                $this->_shopIdConversionTable[$rowIndex]["currency_id"] = $this->getVismaCurrencyId($this->_shopIdConversionTable[$rowIndex]["currency_code"]);
                if (!in_array($this->_shopIdConversionTable[$rowIndex]["shop_id"], $this->_instanceUniqueVismaShopIds)) {
                    $this->_instanceUniqueVismaShopIds[] = $this->_shopIdConversionTable[$rowIndex]["shop_id"];
                }
            }

            if (empty($this->_instanceUniqueVismaShopIds)) {
                $this->log("No Visma Shop Ids in this Magento instance!", LOG_ERR);
            }
        } catch (Exception $exception) {
            $this->logException($exception, "Exception while creating _shopIdConversionTable");
        }
    }

    public function getVismaCurrencyId($currencyCode)
    {
        if (empty($this->_currencyConversionTable)) {
            $this->createCurrencyCodeConversionTable();
        }

        foreach ($this->_currencyConversionTable as $row) {
            if ($row["currency_code"] == $currencyCode) {
                return $row["currency_id"];
            }
        }

        return self::DEFAULT_CURRENCY_ID;
    }

    public function getMagentoCurrencyCode($currencyId)
    {
        if (empty($this->_currencyConversionTable)) {
            $this->createCurrencyCodeConversionTable();
        }

        foreach ($this->_currencyConversionTable as $row) {
            if ($row["currency_id"] == $currencyId) {
                return $row["currency_code"];
            }
        }

        return self::DEFAULT_CURRENCY_CODE;
    }

    public function getVismaCurrencyExchangeRate($currencyId)
    {
        if (empty($this->_vismaCurrencyExchangeRates)) {
            $sqlQuery = "SELECT CurNo, SalesRt FROM Cur";
            $result = $this->getVismaDB()->fetchAll($sqlQuery);
            if (!empty($result)) {
                $this->_vismaCurrencyExchangeRates = [];
                foreach ($result as $row) {
                    $this->_vismaCurrencyExchangeRates[$row['CurNo']] = $row['SalesRt'];
                }
            }
        }

        if (array_key_exists($currencyId, $this->_vismaCurrencyExchangeRates)) {
            return $this->_vismaCurrencyExchangeRates[$currencyId];
        }

        return self::DEFAULT_VISMA_EXCHANGE_RATE;
    }

    public function getVismaCountryId($countryCode)
    {
        $sqlQuery = "SELECT country_id FROM country_codes WHERE country_code = ?";
        return Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $countryCode);
    }

    public function createCurrencyCodeConversionTable()
    {
        // Get list of unique currency codes in this instance
        $sqlQuery1 = "
            SELECT GROUP_CONCAT(tmp.currency_code SEPARATOR \"','\") AS currency_codes FROM
            (SELECT DISTINCT(`value`) AS currency_code, 1 AS fake_group FROM core_config_data WHERE path = 'currency/options/default') tmp
            GROUP BY tmp.fake_group
        ";
        $currencyCodes = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery1);

        // Get list of currency ids from currency codes
        $sqlQuery2 = "SELECT CurNo AS currency_id, ISO AS currency_code FROM Cur WHERE iso IN ('{$currencyCodes}')";
        $this->_currencyConversionTable = $this->getVismaDB()->fetchAll($sqlQuery2);
    }

    public function getMagentoWebsiteIdByStoreId($storeId)
    {
        $sqlQuery = "SELECT website_id FROM core_store WHERE store_id = ?";
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $storeId);
    }

    public function getVismaOrderDB($storeId)
    {
        return Mage::getStoreConfig('integration/general/visma_company_db', $storeId) ?: Awardit_Integration_Helper_Data::DEFAULT_VISMA_ORDER_DB;
    }

    public function getVismaOrderDBOverride($storeId, $currencyId, $countryId)
    {
        $rawOverrides = Mage::getStoreConfig("integration/general/db_override", $storeId);
        if (empty($rawOverrides)) {
            return [];
        }

        $overrides = unserialize($rawOverrides);
        if (
            empty($overrides)
                ||
            !is_array($overrides)
        ) {
            return [];
        }

        foreach ($overrides as $override) {
            if (
                $override["currency_id"] == $currencyId
                    &&
                $override["country_id"] == $countryId
            ) {
                return $override;
            }
        }
    }

    public function translateAttributeSetId($remoteInstance, $remoteAttributeSetId)
    {

        // Try to find the same attribute set here at local instance, as remote instance
        $sqlQuery = "
            SELECT
                T2.attribute_id
            FROM
                attribute_set_translations T1
            JOIN
                attribute_set_translations T2 ON T2.attribute_name = T1.attribute_name AND T2.instance_id = :localInstanceId
            WHERE
                T1.instance_id = :remoteInstanceId
                AND T1.attribute_id = :remoteAttributeSetId
        ";
        $params = [
            "localInstanceId" => $this->getInstanceId(),
            "remoteInstanceId" => $remoteInstance,
            "remoteAttributeSetId" => $remoteAttributeSetId
        ];
        $localAttributeSetId = Mage::getSingleton("core/resource")->getConnection("integration_read")->fetchOne($sqlQuery, $params);

        return intval($localAttributeSetId ?: Mage::getModel("catalog/product")->getDefaultAttributeSetId());
    }

    public function getManufacturerId($text)
    {
        return $this->getOptionId('manufacturer', $this->convertString($text));
    }

    public function getOptionId($attributeCode, $value, $createOnMissing = true)
    {

        try {
            $sqlQuery = "
                SELECT
                    eao.option_id
                FROM eav_attribute ea
                JOIN eav_attribute_option eao ON eao.attribute_id = ea.attribute_id
                JOIN eav_attribute_option_value eaov ON eaov.option_id = eao.option_id
                WHERE
                    ea.attribute_code = :attributeCode
                    AND eaov.value = :attributeValue
                    AND ea.entity_type_id = :typeId
            ";
            $params = [
                "attributeCode" => $attributeCode,
                "attributeValue" => $value,
                "typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()
            ];
            $data = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);

            // If no option was found, create one
            if (
                $createOnMissing
                    &&
                empty($data)
            ) {
                return $this->createOption($attributeCode, $value);
            } else {
                return $data;
            }
        } catch (Exception $exception) {
            $this->logException($exception, "Exception while getting option_id for attribute '{$attributeCode}'!");
        }

        return null;
    }

    public function getSystemAttributeOptionValue($attributeCode, $optionText)
    {
        $optionValue = null;
        if (empty($this->_systemAttributeOptions)) {
            $this->_systemAttributeOptions = [
                "status" => Mage_Catalog_Model_Product_Status::getOptionArray(),
                "visibility" => Mage_Catalog_Model_Product_Visibility::getOptionArray()
            ];
        }

        if (array_key_exists($attributeCode, $this->_systemAttributeOptions)) {
            $optionValue = array_search($optionText, $this->_systemAttributeOptions[$attributeCode]);
        }

        return $optionValue;
    }

    public function createManufacturer($text)
    {
        return createOption("manufacturer", $text);
    }

    public function createOption($attributeCode, $value, $storeId = 0)
    {
        try {
            $sqlQuery = "SELECT ea.attribute_id FROM eav_attribute ea WHERE ea.attribute_code = :attributeCode AND ea.entity_type_id = :typeId";
            $params = [
                "attributeCode" => $attributeCode,
                "typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()
            ];
            $attributeId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery, $params);

            if (empty($attributeId)) {
                $this->log("Unable to find attribute_id for attribute '{$attributeCode}'");
                return null;
            }

            $attribute_option = [
                "option_id" => null,
                "attribute_id" => $attributeId,
                "sort_order" => 0
            ];

            Mage::getSingleton("core/resource")->getConnection("core_write")->insert("eav_attribute_option", $attribute_option);
            $option_id = Mage::getSingleton("core/resource")->getConnection("core_write")->lastInsertId();

            if (empty($option_id)) {
                $this->log("Unable to create new option '{$value}' for attribute '{$attributeCode}'");
                return null;
            }
            $attribute_option_value = [
                "value_id" => null,
                "option_id" => $option_id,
                "store_id" => $storeId,
                "value" => $value
            ];
            Mage::getSingleton("core/resource")->getConnection("core_write")->insert("eav_attribute_option_value", $attribute_option_value);

            return $option_id;
        } catch (Exception $exception) {
            $this->logException($exception, "Unable to create new option '{$value}' for attribute '{$attributeCode}'");
        }

    }

    public function getAttributesForAttributeSet($attributeSetId)
    {

        if (empty($attributeSetId)) {
            $attributeSetId = Mage::getModel("catalog/product")->getDefaultAttributeSetId();
        }

        $attributesToExport = [];
        $sqlQuery = "
            SELECT
                a.attribute_code,
                a.frontend_input,
                a.source_model
            FROM eav_entity_attribute ea
            JOIN eav_attribute a ON a.attribute_id = ea.attribute_id
            JOIN eav_attribute_group g ON g.attribute_group_id = ea.attribute_group_id
            WHERE
                ea.attribute_set_id = ?
                AND g.attribute_group_name NOT IN ('Prices','Images')
                AND a.backend_type IN ('varchar','int','datetime','text','decimal')
                AND a.frontend_input IN ('text','weight','select','date','datetime','boolean','price','textarea')
                AND a.attribute_code NOT IN ('status')
            ORDER BY a.attribute_code
        ";

        $result = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $attributeSetId);
        if (empty($result)) {
            return $attributesToExport;
        }

        foreach ($result as $row) {
            $attributesToExport[$row["attribute_code"]] = [
                "frontend_input" => $row["frontend_input"],
                "source_model" => $row["source_model"]
            ];
        }

        // Ugly fix for 'status' when it doesn't exist in current attribute set
        if (!array_key_exists("status", $attributesToExport)) {
            $attributesToExport["status"] = [
                "frontend_input" => "select",
                "source_model" => "catalog/product_status"
            ];
        }

        return $attributesToExport;
    }

    public function deleteLocalAttributeValues($productId, $attributeCode)
    {
        // Fetch attribute info for wanted attribute code
        $sqlQuery1 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => $attributeCode
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery1, $params);

        // If no attribute was found, return
        if (empty($eava)) {
            return null;
        }

        // Find table name for matching backend type
        $table = "";
        switch ($eava["backend_type"]) {
            case "int":
                $table = "catalog_product_entity_int";
                break;

            case "datetime":
                $table = "catalog_product_entity_datetime";
                break;

            case "decimal":
                $table = "catalog_product_entity_decimal";
                break;

            case "text":
                $table = "catalog_product_entity_text";
                break;

            case "varchar":
                $table = "catalog_product_entity_varchar";
                break;

            default:
                break;
        }

        // If no matching table was found, return
        if (empty($table)) {
            return null;
        }

        // Now try to delete all attribute values except from default store
        $sqlQuery2 = "DELETE cpex.* FROM {$table} cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery2);
        $stmt->bindValue("typeId", $params["typeId"]);
        $stmt->bindValue("attributeId", intval($eava["attribute_id"]));
        $stmt->bindValue("productId", intval($productId));
        $stmt->execute();
        $affectedRows = $stmt->rowCount();
        if ($affectedRows > 0) {
            $this->log("Removed {$affectedRows} rows of local values for attribute '{$attributeCode}'");
        }

        return $affectedRows;

    }

    public function filterPhoneNumber($phonenumber, $countryCode = 'SE')
    {
        if (empty($phonenumber)) {
            return "";
        }

        switch ($countryCode) {
            case 'NO':
                $searchAndReplace = [
                    'search' => [
                        '/\(0\)/', '/[^\+\d]/', '/^0047/', '/^00/', '/^([^\+])/'
                    ],
                    'replace' => [
                        '', '', '+47', '+', '+47\1'
                    ]
                ];
                $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                break;

            case 'DK':
                $searchAndReplace = [
                    'search' => [
                        '/\(0\)/', '/[^\+\d]/', '/^0045/', '/^00/', '/^([^\+])/'
                    ],
                    'replace' => [
                        '', '', '+45', '+', '+45\1'
                    ]
                ];
                $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                break;

            case 'FI':
                $searchAndReplace = [
                    'search' => [
                        '/\(0\)/', '/[^\+\d]/', '/^00358/', '/^358/', '/^0/', '/^[^\+]/'
                    ],
                    'replace' => [
                        '', '', '+358', '+358', '+358', '+358'
                    ]
                ];
                $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                break;

            case 'SE':
                $searchAndReplace = [
                    'search' => [
                        '/\(0\)/', '/^\+46/', '/[^\+\d]/', '/^0046/', '/^46/', '/^(\+)?(0)*7/', '/^00/', '/^0123456$/'
                    ],
                    'replace' => [
                        '', '0', '', '0', '0', '07', '+', ''
                    ]
                ];
                $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                break;

            default:
                $searchAndReplace = [
                    'search' => [
                        '/\(0\)/', '/[^\+\d]/', '/^00/', '/^([^\+])/'
                    ],
                    'replace' => [
                        '', '', '+', '+\1'
                    ]
                ];
                $filteredPhonenumber = preg_replace($searchAndReplace['search'], $searchAndReplace['replace'], $phonenumber);
                break;
        }

        return $filteredPhonenumber;
    }

    public function createZeroPriceProductsTable()
    {
        $rawData = Mage::getStoreConfig("integration/products/zero_price_products");
        if (empty($rawData)) {
            return;
        }

        $data = json_decode($rawData, true);
        if (
            !is_array($data)
                ||
            empty($data)
        ) {
            return;
        }

        $this->_zeroPriceProducts = $data;
    }

    public function isZeroPriceProduct($sku)
    {
        if (empty($this->_zeroPriceProducts)) {
            $this->createZeroPriceProductsTable();
        }

        return array_key_exists($sku, $this->_zeroPriceProducts);
    }

    public function isEmailBlacklisted($email, $storeId)
    {
        $rawBlacklist = Mage::getStoreConfig("integration/orders/email_blacklist", $storeId);
        if (empty($rawBlacklist)) {
            return false;
        }

        $blacklist = unserialize($rawBlacklist);
        if (
            !is_array($blacklist)
                ||
            empty($blacklist)
        ) {
            return false;
        }

        foreach ($blacklist as $blacklistRow) {
            if (
                !empty($blacklistRow["email_address"])
                    &&
                !empty($email)
                    &&
                strtolower($blacklistRow["email_address"]) == strtolower($email)
            ) {
                return true;
            }
        }

        return false;
    }

    public function isItemBlacklisted($storeId, $sku)
    {
        if (!empty($this->_itemBlacklist)) {
            return in_array($sku, $this->_itemBlacklist);
        }

        $rawSkipList = Mage::getStoreConfig("integration/products/item_blacklist", $storeId);
        if (empty($rawSkipList)) {
            return false;
        }

        $skipList = unserialize($rawSkipList);
        if (
            !is_array($skipList)
                ||
            empty($skipList)
        ) {
            return false;
        }

        foreach ($skipList as $item) {
            $this->_itemBlacklist[] = $item["SKU"];
        }

        return in_array($sku, $this->_itemBlacklist);
    }

    public function getBonusQueueItems($attributes)
    {
        $params = [];
        $where = [];
        $limit = "";

        if (
            !is_array($attributes)
                ||
            empty($attributes)
        ) {
            return [];
        }

        foreach ($attributes as $key => $val) {
            switch ($key) {
                case "order_id":
                    $params["orderId"] = $val;
                    $where[] = "ca.order_id = :orderId";
                    break;

                case "increment_id":
                    $params["incrementId"] = $val;
                    $where[] = "ca.increment_id = :incrementId";
                    break;

                case "bonus_type":
                    $params["bonusType"] = $val;
                    $where[] = "ca.bonus_type = :bonusType";
                    break;

                case "processing_date":
                    $params["statusWaiting"] = self::BONUS_QUEUE_PROCESSED_STATUS_WAITING;
                    $params["statusError"] = self::BONUS_QUEUE_PROCESSED_STATUS_ERROR;
                    $params["processingDate"] = $val;
                    $where[] = "((ca.processed_status = :statusWaiting AND ca.processing_date <= :processingDate) OR (ca.processed_status = :statusError))";
                    break;

                case "processing_tries":
                    if ($val) {
                        $where[] = "ca.processing_tries > 0";
                    }
                    break;

                case "processed_status":
                    $params["processedStatus"] = $val;
                    $where[] = "ca.processed_status = :processedStatus";
                    break;

                case "limit":
                    if (!empty($val)) {
                        $limit = "LIMIT {$val}";
                    }
                    break;

                default:
            }
        }

        if (empty($params)) {
            return [];
        }

        $sqlQuery = "
            SELECT
                ca.*,
                sfo.created_at,
                sfo.store_id,
                cio.visma_db
            FROM " . self::INTEGRATION_TABLE_BONUS_QUEUE . " ca
            JOIN " . self::INTEGRATION_TABLE_ORDERS . " cio ON cio.magento_order_id = ca.order_id
            JOIN sales_flat_order sfo ON sfo.entity_id = ca.order_id
            WHERE " . implode(" AND ", $where) . "
            ORDER BY ca.processing_date ASC, ca.order_id ASC
            {$limit}
        ";

        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);
    }

    public function saveBonusQueueItems($items)
    {
        foreach ($items as $item) {
            $this->saveBonusQueueItem($item);
        }
    }

    public function saveBonusQueueItem($item)
    {
        $sqlQuery = "
            INSERT INTO " . self::INTEGRATION_TABLE_BONUS_QUEUE . "
                (order_id, increment_id, bonus_type, bonus_value, extra_data, processing_date, processing_tries)
            VALUES
                (:orderId, :incrementId, :bonusType, :bonusValue, :extraData, :processingDate, :processingTries)
            ON DUPLICATE KEY UPDATE processing_date = VALUES(processing_date)
        ";

        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);
        $stmt->bindValue("orderId", $item["order_id"]);
        $stmt->bindValue("incrementId", $item["increment_id"]);
        $stmt->bindValue("bonusType", $item["bonus_type"]);
        $stmt->bindValue("bonusValue", $item["bonus_value"]);
        $stmt->bindValue("extraData", is_array($item["extra_data"]) ? json_encode($item["extra_data"]) : $item["extra_data"] );
        $stmt->bindValue("processingDate", $item["processing_date"]);
        $stmt->bindValue("processingTries", $item["processing_tries"]);

        try {
            $stmt->execute();
            $this->log("Saved a bonus of type {$item["bonus_type"]} with a value of {$item["bonus_value"]} for order {$item["increment_id"]} to be processed at {$item["processing_date"]}");
        } catch (Exception $exception) {
            // Just log error and send email, do not fail whole order export
            $this->logException($exception, "Exception while saving bonus for order {$item["increment_id"]}");
            $subject = "Unable to save bonus for order {$item["increment_id"]}";
            $body .= "If nothing is done, this bonus will not be processed!\n\n";
            $body  = "Was trying to save this:\n" . print_r($item, true) . "\n";
            $body .= "Exception: {$exception->getMessage()}\n";
            $body .= "Trace:\n{$exception->getTraceAsString()}\n";
            $this->sendAdminEmail($subject, $body, ["it"]);
        }

    }

    public function updateBonusQueueItems($items)
    {
        foreach ($items as $item) {
            $this->updateBonusQueueItem($item);
        }
    }

    public function updateBonusQueueItem($item)
    {
        $params = [];
        $fields = [];
        $where = [];

        if (empty($item) || !is_array($item)) {
            return;
        }

        // If we have both 'order_id' and 'increment_id', remove the latter, 'order_id' is more specific.
        if (!empty($item["order_id"]) && !empty($item["increment_id"])) {
            unset($item["increment_id"]);
        }

        // We need to have at least 'bonus_type' and either 'order_id' or 'increment_id'
        if (empty($item["bonus_type"]) || (empty($item["order_id"]) && empty($item["increment_id"]))) {
            return;
        }

        foreach ($item as $key => $val) {
            switch ($key) {
                case "order_id":
                    $params["orderId"] = $val;
                    $where[] = "order_id = :orderId";
                    break;

                case "increment_id":
                    $params["incrementId"] = $val;
                    $where[] = "increment_id = :incrementId";
                    break;

                case "bonus_type":
                    $params["bonusType"] = $val;
                    $where[] = "bonus_type = :bonusType";
                    break;

                case "bonus_value":
                    $params["bonusValue"] = $val;
                    $fields[] = "bonus_value = :bonusValue";
                    break;

                case "extra_data":
                    $params["extraData"] = is_array($val) ? json_encode($val) : $val;
                    $fields[] = "extra_data = :extraData";
                    break;

                case "processing_date":
                    $params["processingDate"] = $val;
                    $fields[] = "processing_date = :processingDate";
                    break;

                case "processing_tries":
                    if ($val < 0) {
                        $fields[] = "processing_tries = processing_tries{$val}";
                    } else {
                        $params["processingTries"] = $val;
                        $fields[] = "processing_tries = :processingTries";
                    }
                    break;

                case "processed_status":
                    $params["processedStatus"] = $val;
                    $fields[] = "processed_status = :processedStatus";
                    break;

                case "processed_result":
                    $params["processedResult"] = is_array($val) ? json_encode($val) : $val;
                    $fields[] = "processed_result = :processedResult";
                    break;

                default:
            }
        }

        if (
            empty($where)
                ||
            empty($fields)
        ) {
            return;
        }

        // Always set updated_at to current time
        $fields[] = "processed_at = NOW()";

        try {
            $sqlQuery = "UPDATE " . self::INTEGRATION_TABLE_BONUS_QUEUE . " SET " . implode(", ", $fields) . " WHERE " . implode(" AND ", $where) ;
            $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery);

            foreach ($params as $key => $val) {
                $stmt->bindValue($key, $val);
            }
            $stmt->execute();

        } catch (Exception $exception) {
            $order_ref = empty($item["increment_id"]) ? $item["order_id"] : $item["increment_id"];
            $subject = "Exception while saving bonus for order {$order_ref}";
            $this->logException($exception, $subject);
            $body = "SQLQuery = {$sqlQuery}\n";
            $body  = "Was trying to save this:\n" . print_r($item, true) . "\n";
            $body .= "Exception: {$exception->getMessage()}\n";
            $body .= "Trace:\n{$exception->getTraceAsString()}\n";
            $this->sendAdminEmail($subject, $body, [ "it" ]);
        }
    }

    function fetchBinaryWithCurl($url) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_BINARYTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 10,
        ]);
        $binaryData = curl_exec($ch);

        if (curl_errno($ch)) {
            throw new Exception('Curl error: ' . curl_error($ch));
        }
        curl_close($ch);

        return $binaryData;
    }

    public function convertString($strParam, $mbCase = "none")
    {
        $isUTF8 = false;

        if (!empty($strParam)) {
            $strParam = trim($strParam);
        } else {
            return $strParam;
        }

        switch (mb_detect_encoding($strParam, "UTF-8, ISO-8859-1", true)) {
            case 'UTF-8':
                $isUTF8 = true;
                break;

            default:
                break;
        }

        if (!$isUTF8) {
            $strParam = utf8_encode($strParam);
        }

        switch ($mbCase) {
            case "upper":
                $strParam = mb_convert_case($strParam, MB_CASE_UPPER, "UTF-8");
                break;

            case "lower":
                $strParam = mb_convert_case($strParam, MB_CASE_LOWER, "UTF-8");
                break;

            case "title":
                $strParam = mb_convert_case($strParam, MB_CASE_TITLE, "UTF-8");
                break;

            case "none":
            default:
                break;
        }

        return $strParam;
    }

    public function isUTF8($strParam)
    {
        return mb_detect_encoding($strParam, "UTF-8, ISO-8859-1", true) === "UTF-8";
    }

    public function sendAdminEmail($subject, $body, $recipients = [])
    {
        if ($this->getDebugMode()) {
            // Always override email recipients in debug mode
            $recipients = ["it"];
        }

        $filteredRecipients = [];
        if (!is_array($recipients)) {
            if (!empty($recipients)) {
                $recipients = [$recipients];
            } else {
                $recipients = [];
            }
        }

        foreach($recipients as $recipient) {
            if (array_key_exists($recipient, self::$adminEmailRecipients)) {
                foreach (self::$adminEmailRecipients[$recipient] as $address) {
                    $filteredRecipients[] = $address;
                }
            } elseif(stripos($recipient, "@") !== false) {
                $filteredRecipients[] = $recipient;
            }
        }

        if (empty($filteredRecipients)) {
            // Default is to always send to IT
            foreach (self::$adminEmailRecipients['it'] as $address) {
                $filteredRecipients[] = $address;
            }
        }

        $recipientList = implode("', '", $filteredRecipients);
        if (count($filteredRecipients) == 1) {
            $body .= "\n\n(this email was ONLY sent to '{$recipientList}')";
        } else {
            $body .= "\n\n(this email was sent to '{$recipientList}')";
        }
        try {
            foreach ($filteredRecipients as $emailAddress) {
                mail($emailAddress, $subject, $body);
            }
        } catch (Exception $ex) {
            $this->logException($ex, "Exception sending email");
            $this->log($body, LOG_ERR);
        }
    }

    public function sendAdminHTMLEmail($subject, $body, $recipients = [])
    {
        if ($this->getDebugMode()) {
            // Always override email recipients in debug mode
            $recipients = ["it"];
        }

        $filteredRecipients = [];
        if (!is_array($recipients)) {
            if (!empty($recipients)) {
                $recipients = [$recipients];
            } else {
                $recipients = [];
            }
        }

        foreach($recipients as $recipient) {
            if (array_key_exists($recipient, self::$adminEmailRecipients)) {
                foreach (self::$adminEmailRecipients[$recipient] as $address) {
                    $filteredRecipients[] = $address;
                }
            } elseif(stripos($recipient, "@") !== false) {
                $filteredRecipients[] = $recipient;
            }
        }

        if (empty($filteredRecipients)) {
            // Default is to always send to IT
            foreach (self::$adminEmailRecipients['it'] as $address) {
                $filteredRecipients[] = $address;
            }
        }

        $message = "<html><head><title>{$subject}</title></head><body>";
        $message .= $body;

        $recipientList = implode("', '", array_keys($filteredRecipients));
        if (count($filteredRecipients) == 1) {
            $message .= "<p>(this email was ONLY sent to '{$recipientList}')</p>";
        } else {
            $message .= "<p>(this email was sent to '{$recipientList}')</p>";
        }

        $message .= "</body></html>";

        $headers[] = 'MIME-Version: 1.0';
        $headers[] = "Content-type: text/html; charset=utf-8";
        $headers[] = "Content-Transfer-Encoding: base64";

        try {
            mail(
                implode(", ", $filteredRecipients),
                "=?UTF-8?B?" . base64_encode($subject) . "?=",
                base64_encode($message),
                implode("\r\n", $headers)
            );
        } catch (Exception $ex) {
            $this->logException($ex, "Exception sending email");
            $this->log($body, LOG_ERR);
        }
    }

    public function log($message, $logType = LOG_DEBUG, $filename = self::DEFAULT_LOG_FILENAME)
    {
        if ($this->getDebugMode()) {
            echo "{$message}\n";
        }
        Mage::log("{$message}", $logType, $filename, true);
    }

    public function logException($exception, $messages = [])
    {
        if (!is_array($messages)) {
            $messages = [$messages];
        }

        $messages[] = (string)$exception;

        $this->log(trim(implode("\n", $messages), " \t\n\r"), LOG_ERR);
    }

    public function checkIP()
    {
        $IPs = [
            '127.0.0.1', // Localhost
            '5.150.194.196', // Office GBG
            '77.72.102.100', // Office Sthlm
            '94.247.172.162', // GleSYS firewall
            '94.247.172.163', // api1
            '10.30.0.[0-9]{1,3}', // Internal network
        ];

        $remotes = ($_SERVER['HTTP_X_FORWARDED_FOR'] ?? '')
                 . ',' . ($_SERVER['REMOTE_ADDR'] ?? '');

        foreach ($IPs as $IP) {
            if (preg_match("/({$IP})/", $remotes, $m)) {
                return;
            }
        }

        $this->getResponse()->setHttpResponseCode(200);
        die("<p><a href=\"http://www.youtube.com/watch?v=RfiQYRn7fBg\">Ah ah ah, you didn't say the magic word...</a></p>");

    }

}

Awardit_Integration_Helper_Data::$basePath = Mage::getBaseDir("log").DS.'integration'.DS;
