<?php

use Psr\Log\LoggerInterface;

class Awardit_Integration_Model_Producthandler {

    protected readonly LoggerInterface $log;
    protected bool $_saveStatus = true;
    protected array $_blacklistedAttributes = [];

    public function __construct() {
        $this->log = new Awardit_Magento_Logger("integration");
    }

    /**
     * @return array
     */
    public function productHandler()
    {
        $data = [
            "data" => "",
            "status" => 0
        ];

        if (!empty($_POST["data"])) {
            $input = json_decode($_POST["data"], true);

            try {

                if (is_array($input)) {

                    if (!empty($input["action"])) {
                        switch ($input["action"]) {
                            case "getStores":
                            case "getList":
                                $sqlQuery = "SELECT T3.name, T1.website_id, T2.group_id, T3.store_id FROM core_website T1 JOIN core_store_group T2 ON T2.website_id = T1.website_id JOIN core_store T3 ON T3.group_id = T2.group_id WHERE T1.website_id > 0 ORDER BY T3.name";
                                $data["data"] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery);
                                $data["status"] = 1;
                                $data["instance"] = $input["instance"];
                                $data["direction"] = $input["direction"];
                                break;

                            case "getData":
                                $data = $this->getDataFromProduct($input);
                                break;

                            case "getCategories":
                                $data = $this->getCategoriesForStore($input);
                                break;

                            case "setProductsInCategory":
                                $data = $this->setProductsInCategory($input);
                                break;

                            case "setData":
                                $data = $this->setDataOnProduct($input);
                                break;

                            case "updateProductData":
                                $data = $this->updateProductData($input);
                                break;

                            case "purgeProducts":
                                // ($doIndex = false, $doEcho = false, $doLog = false)
                                $data = $this->purgeProducts(false, false, true);
                                break;

                            case "resetProduct":
                                $data = $this->resetProduct($input);
                                break;

                            case "updateItem":
                                if (!empty($input["updates"])) {

                                    $data["data"] = [];

                                    // Update stock
                                    if (!empty($input["updates"]["action1"])) {
                                        $data["data"]["action1"] = 1;
                                        $data["instance"] = $input["instance"];
                                        $data["status"] = 1;
                                    }

                                    // Update status
                                    if (!empty($input["updates"]["action2"])) {
                                        $data["data"]["action2"] = 2;
                                        $data["instance"] = $input["instance"];
                                        $data["status"] = 1;
                                    }

                                    // Update attributes
                                    if (!empty($input["updates"]["action3"])) {
                                        $data["data"]["action3"] = 3;
                                        $data["instance"] = $input["instance"];
                                        $data["status"] = 1;
                                    }
                                }
                                break;

                            case "getQuotesForCustomer":
                                $data = $this->getQuotesForCustomer($input);
                                break;

                            default:
                                break;
                        }
                    } else {
                        $data["status"] = 1;
                        $data["message"] = "Missing action\n";
                    }
                } else {
                    $data["status"] = 1;
                    $data["message"] = "Empty data\n";
                }
            } catch (Exception $exception) {
                $data["status"] = 1;
                $data["message"] = $exception->getMessage() . "\n";
            }
        } else {
            $data["status"] = 1;
            $data["message"] = "Missing input\n";
        }

        return $data;
    }

    /**
     * @return array
     */
    public function scanAttributes()
    {
        $data = [
            "data" => "",
            "status" => 0
        ];

        try {
            // Fetch all attribute sets from this instance
            $sqlQuery1 = "SELECT * FROM eav_attribute_set WHERE entity_type_id = :typeId";
            $result = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery1, ["typeId" => Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId() ]);

            if (!empty($result)) {

                $instanceId = Mage::helper("integration")->getInstanceId();
                $attributeQty = count($result);
                $data["message"] = "Current attribute qty: {$attributeQty}\n";

                // Prepare translation table for update
                $sqlQuery2 = "UPDATE attribute_set_translations SET sync_flag = 1 WHERE instance_id = {$instanceId}";
                $affectedRows2 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery2);
                $data["message"] .= "Previous attribute qty: {$affectedRows2}\n";

                // Create string with data to insert
                $sqlQuery3 = "INSERT INTO attribute_set_translations (attribute_name, instance_id, attribute_id, sync_flag) VALUES\n";
                $values = [];
                foreach ($result as $row) {
                    $safeName = addslashes($row["attribute_set_name"]);
                    $values[] = "('{$safeName}', {$instanceId}, {$row["attribute_set_id"]}, NULL)";
                }
                $sqlQuery3 .= implode(",\n", $values);
                $sqlQuery3 .= "\nON DUPLICATE KEY UPDATE attribute_id = VALUES(attribute_id), sync_flag = NULL";
                $affectedRows3 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery3);
                $data["message"] .= "Uppdated attribute rows: {$affectedRows3}\n";

                // Check to see if there was any update done.
                // If nothing was updated, there must be some kind of problem
                if ($affectedRows3 > 0) {

                    // Remove old attribute sets previously set in preparation
                    $sqlQuery4 = "DELETE FROM attribute_set_translations WHERE instance_id = {$instanceId} AND sync_flag = 1";
                    $affectedRows4 = Mage::getSingleton("core/resource")->getConnection("integration_write")->exec($sqlQuery4);
                    $data["message"] .= "Removed old attributes qty: {$affectedRows4}\n";
                } else {
                    $data["status"] = 1;
                    $data["message"] .= "No updated attribute rows, did something go wrong?\n";
                }
            } else {
                $data["status"] = 1;
                $data["message"] = "No attributes were found?!\n";
            }

        } catch (Exception $exception) {
            $data["status"] = 1;
            $data["message"] = $exception->getMessage() . "\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getDataFromProduct($input)
    {
        $data = [];

        if (!empty($input["store_id"]) && is_numeric($input["store_id"])) {
            $storeId = intval($input["store_id"]);
            if (!empty($input["sku"])) {
                $sku = $input["sku"];
                $productid = Mage::getModel("catalog/product")->getIdBySku($sku);
                if (!empty($productid)) {
                    $product = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productid);
                    $storeIds = $product->getStoreIds();
                    if (in_array($storeId, $storeIds)) {
                        if ($product && $product->getSku() == $sku) {
                            $data = [
                                "status" => 1,
                                "sku" => $sku,
                                "store_id" => $storeId,
                                "store_ids" => $product->getStoreIds(),
                                "attribute_set_id" => $product->getAttributeSetId(),
                                "type_id" => $product->getData("type_id"),
                                "remote_instance" => Mage::helper("integration")->getInstanceId(),
                                "attributes" => [],
                                "related" => [],
                                "gallery" => []
                            ];

                            // Fetch attribute data
                            $attributesToExport = Mage::helper("integration")->getAttributesForAttributeSet($product->getAttributeSetId());
                            foreach ($attributesToExport as $attributeCode => $attributeData) {
                                if ($attributeData["frontend_input"] == "select") {
                                    $data["attributes"][$attributeCode] = $product->getAttributeText($attributeCode);
                                } else {
                                    $data["attributes"][$attributeCode] = $product->getData($attributeCode);
                                }
                            }

                            // Fetch data for related products
                            $relatedCollection = $product->getRelatedProductCollection();
                            foreach ($relatedCollection as $item) {
                                $data["related"][] = [
                                    "id" => $item->getId(),
                                    "sku" => $item->getSku(),
                                    "position" => $item->getPosition()
                                ];
                            }

                            // Fetch gallery data
                            $galleryData = $product->getData("media_gallery");
                            if (!empty($galleryData["images"])) {
                                $data["gallery"] = $galleryData["images"];
                                foreach (array_keys($data["gallery"]) as $index) {
                                    $data["gallery"][$index]["url"] = Mage::getBaseUrl(Mage_Core_Model_Store::URL_TYPE_MEDIA, true) . "catalog/product" . $data["gallery"][$index]["file"];
                                    $data["gallery"][$index]["tags"] = [];

                                    if ($product->getImage() == $data["gallery"][$index]["file"]) {
                                        $data["gallery"][$index]["tags"][] = "image";
                                    }
                                    if ($product->getSmallImage() == $data["gallery"][$index]["file"]) {
                                        $data["gallery"][$index]["tags"][] = "small_image";
                                    }
                                    if ($product->getThumbnail() == $data["gallery"][$index]["file"]) {
                                        $data["gallery"][$index]["tags"][] = "thumbnail";
                                    }
                                }
                            }

                            // Fetch data for configurable products
                            if ($product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE) {
                                $productAttributesOptions = $product->getTypeInstance(true)->getConfigurableOptions($product);
                                if (!empty($productAttributesOptions) && count($productAttributesOptions) > 0) {
                                    $data["configurable"] = [];
                                    foreach ($productAttributesOptions as $productAttributeOption) {
                                        $data["configurable"][] = $productAttributeOption;
                                    }
                                }
                                $data["configurable_attributes"] = $product->getTypeInstance()->getConfigurableAttributesAsArray($product);
                            }
                        } else {
                            $data["status"] = 1;
                            $data["message"] = "Unable to load product #{$sku} at source\n";
                        }
                    } else {
                        $data["status"] = 1;
                        $data["message"] = "Product #{$sku} does not belong to source store #{$storeId}\n";
                    }
                } else {
                    $data["status"] = 1;
                    $data["message"] = "Product #{$sku} does not exist at source\n";
                }
            } else {
                $data["status"] = 1;
                $data["message"] = "Missing sku.\n";
            }
        } else {
            $data["status"] = 1;
            $data["message"] = "Unknown store #{$input["store_id"]}\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @return array
     */
    public function setDataOnProduct($input)
    {
        $data = [];
        $testMode = false;
        $isUpdated = false;
        $isNew = false;

        $setConfigurables = empty($input["set_configurables"]) ? false : true;
        $setAttributes = empty($input["set_attributes"]) ? false : true;
        $setImages = empty($input["set_images"]) ? false : true;
        $setName = empty($input["set_name"]) ? false : true;

        if (!empty($input["store_id"]) && is_numeric($input["store_id"])) {
            $storeId = intval($input["store_id"]);

            if (!empty($input["sku"])) {
                $sku = $input["sku"];
                $productId = Mage::getModel("catalog/product")->getIdBySku($sku);

                // Check if product exists
                if (empty($productId)) {

                    // If not, see if it is a configurable product, if so, create it
                    if ($setConfigurables && !empty($input["configurable"])) {
                        $product = Mage::getModel("catalog/product");
                        $productId = $this->createEmptyConfigurableProduct($input, $product);
                        $storeIds = [$storeId];
                        $isNew = true;
                    }
                } else {

                    // Load product in global scope first
                    $product = Mage::getModel("catalog/product")->load($productId);
                    $storeIds = $product->getStoreIds();
                }

                if (!empty($productId)) {

                    // begin: Handle related products
                    if (!empty($input["related"])) {
                        $relatedLinkData = [];
                        foreach ($input["related"] as $relatedItem) {
                            $relatedItemId = $product->getIdBySku($relatedItem["sku"]);
                            if (!empty($relatedItemId)) {
                                $relatedLinkData[$relatedItemId] = [
                                    "position" => $relatedItem["position"]
                                ];
                            }
                        }
                        if (!empty($relatedLinkData)) {
                            $product->setRelatedLinkData($relatedLinkData);
                            $isUpdated = true;
                        }
                    }
                    // end: Handle related products

                    if (in_array($storeId, $storeIds)) {
                        if ($product && $product->getSku() == $sku) {
                            if (!empty($input["attributes"])) {

                                $input["local_attribute_set_id"] = $product->getAttributeSetId();

                                // begin: Handle attribute set and status
                                if ($setAttributes && !$isNew && !empty($input["remote_instance"]) && !empty($input["attribute_set_id"])) {
                                    $newAttributeSetId = Mage::helper("integration")->translateAttributeSetId($input["remote_instance"], $input["attribute_set_id"]);

                                    if ($newAttributeSetId != $product->getAttributeSetId()) {
                                        // Need to save product when changing attribute set
                                        $isUpdated = true;

                                        if ($testMode) {
                                            $input["local_attribute_set_id"] = $newAttributeSetId;
                                        } else {
                                            $product->setAttributeSetId($newAttributeSetId);
                                            $input["local_attribute_set_id"] = $product->getAttributeSetId();
                                        }
                                    }

                                    if ($this->_saveStatus) {
                                        $currentStatus = $product->getStatus();
                                        $newStatus = Mage::helper("integration")->getSystemAttributeOptionValue("status", $input["attributes"]["status"]);
                                        if ($newStatus !== false && $newStatus != $currentStatus) {
                                            $product->setStatus($newStatus);
                                            $isUpdated = true;
                                        }
                                    }
                                }
                                // end: Handle attribute set

                                // begin: Handle product type (simple, virtual, etc.)
                                if (!$isNew) {
                                    if ($setConfigurables && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE) {
                                        $isUpdated = $this->setConfigurableDataOnProduct($input, $product) ? true : $isUpdated;
                                    } elseif ($setAttributes && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL && $input["type_id"] === Mage_Catalog_Model_Product_Type::TYPE_SIMPLE) {
                                        $product->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_SIMPLE);
                                        $isUpdated = true;
                                    } elseif ($setAttributes && $product->getData("type_id") === Mage_Catalog_Model_Product_Type::TYPE_SIMPLE && $input["type_id"] === Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL) {
                                        $product->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_VIRTUAL);
                                        $isUpdated = true;
                                    }
                                }
                                // end: Handle product type (simple, virtual, etc.)

                                // begin: Handle image gallery
                                if ($setImages && !empty($input["gallery"])) {

                                    // Only add images if product doesn't have one
                                    $galleryData = $product->getData("media_gallery");
                                    if (empty($galleryData["images"])) {
                                        $isUpdated = $this->addImageToProduct($input["gallery"], $product) ? true : $isUpdated;
                                    } else {
                                        // Product have images, skipping adding new ones
                                    }
                                }
                                // end: Handle image gallery

                                if ($isUpdated) {
                                    $isUpdated = false;
                                    if (!$testMode) {
                                        try {
                                            $product->save();
                                        } catch (Exception $e) {
                                            $this->log->error("Producthandler: Exception while saving global product: {$e->getMessage()}", [ "exception" => $e ]);
                                        }
                                    }
                                }

                                if ($setAttributes || $setName) {
                                    // Reload product in local context
                                    if ($setAttributes) {
                                        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
                                        // Set attributes on product
                                        $isUpdated = $this->setAttributesOnProduct($input, $localProduct, $testMode) ? true : $isUpdated;
                                    } elseif($setName) {
                                        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
                                        // Set name on product
                                        if ($localProduct->getName() != $input["attributes"]["name"]) {
                                            $localProduct->setName($input["attributes"]["name"]);
                                            $isUpdated = true;
                                        }
                                    }
                                    if (!$testMode && $isUpdated) {
                                        try {
                                            $localProduct->save();
                                        } catch (Exception $e) {
                                            $this->log->error("Producthandler: Exception while saving local product: {$e->getMessage()}", [ "exception" => $e ]);
                                        }
                                    }
                                }

                                $data = [
                                    "status" => 1,
                                    "sku" => $sku,
                                    "store_id" => $storeId,
                                    "instance" => $input["instance"]
                                ];
                                if (isset($e)) {
                                    $data["trace"] = $e->getTraceAsString();
                                }
                            }
                        } else {
                            $data["status"] = 1;
                            $data["message"] = "Unable to load product #{$sku} at destination\n";
                        }
                    } else {
                        // $isUpdated might be true if related products was set.
                        if ($isUpdated) {
                            if (!$testMode) {
                                $product->save();
                            }
                            $data["message"] = "Product #{$sku} does not belong to destination store #{$storeId} but related products was saved anyway.\n";
                        } else {
                            $data["message"] = "Product #{$sku} does not belong to destination store #{$storeId}\n";
                        }
                        $data["status"] = 1;
                    }
                } else {
                    $data["status"] = 1;
                    $data["message"] = "Product #{$sku} does not exist at destination\n";
                }
            } else {
                $data["status"] = 1;
                $data["message"] = "Missing sku.\n";
            }
        } else {
            $data["status"] = 1;
            $data["message"] = "Unknown store #{$input["store_id"]}\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @return array
     */
    public function updateProductData($input)
    {
        $isUpdated = false;
        $data = [
            "status" => 1
        ];

        if (empty($input["store_id"]) || !is_numeric($input["store_id"])) {
            $data["message"] = "Unknown store #{$input["store_id"]}\n";
            return $data;
        }
        $storeId = intval($input["store_id"]);

        if (empty($input["sku"])) {
            $data["message"] = "Missing sku.\n";
            return $data;
        }
        $sku = $input["sku"];
        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (empty($productId)) {
            $data["message"] = "Product #{$sku} does not exist at destination\n";
        }

        $localProduct = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
        foreach ($input["attributes"] as $attributeCode => $attributeValue) {
            if ($localProduct->getData($attributeCode) != $attributeValue) {
                $localProduct->setData($attributeCode, $attributeValue);
                $isUpdated = true;
            }

        }
        if ($isUpdated) {
            try {
                $localProduct->save();
                $data["message"] = "updated\n";
            } catch (Exception $e) {
                $this->log->error($e);
                $data["status"] = 0;
                $data["message"] = "error!\n";
                $data["message"] .= "Exception while saving local product:\n";
                $data["message"] .= $e->getMessage();
            }
        } else {
            $data["message"] = "no update needed\n";
        }

        return $data;
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @param bool $testMode
     * @return bool
     */
    public function setAttributesOnProduct($input, &$product, $testMode = false)
    {
        $isUpdated = false;
        $attributesToImport = Mage::helper("integration")->getAttributesForAttributeSet($input["local_attribute_set_id"]);

        foreach ($input["attributes"] as $attributeCode => $attributeData) {

            // Skip "status" in local context. Only set this (elsware) for new configurable products.
            if ($attributeCode == "status") {
                continue;
            }

            // Skip blacklisted attributes, if specified
            if (in_array($attributeCode, $this->_blacklistedAttributes)) {
                continue;
            }

            if (array_key_exists($attributeCode, $attributesToImport)) {
                if ($attributesToImport[$attributeCode]["frontend_input"] == "select") {

                    // Check if attribute is system och custom.
                    if (empty($attributesToImport[$attributeCode]["source_model"]) || stripos($attributesToImport[$attributeCode]["source_model"], "eav/entity_attribute_source") === 0) {

                        // Handle custom made attributes here
                        if ($product->getAttributeText($attributeCode) != $attributeData) {
                            $optionId = Mage::helper("integration")->getOptionId($attributeCode, $attributeData, !$testMode);
                            if ($optionId !== false) {
                                if (!$testMode) {
                                    $product->setData($attributeCode, $optionId)->getResource()->saveAttribute($product, $attributeCode);
                                    $isUpdated = true;
                                }
                            } else {
                                // Unable to get or create optionId. Just skip this attribute.
                            }
                        }
                    } else {

                        // Handle system made attributes here
                        $optionValue = Mage::helper("integration")->getSystemAttributeOptionValue($attributeCode, $attributeData);
                        if (!empty($optionValue)) {
                            if (!$testMode) {
                                $product->setData($attributeCode, $optionValue)->getResource()->saveAttribute($product, $attributeCode);
                                $isUpdated = true;
                            }
                        }
                    }
                } else {
                    if ($product->getData($attributeCode) != $attributeData) {
                        if (!$testMode) {
                            $product->setData($attributeCode, $attributeData)->getResource()->saveAttribute($product, $attributeCode);
                            $isUpdated = true;
                        }
                    }
                }
            }
        }

        return $isUpdated;
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @return bool
     */
    public function addImageToProduct($input, &$product)
    {
        $isUpdated = false;

        try {
            $filePath = Mage::getBaseDir("media") . DS . "import";
            if (!is_dir($filePath) && !mkdir($filePath, 0777, true)) {
                $this->log->error("Producthandler: Unable to create directory {$filePath}");
                return $isUpdated;
            }

            foreach ($input as $imageData) {
                $filenameParts = explode("/", $imageData["file"]);
                $filename = $filenameParts[count($filenameParts) - 1];
                $filePathAndName = $filePath . DS . $filename;

                $rawData = file_get_contents($imageData["url"]);
                if ($rawData !== false) {
                    $bytes = file_put_contents($filePathAndName, $rawData);
                    if ($bytes !== false) {

                        $product->addImageToMediaGallery($filePathAndName, $imageData["tags"], true);
                        $gallery = $product->getData("media_gallery");
                        $lastImage = array_pop($gallery["images"]);

                        $lastImage["label"] = $imageData["label"];
                        $lastImage["position"] = $imageData["position"];
                        $lastImage["disabled"] = $imageData["disabled"];
                        $lastImage["label_default"] = $imageData["label_default"];
                        $lastImage["position_default"] = $imageData["position_default"];
                        $lastImage["disabled_default"] = $imageData["disabled_default"];

                        array_push($gallery["images"], $lastImage);
                        $product->setData("media_gallery", $gallery);
                        $isUpdated = true;
                    } else {
                        $this->log->error("Producthandler: Unable to create file {$filePathAndName}");
                    }
                } else {
                    $this->log->error("Producthandler: Unable to read data from {$imageData["url"]}");
                }
                unset($rawData);
            }
        } catch (Exception $e) {
            $this->log->error("Exception while adding image(s): {$e->getMessage()}", [ "exception" => $e ]);
        }

        return $isUpdated;
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @return null|int
     */
    public function createEmptyConfigurableProduct($input, &$product)
    {
        $newAttributeSetId = Mage::helper("integration")->translateAttributeSetId($input["remote_instance"], $input["attribute_set_id"]); // TODO: fix call
        $websiteId = Mage::helper("integration")->getMagentoWebsiteIdByStoreId($input["store_id"]);

        try {
            $product
                ->setAttributeSetId($newAttributeSetId)
                ->setTypeId(Mage_Catalog_Model_Product_Type::TYPE_CONFIGURABLE)
                ->setVisibility(Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH)
                ->setWebsiteIds([ $websiteId ])
                ->setStoreIds([ $input["store_id"] ])
                ->setCreatedAt(strtotime("now"))
                ->setSku($input["sku"])
                ->setPrice(0)
                ->setName($input["attributes"]["name"])
                ->setDescription($input["attributes"]["description"])
                ->setShortDescription($input["attributes"]["short_description"])
                ->setTaxClassId(0) // TODO: set correct VAT, use 0% as default for now.
            ;

            $configurableAttributesData = [];
            foreach ($input["configurable_attributes"] as $row) {
                $configurableAttributesData[] = [
                    "label" => $row["label"],
                    "values" => [],
                    "attribute_id" => Mage::getSingleton("eav/config")->getAttribute("catalog_product", $row["attribute_code"])->getId(),
                    "attribute_code" => $row["attribute_code"],
                    "frontend_label" => $row["frontend_label"],
                    "store_label" => $row["store_label"],
                ];
            }
            $product->setCanSaveConfigurableAttributes(true);
            $product->setConfigurableAttributesData($configurableAttributesData);
            $this->setConfigurableDataOnProduct($input, $product, true);

            $product->save();

        } catch (Exception $e) {
            $this->log->error("Unable to create new empty configurable product: {$e->getMessage()}", [ "exception" => $e ]);

            return null;
        }

        return $product->getId();
    }

    /**
     * @param array $input
     * @param Mage_Catalog_Model_Product $product
     * @param bool $isNew
     * @return bool
     */
    public function setConfigurableDataOnProduct($input, &$product, $isNew = false)
    {
        $data = [
            "used_product_attribute_ids" => [],
            "attributes" => [],
            "status" => Mage_Catalog_Model_Product_Status::STATUS_DISABLED
        ];

        $optionValue = Mage::helper("integration")->getSystemAttributeOptionValue("status", $input["attributes"]["status"]);
        if ($optionValue !== false) {
            $data["status"] = $optionValue;
        }

        try {
            if (!empty($input["configurable"])) {

                foreach ($input["configurable"] as $attributes) {

                    $attributeCode = null;
                    $attributeId = null;

                    foreach ($attributes as $options) {
                        $sku = $options["sku"];

                        if (empty($attributeCode)) {
                            $attributeCode = $options["attribute_code"];
                            $attributeId = Mage::getResourceModel("eav/entity_attribute")->getIdByCode("catalog_product", $attributeCode);

                            if (!empty($attributeId)) {
                                $data["attributes"][$attributeId] = [
                                    "attribute_code" => $attributeCode,
                                    "attribute_id" => $attributeId,
                                    "options" => []
                                ];
                                $data["used_product_attribute_ids"][] = $attributeId;
                            }
                        }

                        if (!empty($attributeId) && !empty($attributeCode)) {
                            $optionLabel = $options["option_title"];
                            $optionId = Mage::helper("integration")->getOptionId($attributeCode, $optionLabel, false);

                            if (!empty($optionId)) {
                                $data["attributes"][$attributeId]["options"][$optionId] = [
                                    "sku" => $sku,
                                    "option_id" => $optionId,
                                    "option_label" => $optionLabel,
                                    "pricing_is_percent" => $options["pricing_is_percent"],
                                    "pricing_value" => $options["pricing_value"]
                                ];
                            }
                        }
                    }
                }
            }

            if (!empty($data["used_product_attribute_ids"]) && !empty($data["attributes"])) {
                if ($isNew) {
                    $product->getTypeInstance()->setUsedProductAttributeIds($data["used_product_attribute_ids"]);
                }
                $configurableProductsData = [];
                foreach ($data["attributes"] as $attributeId => $attributeData) {
                    foreach ($attributeData["options"] as $optionId => $optionData) {
                        $productId = Mage::getModel("catalog/product")->getIdBySku($optionData["sku"]);
                        $configurableProductsData[$productId] = [
                            0 => [
                                "label" => $optionData["option_label"],
                                "attribute_id" => $attributeId,
                                "value_index" => $optionData["option_id"],
                                "is_percent" => $optionData["pricing_is_percent"],
                                "pricing_value" => $optionData["pricing_value"]
                            ]
                        ];
                    }
                }

                $product->setConfigurableProductsData($configurableProductsData);
                if ($this->_saveStatus) {
                    $product->setStatus($data["status"]); // Set status in global scope
                }

                return true;
            }
        } catch (Exception $e) {
            $this->log->error("Exception while setting configurable data on product: {$e->getMessage()}", [ "exception" => $e ]);
        }
        return false;
    }

    /**
     * @param array $input
     * @return array
     */
    public function getCategoriesForStore($input)
    {
        $storeId = null;
        $data = $input;
        $data["data"] = [];

        if (!empty($input["store"])) {
            $storeId = intval($input["store"]);
        }

        if (!empty($storeId)) {
            $sqlQuery1 = "
                SELECT
                    g.root_category_id
                FROM core_store s
                JOIN core_store_group g ON g.website_id = s.website_id
                WHERE s.store_id = :storeId
            ";
            $params1 = ["storeId" => $storeId];
            $rootCategoryId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery1, $params1);

            $sqlQuery2 = "
                SELECT
                    a.attribute_id
                FROM eav_entity_type t
                JOIN eav_attribute a ON a.entity_type_id = t.entity_type_id
                WHERE t.entity_type_code = 'catalog_category' AND a.attribute_code = 'name'
            ";
            $attributeId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchOne($sqlQuery2);

            $sqlQuery3 = "
                SELECT
                    tmp.entity_id AS category_id,
                    COALESCE (tmp.local_value, tmp.global_value) AS name,
                    tmp.position,
                    tmp.`level`,
                    tmp.parent_id,
                    tmp.path
                FROM
                (
                    SELECT
                        IF (v.store_id = 0, v.value, null) AS global_value,
                        IF (v.store_id > 0, v.value, null) AS local_value,
                        c.entity_id,
                        c.position,
                        c.`level`,
                        c.parent_id,
                        c.path
                    FROM catalog_category_entity c
                    JOIN catalog_category_entity_varchar v ON v.entity_id = c.entity_id AND v.attribute_id = :attributeId
                    WHERE (c.path = :path1 OR c.path LIKE :path2) AND (v.store_id = 0 OR v.store_id = :storeId)
                ) tmp
                GROUP BY tmp.entity_id
                ORDER BY tmp.path #tmp.level, tmp.position, tmp.parent_id
            ";
            $params3 = [
                "attributeId" => $attributeId,
                "path1" => "1/{$rootCategoryId}",
                "path2" => "1/{$rootCategoryId}/%",
                "storeId" => $storeId
            ];
            $data["data"] = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery3, $params3);
            if (!empty($data["data"])) {
                $data["status"] = 1;
            }
        }

        return $data;

    }

    /**
     * @param array $input
     * @return array
     */
    public function setProductsInCategory($input)
    {
        $returnData = [
            "status" => null,
            "message" => ""
        ];

        if (!empty($input["store_id"])) {
            $storeId = intval($input["store_id"]);
        }
        // Had problems with local context, using global for now.
        $storeId = 0;

        if (!empty($input["category_id"])) {
            $categoryId = intval($input["category_id"]);
        }

        $overwritePosition = !empty($input["position"]);

        try {
            if (!empty($input["skus"])) {

                $model = Mage::getSingleton("catalog/category_api");
                $placeholders = [];
                $skuToId = [];

                // Build parameter list
                foreach ($input["skus"] as $skuIndex => $sku) {
                    $placeholders[":param{$skuIndex}"] = $sku;
                }

                // Use parameter list to fetch product id
                if (!empty($placeholders)) {
                    $placeholderString = implode(",", array_keys($placeholders));
                    $sqlQuery = "SELECT p.sku, p.entity_id FROM catalog_product_entity p WHERE p.sku IN ({$placeholderString})";
                    $skuToId = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery, $placeholders);
                }

                // We now have a list of product id:s for all sku:s.
                if (!empty($skuToId)) {

                    $oldList = $model->assignedProducts($categoryId, $storeId);

                    // Check if we are supposed to overwrite current list of products in category
                    if (!empty($oldList) && !empty($input["overwrite"])) {
                        $returnData["message"] .= "Clearing all products from category";
                        foreach($oldList as $oldEntry) {
                            $model->removeProduct($categoryId, $oldEntry["product_id"]);
                        }
                        $returnData["message"] .= " - OK\n";
                        $oldList = [];
                    }

                    // Loop through list of sku/id and assign product to category
                    $position = 0;
                    foreach ($placeholders as $sku) {
                        if (array_key_exists($sku, $skuToId)) {
                            $productId = intval($skuToId[$sku]["entity_id"]);
                            if ($overwritePosition) {
                                $position++;
                            }
                            if (!empty($oldList)) {
                                foreach($oldList as $oldEntry) {
                                    if ($productId == $oldEntry["product_id"]) {
                                        $returnData["message"] .= "Updating product [{$sku}] to position {$position}";
                                        $model->updateProduct($categoryId, $productId, $position);
                                        $returnData["message"] .= " - OK\n";
                                        continue 2;
                                    }
                                }
                            }
                            try {
                                $returnData["message"] .= "Adding product [{$sku}] to position {$position}";
                                $model->assignProduct($categoryId, $productId, $position);
                                $returnData["message"] .= " - OK\n";
                            } catch (Exception $exception) {
                                $returnData["message"] .= " - Failed!\n";
                                if ($exception->getMessage() == "not_exists") {
                                    $returnData["message"] .= "Product [{$sku}] is not active in selected store!\n";
                                    $returnData["status"] = 0;
                                    continue;
                                } else {
                                    $returnData["message"] .= "Exception: " . $exception->getMessage() . "\n";
                                    $returnData["message"] .= $exception->getTraceAsString();
                                    $returnData["status"] = 0;
                                    break;
                                }
                            }
                        } else {
                            $returnData["message"] .= "Product [{$sku}] does not exist in selected Magento instance!\n";
                            $returnData["status"] = 0;
                        }
                    }
                } else {
                    $returnData["message"] .= "Found no matching products for wanted list of SKUs\n";
                }
            } else {
                $returnData["message"] .= "Missing list of SKUs\n";
            }
        } catch (Exception $e) {
            $returnData["message"] .= "Exception: " . $e->getMessage() . "\n";
            $returnData["message"] .= $e->getTraceAsString();
            $returnData["status"] = 0;

            $this->log->error("Exception while assigning products to category: {$e->getMessage()}", [ "exception" => $e ]);
        }

        if ($returnData["status"] === null) {
            $returnData["status"] = 1;
        }

        if (empty($returnData["message"])) {
            $returnData["message"] = "Nothing done!";
        }

        return $returnData;

    }

    /**
     * @param array $input
     * @return array
     */
    public function getQuotesForCustomer($input)
    {
        $data = [];

        if (!empty($input["customer_id"]) && is_numeric($input["customer_id"])) {
            $customerId = intval($input["customer_id"]);
            $sqlQuery = "
                SELECT
                    q.entity_id AS quote_id,
                    q.store_id,
                    cs.`name` AS store_name,
                    ce.email,
                    q.created_at,
                    qi.sku,
                    qi.`name`,
                    q.grand_total,
                    qp.additional_information
                FROM sales_flat_quote q
                JOIN sales_flat_quote_payment qp ON qp.quote_id = q.entity_id
                JOIN sales_flat_quote_item qi ON qi.quote_id = q.entity_id
                JOIN core_store cs ON cs.store_id = q.store_id
                JOIN customer_entity ce ON ce.entity_id = q.customer_id
                WHERE qp.additional_information IS NOT NULL AND q.store_id IN (6,7,8,9) AND q.customer_id = :customerId
            ";
            $data["status"] = 1;
            $data["message"] = "Got info for customer #{$customerId}";
            $data["data"] = [];
            $quotes = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, [ "customerId" => $customerId ]);
            foreach ($quotes as $quote) {
                $additionalInformation = unserialize($quote["additional_information"]);
                if (!empty($additionalInformation["intent_id"])) {
                    $data["data"][] = [
                        "quote_id" => intval($quote["quote_id"]),
                        "store_id" => intval($quote["store_id"]),
                        "store_name" => $quote["store_name"],
                        "email" => $quote["email"],
                        "created_at" => $quote["created_at"],
                        "sku" => $quote["sku"],
                        "item_name" => $quote["name"],
                        "grand_total" => floatval($quote["grand_total"]),
                        "intent_id" => $additionalInformation["intent_id"]
                    ];
                }
            }
        } else {
            $data["status"] = 1;
            $data["message"] = "Unknown customer\n";
        }

        return $data;
    }

    public function purgeProducts($doIndex = false, $doEcho = false, $doLog = false)
    {
        $stats = [
            "initialQty" => 0,
            "filteredQty" => 0,
            "totalAffectedRows" => 0,
            "updatedProductsQty" => 0,
            "actions" => []
        ];

        $msg = "Product purge started.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) $this->log->debug($msg);

        // Fetch list of SKU:s from Magento where SKU matches those from Visma (only simple and virtual products) (matching example: '12345' or '12345-Strukt')
        $sqlQuery1 = "SELECT cpe.sku, cpe.entity_id FROM catalog_product_entity cpe WHERE cpe.type_id IN ('simple','virtual') AND cpe.sku REGEXP '^[0-9]{5}(-[a-zA-Z]+)?$'";
        $magentoList = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery1);
        if (empty($magentoList)) {
            $msg = "Found no products matching SKU filter";
            if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg;
            if ($doLog) $this->log->debug($msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        }
        $stats["initialQty"] = count($magentoList);
        $msg = "Found {$stats["initialQty"]} mathing products to check in Visma.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) $this->log->debug($msg);

        // Split result list in batches
        $batches = array_chunk($magentoList, 100, true);
        foreach ($batches as $batch) {
            // Fetch status from Visma
            $inStr = implode("', '", array_keys($batch));
            $sqlQuery2 = "SELECT Prod.ProdNo AS sku, Prod.Gr2 AS visma_status FROM Prod WHERE Prod.ProdNo IN ('{$inStr}')";
            $vismaList = Mage::helper("integration")->getVismaDB()->query($sqlQuery2, array_keys($batch));
            foreach ($vismaList as $row) {
                // If Visma status is ok, remove product from list
                if ($row["visma_status"] > 0 && $row["visma_status"] < 4 && array_key_exists($row["sku"], $magentoList)) {
                    unset($magentoList[$row["sku"]]);
                }
            }
        }
        $stats["filteredQty"] = count($magentoList);

        if (empty($magentoList)) {
            $msg = "Found no products with status >= 4 in Visma";
            if ($doLog) $this->log->debug($msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        } else {
            $msg = "Checked in Visma and found {$stats["filteredQty"]} disabled products.";
            $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
            if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
            if ($doLog) $this->log->debug($msg);
        }

        // Fetch info for attribute 'status'
        $sqlQuery3 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => "status"
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery3, $params);
        if (empty($eava["attribute_id"])) {
            $msg = "Magento is missing attribute 'status'?!";
            if ($doLog) $this->log->debug($msg);
            return [
                "status" => 1,
                "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                "debug" => $stats
            ];
        }

        $sqlQuery4 = "DELETE cpex.* FROM catalog_product_entity_int cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";
        $stmt = Mage::getSingleton("core/resource")->getConnection("core_write")->prepare($sqlQuery4);

        foreach ($magentoList as $row) {
            $stmt->bindValue("typeId", $params["typeId"]);
            $stmt->bindValue("attributeId", intval($eava["attribute_id"]));
            $stmt->bindValue("productId", intval($row["entity_id"]));
            $stmt->execute();
            $affectedRows = $stmt->rowCount();
            $stats["totalAffectedRows"] += $affectedRows;

            $globalProduct = Mage::getModel("catalog/product")->load($row["entity_id"]);
            if ((!$doIndex && $affectedRows > 0) || $globalProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
                $globalProduct->setStatus(Mage_Catalog_Model_Product_Status::STATUS_DISABLED);
                $globalProduct->save();
                $stats["updatedProductsQty"]++;
            }
        }

        $msg = "Delete and updates done.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
        if ($doLog) $this->log->debug($msg);

        if ($doIndex && ($stats["totalAffectedRows"] > 0 || $stats["updatedProductsQty"] > 0)) {
            $indexCodes = [
                "catalog_product_price" => true,
                "catalog_product_flat" => true
            ];

            try {
                foreach ($indexCodes as $indexCode => $indexStatus) {
                    if ($indexStatus) {

                        $msg = "Running indexer for '{$indexCode}'.";
                        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
                        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
                        if ($doLog) $this->log->debug($msg);

                        $process = Mage::getModel("index/indexer")->getProcessByCode($indexCode);
                        $process->reindexAll();

                        $msg = "Index done for '{$indexCode}'.";
                        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
                        if ($doEcho) echo "[" . date("Y-m-d H:i:s") . "] " . $msg . "\n";
                        if ($doLog) $this->log->debug($msg);
                    }
                }
            } catch (Exception $e) {
                $msg = "Exception running indexer for '{$indexCode}': ".$e->getMessage();
                $this->log->error($msg, [ "exception" => $e ]);
                return [
                    "status" => 1,
                    "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
                    "debug" => $stats
                ];
            }
        }

        $msg = "Purge done. Deleted {$stats["totalAffectedRows"]} local values and disabled {$stats["updatedProductsQty"]} products.";
        $stats["actions"][] = "[" . date("Y-m-d H:i:s") . "] " . $msg;
        if ($doLog) $this->log->debug($msg);
        return [
            "status" => 1,
            "message" => ($doEcho ? "[" . date("Y-m-d H:i:s") . "] " : "") . $msg,
            "debug" => $stats
        ];

    }

    public function listDisabledProducts()
    {
        $stats = [
            "initialQty" => 0,
            "filteredQty" => 0,
            "totalLocalValues" => 0,
            "totalUpdateableProducts" => 0
        ];
        echo "Listing products from this Magento having status >= 4 in Visma.\n";

        // Fetch list of SKU:s from Magento where SKU matches those from Visma (only simple and virtual products) (matching example: '12345' or '12345-Strukt')
        $sqlQuery1 = "SELECT cpe.sku, cpe.entity_id FROM catalog_product_entity cpe WHERE cpe.type_id IN ('simple','virtual') AND cpe.sku REGEXP '^[0-9]{5}(-[a-zA-Z]+)?$'";
        $magentoList = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery1);
        if (empty($magentoList)) {
            echo "Found no products matching SKU filter.\n";
            return;
        }
        $stats["initialQty"] = count($magentoList);
        echo "Found {$stats["initialQty"]} mathing products to check in Visma.\n";

        // Split result list in batches
        $batches = array_chunk($magentoList, 100, true);
        foreach ($batches as $batch) {
            // Fetch status from Visma
            $inStr = implode("', '", array_keys($batch));
            $sqlQuery2 = "SELECT Prod.ProdNo AS sku, Prod.Gr2 AS visma_status FROM Prod WHERE Prod.ProdNo IN ('{$inStr}')";
            $vismaList = Mage::helper("integration")->getVismaDB()->query($sqlQuery2, array_keys($batch));
            foreach ($vismaList as $row) {
                // If Visma status is ok, remove product from list
                if ($row["visma_status"] > 0 && $row["visma_status"] < 4 && array_key_exists($row["sku"], $magentoList)) {
                    unset($magentoList[$row["sku"]]);
                }
            }
        }
        $stats["filteredQty"] = count($magentoList);

        if (empty($magentoList)) {
            echo "Found no products with status >= 4 in Visma.\n";
            return;
        } else {
            echo "Checked in Visma and found {$stats["filteredQty"]} disabled products.\n";
        }

        // Fetch info for attribute 'status'
        $sqlQuery3 = "SELECT eava.* FROM eav_attribute eava WHERE eava.entity_type_id = :typeId AND eava.attribute_code = :code";
        $params = [
            "typeId" => intval(Mage::getModel("eav/entity")->setType("catalog_product")->getTypeId()),
            "code" => "status"
        ];
        $eava = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchRow($sqlQuery3, $params);
        if (empty($eava["attribute_id"])) {
            echo "Magento is missing attribute 'status'?!\n";
            return;
        }

        $sqlQuery4 = "SELECT cpex.store_id, cpex.value AS localValuesQty FROM catalog_product_entity_int cpex WHERE cpex.entity_type_id = :typeId AND cpex.attribute_id = :attributeId AND cpex.store_id > 0 AND cpex.entity_id = :productId";

        foreach ($magentoList as $row) {
            $p = [
                "typeId" => $params["typeId"],
                "attributeId" => intval($eava["attribute_id"]),
                "productId" => intval($row["entity_id"])
            ];
            $localValues = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAssoc($sqlQuery4, $p);
            $localValuesQty = count($localValues);
            $stats["totalLocalValues"] += $localValuesQty;

            $globalProduct = Mage::getModel("catalog/product")->load($row["entity_id"]);
            if ($localValuesQty > 0 || $globalProduct->getStatus() == Mage_Catalog_Model_Product_Status::STATUS_ENABLED) {
                if ($localValuesQty > 0) {
                    echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' was about do be disabled and have local value in stores (" . implode(",", array_keys($localValues)) . ").\n";
                } else {
                    echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' was about do be disabled and have no local values.\n";
                }
                $stats["totalUpdateableProducts"]++;
            } else {
                echo "Product [{$globalProduct->getSku()}] '{$globalProduct->getName()}' is already disabled.\n";
            }
        }

        echo "Would have deleted {$stats["totalLocalValues"]} local values and disabled {$stats["totalUpdateableProducts"]} products.\n";
        return;
    }

    /**
     * @param array $input
     * @return array
     */
    public function resetProduct($input)
    {
        $isUpdated = false;
        $data = [
            "status" => 1
        ];

        if (empty($input["attributes"])) {
            $data["message"] = "No attributes to reset?\n";
            return $data;
        }

        if (empty($input["sku"])) {
            $data["message"] = "Missing sku.\n";
            return $data;
        }
        $sku = $input["sku"];

        $productId = Mage::getModel("catalog/product")->getIdBySku($sku);
        if (empty($productId)) {
            $data["message"] = "Product [{$sku}] does not exist at destination\n";
            return $data;
        }

        // Remove local values for supplied attributes
        foreach (array_keys($input["attributes"]) as $attributeCode) {
            $affectedRows = Mage::helper("integration")->deleteLocalAttributeValues($productId, $attributeCode);
            if ($affectedRows) {
                $isUpdated = true;
            }
        }

        // Need to remove local values before loading product, that's why we loop twice
        $globalProduct = Mage::getModel("catalog/product")->setStoreId(0)->load($productId);
        foreach ($input["attributes"] as $attributeCode => $attributeValue) {
            // We use "-" as indicator for when attribute does not need to be updated
            if ($attributeValue !== "-" && $globalProduct->getData($attributeCode) != $attributeValue) {
                $globalProduct->setData($attributeCode, $attributeValue);
                $isUpdated = true;
            }
        }

        if ($isUpdated) {
            // Save triggers calculation of upsert index
            $globalProduct->save();
        }

        return $data;
    }

}
