<?php
class Awardit_Integration_Model_Cli_Cron extends Awardit_Integration_Model_Cli {

    /**
     * @var int
     */
    const CRON_DEFAULT_ITERATION_LIMIT = 999;

    /**
     * @var string
     */
    const CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT = "+30 minutes";     // Time limit for importing products

    /**
     * @var string
     */
    const CRON_MAX_RUNNING_TIME_ORDER_PROCESSIING = "+5 minutes";   // Time limit for processing orders

    /**
     * @var string
     */
    const CRON_MAX_RUNNING_TIME_ACCRUAL_PROCESSING = "+10 minutes"; // Time limit for processing accrual

    protected ?DateTime $_maxRunningTime = null;

    public function getLimit(): int
    {
        if (parent::getLimit() != Awardit_Integration_Model_Cli::CLI_DEFAULT_ITERATION_LIMIT) {
            return parent::getLimit();
        } else {
            return Awardit_Integration_Model_Cli_Cron::CRON_DEFAULT_ITERATION_LIMIT;
        }
    }

    public function setMaxRunningTime(string $offset): void
    {
        if (!empty($offset)) {
            $this->_maxRunningTime = new DateTime(date("Y-m-d H:i:s", strtotime($offset)));
        } else {
            $this->_maxRunningTime = null;;
        }
    }

    public function getMaxRunningTime(): ?DateTime
    {
        return $this->_maxRunningTime;
    }

    // Return true while max running time is not reached OR no max running time is specified
    public function checkRunningTime(?DateTime $now = null): bool
    {
        if ($now === null) {
            $now = new DateTime("now");
        }
        if ($this->_maxRunningTime !== null) {
            return $now < $this->_maxRunningTime;
        }
        return true;
    }

    // -m=Cron -f=importProducts
    public function CLI_importProducts($param)
    {
        $start = new DateTime("now");
        $this->log->debug("Cron: Starting to import updated products.");

        $wantedParams = [ "updateStock" => "int" ];
        $extractedParams = $this->extractParams($param, $wantedParams);
        $updateStock = null;
        if (array_key_exists("updateStock", $extractedParams)) {
            $updateStock = empty($extractedParams["updateStock"]) ? false : true;
        }

        $processes = [
            "tag" => true,
            "import" => true,
        ];
        $this->_processProducts($processes, $updateStock);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        $this->log->debug("Cron: product import script ran for {$diff->format("%H:%I:%S")}");
    }

    // -m=Cron -f=exportOrders
    public function CLI_exportOrders($param)
    {
        $start = new DateTime("now");
        $this->log->debug("Cron: Starting to export all orders.");
        $this->setMaxRunningTime(null);

        $processes = [
            "scan" => true,
            "prepare" => true,
            "transfer" => true,
            "update" => false
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        $this->log->debug("Cron: Order export script ran for {$diff->format("%H:%I:%S")}");

    }

    // -m=Cron -f=updateOrders
    public function CLI_updateOrders($param)
    {
        $start = new DateTime("now");
        $this->log->debug("Cron: Starting to update all orders.");
        $this->setMaxRunningTime(null);

        $processes = [
            "scan" => false,
            "prepare" => false,
            "transfer" => false,
            "update" => true
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        $this->log->debug("Cron: Order update script ran for {$diff->format("%H:%I:%S")}");

    }

    // -m=Cron -f=processOrders
    public function CLI_processOrders($param)
    {
        $start = new DateTime("now");
        $this->log->debug("Cron: Starting to process all orders.");
        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_ORDER_PROCESSIING);

        $processes = [
            "scan" => true,
            "prepare" => true,
            "transfer" => true,
            "update" => true
        ];
        $this->_processOrders($processes);

        $end = new DateTime("now");
        $diff = $start->diff($end);
        $this->log->debug("Cron: Order process script ran for {$diff->format("%H:%I:%S")}");

    }

    // -m=Cron -f=timedPricesInside
    public function CLI_timedPricesInside($param)
    {
        $this->log->debug("Cron: Starting timed prices (inside) processing.");
        $this->_processTimedPrices("inside");
    }

    // -m=Cron -f=timedPricesOutside
    public function CLI_timedPricesOutside($param)
    {
        $this->log->debug("Cron: Starting timed prices (outside) processing.");
        $this->_processTimedPrices("outside");
    }

    protected function _processProducts($processes, $updateStock)
    {
        if (empty($processes)) {
            return;
        }

        $model = Mage::getModel("integration/Cli_LocalProductImport") ?: Mage::getModel("integration/Cli_ProductImport");
        $model->setLimit($this->getLimit());
        $model->setDebugMode($this->getDebugMode());
        if ($updateStock !== null) {
            $model->setUpdateStock($updateStock);
        }

        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT);
        $taggedProducts = 0;
        $importedProducts = 0;

        if (!empty($processes["tag"])) {
            $productList = $model->getAllUpdatedProducts();
            $model->tagProductsForImport($productList);
            $taggedProducts = count($productList);
            $this->log->debug("Tagged {$taggedProducts} products for import.");
        }

        if (!empty($processes["import"])) {
            do {
                $integrationProduct = $model->getProductToImport();
                if (empty($integrationProduct)) {
                    break;
                }

                $model->mergeDataToProduct($integrationProduct);
                $model->updateIntegrationData($integrationProduct);
                if ($integrationProduct["import_status"] == Awardit_Integration_Model_Cli_ProductImport::IMPORT_STATUS_FETCHED) {
                    $model->importSingleProduct($integrationProduct);
                    $model->updateIntegrationData($integrationProduct);
                }

                if ($this->getLimit() > 0) {
                    if ($importedProducts++ >= $this->getLimit()) {
                        $this->log->debug("Limit reached, exiting.");
                        break;
                    }
                }

            } while ($this->checkRunningTime());

            if (!$this->checkRunningTime()) {
                $this->log->debug("Time limit reached, exiting.");
            }
        }

        $this->log->debug("Imported {$importedProducts} products.");

    }

    protected function _processOrders($processes): void
    {
        if (empty($processes)) {
            return;
        }

        $model = Mage::getModel("integration/Cli_LocalOrderExport") ?: Mage::getModel("integration/Cli_OrderExport");
        $model->setLimit($this->getLimit());
        $model->setDebugMode($this->getDebugMode());
        $preparedOrders = 0;
        $updatedOrders = 0;
        $iterations = 0;

        if (!empty($processes["scan"])) {
            $model->scanForNewOrders();
        }

        if (!empty($processes["prepare"])) {
            $integrationOrders = $model->getAllOrdersToExport();

            if (!empty($integrationOrders)) {
                $maxQtyToExport = count($integrationOrders);

                $this->log->debug("Found {$maxQtyToExport} order to prepare.");

                foreach ($integrationOrders as $integrationOrder) {
                    $model->prepareSingleOrder($integrationOrder);

                    $preparedOrders++;
                    if ($this->getLimit() > 0) {
                        if ($preparedOrders >= $this->getLimit()) {
                            $this->log->debug("Limit reached, exiting.");
                            break;
                        }
                    }

                    if (!$this->checkRunningTime()) {
                        break;
                    }
                }
            }
        }

        if (!empty($processes["transfer"])) {
            if ($preparedOrders > 0) {
                if (!Mage::helper($this->_defaultHelper)->getSaveXMLDataToDB()) {
                    $model->transferAllOrdersToVisma();
                }
            }
        }

        if (!empty($processes["update"]) && $this->checkRunningTime()) {
            $ordersToUpdate = $model->getAllOrdersToUpdate();
            if (!empty($ordersToUpdate)) {
                $maxQtyToUpdate = count($ordersToUpdate);
                $this->log->debug("Found {$maxQtyToUpdate} orders to update.");
                foreach ($ordersToUpdate as $integrationOrder) {
                    $model->updateSingleOrder($integrationOrder);
                    $updatedOrders += $model->getWasLastOrderUpdated() ? 1 : 0;
                    $iterations++;
                    if ($this->getLimit() > 0) {
                        if ($iterations >= $this->getLimit()) {
                            $this->log->debug("Limit reached, exiting.");
                            break;
                        }
                    }

                    if (!$this->checkRunningTime()) {
                        break;
                    }
                }
            }
        }

        if (!$this->checkRunningTime()) {
            $this->log->debug("Time limit reached, exiting.");
        }

        if (!empty($preparedOrders) && !empty($updatedOrders)) {
            $this->log->debug("Exported {$preparedOrders} and updated {$updatedOrders} orders.");
        } elseif(!empty($preparedOrders)) {
            $this->log->debug("Exported {$preparedOrders} orders.");
        } elseif(!empty($updatedOrders)) {
            $this->log->debug("Updated {$updatedOrders} orders.");
        } else {
            $this->log->debug("Exported and updated no orders.");
        }

    }

    protected function _processTimedPrices($mode): void
    {
        $start = new DateTime("now");
        $this->setMaxRunningTime(Awardit_Integration_Model_Cli_Cron::CRON_MAX_RUNNING_TIME_PRODUCT_IMPORT);

        $model = $this->getTimedPricesModel();
        if ($mode == "inside") {
            $model->setDoInside(true);
            $model->setDoOutside(false);
            $model->setDateAdjustment("tomorrow 00:01:00");
        } elseif ($mode == "outside") {
            $model->setDoInside(true);
            $model->setDoOutside(true);
        } else {
            throw new Exception("Unknown mode: {$mode}");
        }

        $model->processTimedPrices();
        $this->log->debug("Cron: Running reindexing process.");
        $model->doReindex();

        $end = new DateTime("now");
        $diff = $start->diff($end);
        $this->log->debug("Cron: Timed prices processing script ran for {$diff->format("%H:%I:%S")}");

    }

}
