<?php

use Psr\Log\LoggerInterface;

class Awardit_Integration_Model_Cli {
    const CLI_DEFAULT_ITERATION_LIMIT = 100;

    protected readonly LoggerInterface $log;
    protected string $_defaultHelper = "integration";
    protected bool $_debugMode = false;
    protected bool $_testMode = false;
    protected int $_limit = Awardit_Integration_Model_Cli::CLI_DEFAULT_ITERATION_LIMIT;
    protected ?Awardit_Integration_Model_TimedPrices $_timedPricesModel = null;

    public function __construct() {
        $this->log = new Awardit_Magento_Logger("integration");
    }

    public function getTimedPricesModel(): Awardit_Integration_Model_TimedPrices
    {
        if (empty($this->_timedPricesModel)) {
            $this->_timedPricesModel = Mage::getModel("integration/TimedPrices");
        }
        return $this->_timedPricesModel;
    }

    public function setDebugMode(bool $mode): void
    {
        $this->_debugMode = $mode ? true : false;
    }

    public function getDebugMode(): bool
    {
        return $this->_debugMode;
    }

    public function setTestMode(bool $mode): void
    {
        $this->_testMode = $mode ? true : false;
    }

    public function getTestMode(): bool
    {
        return $this->_testMode;
    }

    public function getDefaultLimit(): int
    {
        return Awardit_Integration_Model_Cli::CLI_DEFAULT_ITERATION_LIMIT;
    }

    public function setLimit(int $limit): void
    {
        $this->_limit = intval($limit);
    }

    public function getLimit(): int
    {
        return $this->_limit;
    }

    /**
     *
     * @param string|null $inputParamStr
     * @param array $wantedParams
     * @return array
     * @throws Exception
     */
    public function extractParams(?string $inputParamStr, array $wantedParams): array
    {
        //
        /*
         * example input: param1:1234,param2:49.99
         *
         * example config: (! denotes required parameter)
         * $wantedParams = [
         *     "param1" => "int",
         *     "!param2" => "float"
         * ];
         *
         * example output:
         * $extractedParams = [
         *     "param1" => 1234,
         *     "param2" => 49.99
         * ];
         *
         */

        if (empty($inputParamStr)) {
            $inputParamStr = "dummy:1701";
        }

        $inputParams = explode(",", $inputParamStr);
        if (empty($inputParams) || !is_array($inputParams)) {
            throw new Exception("Missing required parameter!");
        }

        $rawExtractedParams = [];
        foreach ($inputParams as $inputParam) {
            $keyVal = explode(":", $inputParam);
            if (empty($keyVal) || !is_array($keyVal) || count($keyVal) !== 2) {
                throw new Exception("Malformed parameter!");
            }
            $rawExtractedParams[$keyVal[0]] = $keyVal[1];
        }

        $parsedParams = [];
        foreach ($wantedParams as $param => $type) {

            $isRequired = false;
            if (stripos($param, "!") === 0) {
                $isRequired = true;
                $param = substr($param, 1);
            }

            if ($isRequired && !array_key_exists($param, $rawExtractedParams)) {
                throw new Exception("Missing required parameter: {$param}");
            } elseif(array_key_exists($param, $rawExtractedParams)) {
                switch ($type) {
                    case "int":
                        $parsedParams[$param] = intval($rawExtractedParams[$param]);
                        break;
                    case "float":
                        $parsedParams[$param] = floatval($rawExtractedParams[$param]);
                        break;
                    case "date":
                        $parsedParams[$param] = date("Y-m-d", strtotime($rawExtractedParams[$param]));
                        break;
                    case "string":
                    default:
                        $parsedParams[$param] = $rawExtractedParams[$param];
                        break;
                }
            }
        }

        return $parsedParams;

    }

}
