<?php

namespace Awardit\GoogleMerchant\Feed;

use DomainException;
use JSON_PRESERVE_ZERO_FRACTION;
use Mage;
use Mage_Catalog_Model_Category;
use Mage_Catalog_Model_Resource_Category_Collection;
use Mage_Core_Model_Store;
use Phrity\Util\Numerics;
use Symfony\Component\Serializer\Encoder\{
    CsvEncoder,
    JsonEncoder,
    XmlEncoder,
    YamlEncoder
};

class Products
{
    public const CACHE_LIFETIME = 7200; // 2h
    public const CACHE_PREFIX = 'googlemerchant-feed-product-';

    /** @var Mage_Core_Model_Store */
    private $store;

    /** @var Numerics */
    private $numerics;

    public function __construct(Mage_Core_Model_Store $store)
    {
        $this->store = $store;
        $this->numerics = new Numerics(2, 'c');
        Mage::app()->setCurrentStore($this->store->getCode());
    }

    /**
     * Render products according to format
     */
    public function render(string $format = 'csv'): string
    {
        switch ($format) {
            case 'csv':
                $encoder = new CsvEncoder();
                break;
            case 'json':
                $encoder = new JsonEncoder(null, null, [
                    'json_encode_options' => JSON_PRESERVE_ZERO_FRACTION | JSON_PRETTY_PRINT,
                ]);
                break;
            case 'scsv':
                $encoder = new CsvEncoder(['csv_delimiter' => ';']);
                break;
            case 'xml':
                $encoder = new XmlEncoder();
                break;
            case 'yaml':
                $encoder = new JsonEncoder();
                break;
            default:
                throw new DomainException("Invalid format '{$format}'; use 'csv', 'json', 'scsv', 'xml' or 'yaml'");
        }

        $products = $this->getProducts();
        return $encoder->encode($products, $format);
    }

    /**
     * Render products according to format
     */
    public function getMimeType(string $format = 'csv'): string
    {
        switch ($format) {
            case 'csv':
                return 'text/csv; charset=UTF-8';
            case 'json':
                return 'application/json; charset=UTF-8';
            case 'scsv':
                return 'application/vnd.ms-excel; charset=UTF-8';
            case 'xml':
                return 'application/xml; charset=UTF-8';
            case 'yaml':
                return 'application/yaml; charset=UTF-8';
            default:
                throw new DomainException("Invalid format '{$format}'; use 'csv', 'json', 'scsv', 'xml' or 'yaml'");
        }
    }

    /**
     * Load and format products for current store
     */
    private function getProducts(): array
    {
        // Possibly use cached prodocut data
        $cache = Mage::app()->getCacheInstance();
        $cache_key = self::CACHE_PREFIX . $this->store->getId();
        if ($data = $cache->load($cache_key)) {
            return json_decode($data, true);
        }

        $data = $this->fetchProducts();

        // Store result in cache
        $cache->save(json_encode($data), $cache_key, [], self::CACHE_LIFETIME);

        return $data;
    }

    /**
     * Fetch products for store
     *
     * @return array
     */
    public function fetchProducts(): array
    {
        $data = [];

        // Load products
        $collection = Mage::getResourceModel('catalog/product_collection');
        $collection->setStore($this->store);
        $collection->setVisibility(Mage::getSingleton('catalog/product_visibility')->getVisibleInCatalogIds());
        $collection->addAttributeToFilter('sku', ['neq' => '']); // Required in feed,
        $collection->addAttributeToFilter('name', ['neq' => '']); // Required in feed,

        // Format products
        foreach ($collection as $product) {
            $product->setStoreId($this->store->getId())->load($product->getId());
            $images = $product->getMediaGalleryImages();
            if (count($images) < 1) {
                continue; // Required in feed, ignore if missing
            }
            $image = $images->getFirstItem();
            $inventory = Mage::getModel('cataloginventory/stock_item')->loadByProduct($product);
            $availability = !$inventory->getUseConfigManageStock() || $inventory->getQty() > 0;
            $price = $this->numerics->format($product->getPrice());
            $currency = $this->store->getBaseCurrencyCode();
            $type = $this->getCategoryPaths($product->getCategoryCollection());

            $data[] = [
                'id'            => mb_strcut($product->getSku(), 0, 50),
                'title'         => mb_strcut($product->getName(), 0, 150),
                'description'   => mb_strcut($product->getDescription(), 0, 5000),
                'link'          => $product->getUrlInStore(),
                'image_link'    => $image->getUrl(),
                'availability'  => $availability ? 'in_stock' : 'out_of_stock',
                'price'         => "{$price} {$currency}",
                'product_type'  => mb_strcut($type, 0, 750),
            ];
        }

        return $data;
    }

    /**
     * Get all category tree in Google format
     */
    private function getCategoryPaths(Mage_Catalog_Model_Resource_Category_Collection $categories): string
    {
        $paths = [];
        foreach ($categories as $category) {
            if ($path = $this->getCategoryPath($category)) {
                $paths[] = implode(' > ', $path);
            }
        }
        $paths = array_filter($paths, function ($verify) use ($paths) {
            foreach ($paths as $path) {
                if (substr_count($path, $verify) && strlen($verify) < strlen($path)) {
                    return false;
                }
            }
            return true;
        });
        return implode(', ', $paths);
    }

    /**
     * Expand category inheritance
     */
    private function getCategoryPath(Mage_Catalog_Model_Category $category): array
    {
        $category->setStoreId($this->store->getId())->load($category->getId());
        $parent = $category->getParentCategory();
        $path = $parent->getId() && $parent->getId() != $this->store->getRootCategoryId()
            ? $this->getCategoryPath($parent)
            : [];
        if ($category->getIsActive() && $category->getId() != $this->store->getRootCategoryId()) {
            $path[] = trim(str_replace([',', '>'], ' ', $category->getName())); // Used as separators, replace
        }
        return $path;
    }
}
