<?php
declare(strict_types=1);

class Awardit_FreeShippingCalculator_Helper_Data extends Mage_Core_Helper_Abstract
{
    public function getAmountNeededForFreeShipping($quote = null, $storeId = null)
    {
        if (!$quote) {
            $quote = Mage::getSingleton('checkout/session')->getQuote();
        }

        if (!$quote || !$quote->getId()) {
            return null;
        }


        // Check tax configuration setting
        $priceIncludesTax = Mage::getStoreConfig('tax/calculation/price_includes_tax', $storeId);

        // Calculate subtotal based on tax settings
        if ($priceIncludesTax) {
            // Including Tax
            $currentSubtotal = (float) $quote->getShippingAddress()->getSubtotalInclTax();
        } else {
            // Excluding Tax (default)
            $currentSubtotal = (float) $quote->getSubtotal();
        }
        $threshold = $this->getFreeShippingThresholdFromRules($storeId);

        if ($threshold === null) {
            return null;
        }

        $amountNeeded = $threshold - $currentSubtotal;

        return $amountNeeded > 0 ? $amountNeeded : 0;
    }

    public function getFreeShippingThresholdFromRules($storeId = null)
    {
        $websiteId = Mage::app()->getStore($storeId)->getWebsiteId();
        $customerGroupId = Mage::getSingleton('customer/session')->getCustomerGroupId();
        $now = Mage::getModel('core/date')->date('Y-m-d');

        $quote = Mage::getSingleton('checkout/session')->getQuote();

        // Prepare quote address for rule validation
        $address = $quote->getShippingAddress();
        if (!$address->getCountryId()) {
            // Set default country if not set
            $address->setCountryId(Mage::getStoreConfig('general/country/default'));
        }

        $rules = Mage::getModel('salesrule/rule')->getCollection()
            ->addFieldToFilter('is_active', 1)
            ->addFieldToFilter('simple_free_shipping', array('neq' => 0))
            ->addWebsiteFilter($websiteId);

        // Add customer group filter manually
        $rules->getSelect()
            ->joinInner(
                array('customer_group_ids' => $rules->getTable('salesrule/customer_group')),
                'main_table.rule_id = customer_group_ids.rule_id AND customer_group_ids.customer_group_id = ' . (int)$customerGroupId,
                array()
            )
            ->where('from_date is null or from_date <= ?', $now)
            ->where('to_date is null or to_date >= ?', $now);

        $thresholds = array();

        foreach ($rules as $rule) {
            // Load the full rule model to get conditions
            $fullRule = Mage::getModel('salesrule/rule')->load($rule->getId());

            // Check if rule name is exactly "Free Shipping"
            if ($fullRule->getName() !== 'Free Shipping') {
                continue; // Skip rules that don't have the exact name "Free Shipping"
            }

            // Don't check if rule validates - we want to show progress even if threshold not met
            // Comment out validation to always show the threshold
            // if (!$this->isRuleValidForCart($fullRule, $quote, $address)) {
            //     continue; // Skip this rule as it doesn't apply to current cart
            // }

            try {
                $threshold = $this->extractSubtotalThresholdFromRule($fullRule);
                if ($threshold !== null) {
                    $thresholds[] = $threshold;
                }
            } catch (Exception $e) {
                // Log error if needed
                Mage::logException($e);
            }
        }

        $finalThreshold = null;
        if (!empty($thresholds)) {
            $finalThreshold = min($thresholds);
        }

        return $finalThreshold;
    }

    public function extractSubtotalThresholdFromRule($rule)
    {
        // First try the serialized conditions approach (if available)
        $serialized = $rule->getData('conditions_serialized');
        if ($serialized && strlen($serialized) > 0) {
            if (strpos($serialized, 'base_subtotal') !== false || strpos($serialized, 'subtotal') !== false) {
                // Look for value in serialized string
                preg_match('/subtotal.*?value["\']:\s*["\']?(\d+(?:\.\d+)?)["\']?/i', $serialized, $matches);
                if (!empty($matches[1])) {
                    $threshold = (float)$matches[1];
                    return $threshold;
                }
            }
        }

        // If serialized is empty or didn't work, use the conditions array directly
        $conditions = $rule->getConditions();
        if ($conditions) {
            $conditionsArray = $conditions->asArray();
            if (isset($conditionsArray['conditions']) && is_array($conditionsArray['conditions'])) {
                foreach ($conditionsArray['conditions'] as $condition) {
                    if (isset($condition['attribute']) &&
                        ($condition['attribute'] == 'base_subtotal' || $condition['attribute'] == 'subtotal') &&
                        isset($condition['operator']) &&
                        ($condition['operator'] == '>=' || $condition['operator'] == '>') &&
                        isset($condition['value'])) {
                        return (float)$condition['value'];
                    }
                }
            }
        }

        // Finally try the recursive object approach
        $threshold = $this->parseConditionsForSubtotal($conditions);

        return $threshold;
    }

    public function parseConditionsForSubtotal($conditions)
    {
        $attribute = $conditions->getAttribute();
        $operator = $conditions->getOperator();
        $value = $conditions->getValue();

        // Check if this is a subtotal condition
        if ($attribute == 'base_subtotal' || $attribute == 'subtotal') {
            $numValue = (float) $value;
            if ($operator == '>=' || $operator == '>') {
                return $numValue;
            }
        }

        // If this condition has child conditions, recursively check them
        if ($conditions->getConditions()) {
            $childConditions = $conditions->getConditions();
            foreach ($childConditions as $childCondition) {
                $threshold = $this->parseConditionsForSubtotal($childCondition);
                if ($threshold !== null) {
                    return $threshold;
                }
            }
        }

        return null;
    }

    /**
     * Check if a rule's conditions are met by the current cart
     * This validates ALL conditions (weight, subtotal, etc.) not just the subtotal
     */
    protected function isRuleValidForCart($rule, $quote, $address)
    {
        // Use Magento's built-in validation to check if ALL rule conditions are met
        try {
            // Set the quote items on the address for validation
            $address->setAllItems($quote->getAllVisibleItems());

            // Ensure address has all necessary data for validation
            $address->setSubtotal($quote->getSubtotal());
            $address->setBaseSubtotal($quote->getBaseSubtotal());
            $address->setGrandTotal($quote->getGrandTotal());
            $address->setBaseGrandTotal($quote->getBaseGrandTotal());

            // Calculate and set weight
            $totalWeight = 0;
            foreach ($quote->getAllVisibleItems() as $item) {
                $totalWeight += $item->getWeight() * $item->getQty();
            }
            $address->setWeight($totalWeight);

            // Get the rule conditions
            $conditions = $rule->getConditions();


            // Validate the conditions against the address (which contains cart data)
            if ($conditions && $conditions->validate($address)) {
                if (Mage::getStoreConfig('dev/log/active')) {
                    Mage::log("✅ Rule PASSED validation", null, 'free_shipping_debug.log');
                }
                return true;
            }

            if (Mage::getStoreConfig('dev/log/active')) {
                Mage::log("❌ Rule FAILED validation", null, 'free_shipping_debug.log');
            }
            return false;
        } catch (Exception $e) {
            Mage::logException($e);
            if (Mage::getStoreConfig('dev/log/active')) {
                Mage::log("💥 Exception during validation: " . $e->getMessage(), null, 'free_shipping_debug.log');
            }
            return false;
        }
    }

    public function formatAmountNeeded($amount, $storeId = null)
    {
        if ($amount === null) {
            return null;
        }

        if ($amount == 0) {
            return $this->__('You qualify for free shipping!');
        }

        $formattedAmount = Mage::helper('core')->currency($amount, true, false);
        return $this->__('Add %s more to get free shipping', $formattedAmount);
    }

    public function isEligibleForFreeShipping($quote = null)
    {
        $amountNeeded = $this->getAmountNeededForFreeShipping($quote);
        return $amountNeeded !== null && $amountNeeded == 0;
    }
}