<?php

declare(strict_types=1);

class Awardit_DonationConfirmationEmail_Model_Observer {

    const ALLOWED_ORDER_STATUS = ['processing', 'complete'];
    const CONFIG_ENABLED = "donationconfirmationemail/email/enabled";
    const CONFIG_TEMPLATE = "donationconfirmationemail/email/template";

    public function quoteSubmitSuccess(Varien_Event_Observer $observer): void
    {
        /**
         * @var Mage_Sales_Model_Order
         */
        $order = $observer->getOrder();

        if(!$order->getStore()->getConfig(self::CONFIG_ENABLED)) {
            return;
        }

        if(!in_array($order->getStatus(),self::ALLOWED_ORDER_STATUS)) {
            return;
        }

        $attributeSetId = Mage::getSingleton("catalog/config")
            ->getAttributeSetId(Mage_Catalog_Model_Product::ENTITY, "Donations");

        foreach ($order->getAllItems() as $item) {
            if( (int) $item->getProduct()->getAttributeSetId() === (int) $attributeSetId ) {
                $this->sendEmail($order, $item);
            }
        }
    }

    private function sendEmail(Mage_Sales_Model_Order $order, Mage_Sales_Model_Order_Item $item): void {
        $store = $order->getStore();
        $billingAddress = $order->getBillingAddress();

        if(!$billingAddress) {
            Mage::log(sprintf(
                "%s: Order '%s' is missing billing address.",
                __METHOD__,
                $order->getIncrementId()
            ));
            return;
        }

        $firstname = $billingAddress->getFirstname();
        $lastname = $billingAddress->getLastname();

        /**
         * @var Awardit_DonationConfirmationEmail_ProductExtension
         */
        $product = $item->getProduct();

        $donationMessage = $product->getDonationMessage();
        if($donationMessage === null) {
            Mage::log(sprintf(
                "%s: Order '%s' is missing donation message for product with sku '%s'.",
                __METHOD__,
                $order->getIncrementId(),
                $product->getSku()
            ));
            return;
        }
        $template = $store->getConfig(self::CONFIG_TEMPLATE);

        if(empty($template)) {
            Mage::log(sprintf(
                "%s: Template for store '%d' is missing.",
                __METHOD__,
                $store->getId(),
            ));
            return;
        }

        $senderEmail = $store->getConfig("trans_email/ident_general/email");
        $senderName = $store->getConfig("trans_email/ident_general/name");
        $email = Mage::getModel("core/email_template")->load($template);

        if($senderName != null && $senderEmail != null) {
            $email->setSenderName($senderName);
            $email->setSenderEmail($senderEmail);
        }

        $customerEmail = trim($order->getCustomerEmail());

        // If to use the email queue
        $emailQueue = Mage::getModel('core/email_queue');
        $emailQueue->setEventType($template)
            ->setIsForceCheck(0);

        $email->setQueue($emailQueue);

        $greeting = preg_replace("/(?:\n\r?)+/", "<br />", htmlspecialchars(trim($donationMessage), ENT_QUOTES, 'UTF-8'));


        $variables = [
            "donationMessage" => $greeting,
            "customerFirstname" => $firstname,
            "customerLastname" => $lastname,
            "customerEmail" => $order->getCustomerEmail(),
            "product" => $item->getProduct()
        ];

        if( ! $email->send($customerEmail, $firstname ." ". $lastname, $variables)) {
            Mage::throwException("Failed sending email");
        }
    }
}
