<?php

declare(strict_types=1);

namespace Donation;

use Awardit_DonationConfirmationEmail_Model_Observer;
use Crossroads\Magento\Test\Integration\Config;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Mage;
use Throwable;
use PHPUnit\Framework\TestCase;
use Varien_Object;

class DonationTest extends TestCase {
    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::initAdmin();

        Config::unsetConfigData("payment/checkmo/order_status");
        Config::unsetConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_ENABLED);
        Config::unsetConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_TEMPLATE);

        $resource = Mage::getSingleton('core/resource');
        $table = $resource->getTableName('core/email_queue');

        $writeConnection = $resource->getConnection('core_write');
        assert($writeConnection !== false);
        $writeConnection->delete($table);
        MagentoManager::reset();
    }

    public function tearDown(): void {
        MagentoManager::logQueries();

        Config::unsetConfigData("payment/checkmo/order_status");
        Config::unsetConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_ENABLED);
        Config::unsetConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_TEMPLATE);
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        $this->tearDown();
        throw $t;
    }

    /**
     * @throws \Throwable
     * @throws \Mage_Core_Model_Store_Exception
     */
    public function testPlaceOrderDisabled(): void {

        MagentoManager::init();
        Config::setConfigData("default/payment/checkmo/order_status", "processing");
        $this->makeOrder();

        $emailQueue = Mage::getModel('core/email_queue');
        $collection = $emailQueue->getCollection();
        $this->assertNotFalse($collection);
        $this->assertEquals(0, $collection->count());
    }

    public function testPlaceOrderEnabled(): void {
        MagentoManager::init();

        $template = Mage::getModel("core/email_template");
        $template->loadByCode("test_donation_email");
        Config::setConfigData("payment/checkmo/order_status", "processing");
        Config::setConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_ENABLED, '1');
        Config::setConfigData(Awardit_DonationConfirmationEmail_Model_Observer::CONFIG_TEMPLATE, $template->getId());

        MagentoManager::reset();

        $this->makeOrder();

        $emailQueue = Mage::getModel('core/email_queue');
        $collection = $emailQueue->getCollection();
        $this->assertNotFalse($collection);
        $this->assertEquals(1, $collection->count());
        $email = $collection->getFirstItem();
        $this->assertEquals('message: Tack för din donation på X poäng till välgörenhetsändamål Djurens rätt
customer: Test Testsson
email: test@example.com', $email->getMessageBody());

        $this->assertEquals([['test@example.com', 'Test Testsson', '0']], $email->getRecipients());
    }

    public function makeOrder(): void {
        MagentoManager::init();

        $quote = Mage::getSingleton('checkout/session')->getQuote();

        $quote->setStoreId(Mage::app()->getStore()->getId());

        $quote->setIsActive(1);
        // Simple requires shipping for price calculation
        $quote->setBillingAddress(Mage::getModel("sales/quote_address")->addData([
            "postcode" => "12345",
            "lastname" => "Testsson",
            "street" => "Paradisäppelvägen 111",
            "city" => "Ankeborg",
            "telephone" => "0123456789",
            "country_id" => "SE",
            "firstname" => "Test",
        ]));

        $quote->setCustomerEmail("test@example.com");

        $simple = Mage::getModel("catalog/product");
        $simple->setStoreId(Mage::app()->getStore()->getId())
            ->load(Mage::getModel("catalog/product")->getIdBySku("test-donation"));
        $quote->addProduct($simple, new Varien_Object([
            "qty" => 1,
        ]));

        $quote->setTotalsCollectedFlag(false);
        $quote->getShippingAddress()->setCollectShippingRates(true);
        $quote->collectTotals();
        $quote->save();

        $quote->getPayment()->importData([
            "method" => "checkmo",
        ]);

        $quote->getPayment()->save();
        $quote->save();

        $service = Mage::getModel("sales/service_quote", $quote);
        $service->submitAll();

    }
}