<?php

declare(strict_types=1);

// TODO: Move to separate composer library under awardit/doctrine-dblib

// Variation of Doctrine\DBAL\Driver\PDO\SQLSrv using dblib.
// We cannot just extend the existing classes since they are final.

namespace Awardit\Doctrine\DBAL\Dblib;

use Doctrine\DBAL\Driver\AbstractSQLServerDriver;
use Doctrine\DBAL\Driver\AbstractSQLServerDriver\Exception\PortWithoutHost;
use Doctrine\DBAL\Driver\Exception;
use Doctrine\DBAL\Driver\PDO\Connection as PDOConnection;
use Doctrine\DBAL\Driver\PDO\Exception as PDOException;
use PDO;
use SensitiveParameter;

use function is_int;
use function sprintf;

/**
 * @psalm-api
 * @psalm-suppress InternalMethod
 * @psalm-suppress InternalClass
 * @psalm-import-type Params from \Doctrine\DBAL\DriverManager as DriverParams
 */
final class Driver extends AbstractSQLServerDriver
{
    /**
     * {@inheritDoc}
     *
     * @psalm-param DriverParams $params
     * @return Connection
     */
    public function connect(
        #[SensitiveParameter]
        array $params
    ) {
        $driverOptions = [];
        $dsnOptions = [];

        if (isset($params['driverOptions'])) {
            foreach ($params['driverOptions'] as $option => $value) {
                if (is_int($option)) {
                    $driverOptions[$option] = $value;
                } else {
                    $dsnOptions[$option] = $value;
                }
            }
        }

        if (! empty($params['persistent'])) {
            $driverOptions[PDO::ATTR_PERSISTENT] = true;
        }

        $safeParams = $params;
        unset($safeParams['password'], $safeParams['url']);

        try {
            $dsn = $this->constructDsn($safeParams, $dsnOptions);
            $pdo = new PDO(
                $dsn,
                $params['user'] ?? '',
                $params['password'] ?? '',
                $driverOptions,
            );
        } catch (\PDOException $exception) {
            throw PDOException::new($exception);
        }

        return new Connection(new PDOConnection($pdo));
    }

    /**
     * Constructs the dblib PDO DSN.
     *
     * @param mixed[]  $params
     * @param string[] $connectionOptions
     * @psalm-param DriverParams $params
     *
     * @throws Exception
     */
    private function constructDsn(array $params, array $connectionOptions): string
    {
        $dsn = 'dblib:host=';

        if (isset($params['host'])) {
            $dsn .= $params['host'];

            if (isset($params['port'])) {
                $dsn .= ':' . $params['port'];
            }
        } elseif (isset($params['port'])) {
            throw PortWithoutHost::new();
        }

        if (isset($params['dbname'])) {
            $connectionOptions['dbname'] = $params['dbname'];
        }

        return $dsn . $this->getConnectionOptionsDsn($connectionOptions);
    }

    /**
     * Converts a connection options array to the DSN
     *
     * @param string[] $connectionOptions
     */
    private function getConnectionOptionsDsn(array $connectionOptions): string
    {
        $connectionOptionsDsn = '';

        foreach ($connectionOptions as $paramName => $paramValue) {
            $connectionOptionsDsn .= sprintf(';%s=%s', $paramName, $paramValue);
        }

        return $connectionOptionsDsn;
    }
}
