<?php

declare(strict_types=1);

namespace Awardit\CodeLock;

use Throwable;
use Mage;
use Crossroads\Magento\Test\Integration\MagentoManager;
use Crossroads\Magento\Test\Integration\Request;
use PHPUnit\Framework\TestCase;
use Spatie\Snapshots\MatchesSnapshots;

class RequestTest extends TestCase {
    use MatchesSnapshots;

    public function setUp(): void {
        MagentoManager::reset();
        MagentoManager::init('default');
        Mage::setIsDeveloperMode(false);
    }

    public function tearDown(): void {
        MagentoManager::logQueries();
    }

    public function onNotSuccessfulTest(Throwable $t): void {
        MagentoManager::logQueries();

        throw $t;
    }

    public function testGraphqlException(): void {
        $res = MagentoManager::runRequest(
            new Request(
                "POST /graphql",
                ["Content-Type" => "application/graphql"],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockHeaderWrongCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                "POST /graphql",
                ["Content-Type" => "application/graphql", "X-Codelock-Code" => 'wrongcode'],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockHeaderCorrectCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                "POST /graphql",
                ["Content-Type" => "application/graphql", "X-Codelock-Code" => 'letmein'],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockHeaderCorrectSecondCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                "POST /graphql",
                ["Content-Type" => "application/graphql", "X-Codelock-Code" => 'letmeinagain'],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockParamWrongCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                "POST /graphql?codelock_code=wrongcode",
                ["Content-Type" => "application/graphql"],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(400, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockParamCorrectCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                sprintf("POST /graphql?codelock_code=%s", 'letmein'),
                ["Content-Type" => "application/graphql"],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }

    public function testGraphqlCodelockParamCorrectSecondCode(): void {
        $res = MagentoManager::runRequest(
            new Request(
                sprintf("POST /graphql?codelock_code=%s", 'letmeinagain'),
                ["Content-Type" => "application/graphql"],
                'query { info { baseUrl } }'),
            'default'
        );

        $this->assertEquals(200, $res->getHttpResponseCode());
        $this->assertEquals("application/json; charset=utf-8", $res->getHeader("Content-Type"));
        $this->assertMatchesJsonSnapshot($res->getBody());
    }
}
