<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use stdClass;
use PHPUnit\Framework\TestCase;

class CustomResourceResponseTest extends TestCase
{
    public function testToLambdaResponse(): void
    {
        $response = new CustomResourceResponse(
            CustomResourceResponse::STATUS_SUCCESS,
            null,
            null
        );
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];
        $event2 = [
            "StackId" => "stackid",
            "RequestType" => "Update",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "SUCCESS",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
        ], $response->toLambdaResponse($event));
        $this->assertEquals([
            "Status" => "SUCCESS",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "a7e57c6",
        ], $response->toLambdaResponse($event2));
    }

    public function testToLambdaResponseFailureNoReason(): void
    {
        if (version_compare(PHP_VERSION, '8.0.0') < 0) {
            $this->markTestSkipped("Skipping due to PHP 7.4 handling asserts differently");

            /** @psalm-suppress UnevaluatedCode We are not always 7.4 */
            return;
        }

        $this->expectExceptionMessageMatches("/A failure response must have a reason/");

        new CustomResourceResponse(CustomResourceResponse::STATUS_FAILURE);
    }

    public function testToLambdaResponseReason(): void
    {
        $response = new CustomResourceResponse(
            CustomResourceResponse::STATUS_FAILURE,
            "Some failure",
            null
        );
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "FAILURE",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "Reason" => "Some failure",
        ], $response->toLambdaResponse($event));
    }

    public function testToLambdaResponseData(): void
    {
        $response = new CustomResourceResponse(
            CustomResourceResponse::STATUS_SUCCESS,
            null,
            [ "Value" => "Here" ],
        );
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "SUCCESS",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "Data" => ["Value" => "Here"],
        ], $response->toLambdaResponse($event));
    }

    public function testSuccess(): void
    {
        $response = CustomResourceResponse::success([ "Value" => "Here" ]);
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "SUCCESS",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "Data" => ["Value" => "Here"],
        ], $response->toLambdaResponse($event));
    }

    public function testSuccess2(): void
    {
        $response = CustomResourceResponse::success();
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "SUCCESS",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
        ], $response->toLambdaResponse($event));
    }

    public function testFailure(): void
    {
        /** @psalm-suppress DeprecatedMethod */
        $response = CustomResourceResponse::failure("Some reason");
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "FAILURE",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "Reason" => "Some reason",
        ], $response->toLambdaResponse($event));
    }

    public function testFailure2(): void
    {
        /** @psalm-suppress DeprecatedMethod */
        $response = CustomResourceResponse::failure("Some reason", ["extra" => "data"]);
        $event = [
            "StackId" => "stackid",
            "RequestType" => "Create",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "ResourceProperties" => [],
        ];

        $this->assertEquals([
            "Status" => "FAILURE",
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "PhysicalResourceId" => "physid",
            "Reason" => "Some reason",
            "Data" => ["extra" => "data"],
        ], $response->toLambdaResponse($event));
    }
}
