<?php

declare(strict_types=1);

namespace Awardit\Aws;

use JsonException;

/**
 * Helper class for JSON.
 *
 * Mainly provides utilities for working with JSON as associative arrays since
 * that is what the AWS SDK mainly uses.
 */
class Json
{
    /**
     * Encodes an associative array as JSON, preserving floats and makes sure
     * slashes are not escaped.
     *
     * @api
     * @param Array<string, mixed> $value
     */
    public static function encode(array $value): string
    {
        return json_encode(
            $value,
            JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR
        );
    }

    /**
     * Decodes JSON into an associative array.
     *
     * @api
     * @throws JsonException if the JSON decode fails or if the result is not
     *                       an associative array.
     * @return Array<string, mixed>
     */
    public static function decode(string $value): array
    {
        $data = json_decode($value, true, 512, JSON_OBJECT_AS_ARRAY | JSON_THROW_ON_ERROR);

        if (!is_array($data)) {
            throw new JsonException(sprintf(
                "Invalid JSON body: Expected associative array, got %s",
                get_debug_type($data),
            ));
        }

        /** @var Array<string, mixed> */
        return $data;
    }
}
