# AWS Helper Library for PHP

## Installation

```bash
composer config repositories.crossroads '{"type":"composer","url":"https://composer.crossroads.se","canonical": false}'
composer require awardit/aws
```

## Features

### Logger

Provides a factory creating loggers compatible with AWS CloudWatch and
The Twelve-Factor App specification.

```php
use Awardit\Aws\Logger;

// Register a global error handler
Logger::registerErrorHandler();

// Create PSR LoggerInterface implementations with any given name
$log = Logger::getLogger("my-app");
$log->debug("Hello!");
```

### Lambda Runtime

Runtime for running PHP as an AWS Lambda.

```php
use Awardit\Aws\Lambda\Runtime;
use Awardit\Aws\Lambda\CallbackHandler;

Logger::registerErrorHandler();
$runtime = Runtime::fromEnv();

try {
    $app = new CallbackHandler(function (array $request) use ($log): array {
        return [
            "statusCode" => 200,
            "body" => "Lambda response",
        ];
    });
} catch (Throwable $t) {
    $runtime->sendInitializationException($t);

    throw $t;
}

$runtime->run($app);
```

It also includes helper interfaces and handler for implementing
CloudFormation Custom Resources.

### Environment Utilities

Utilities to manage and obtain environment variables. Environment variables are
used by AWS and containers to provide configuration information to the process.

```php
use Awardit\Aws\Env;

// Load a .env from current directory, if present
Env::loadDotenv(__DIR__);

$value = Env::getValue("SOME_VALUE");

// Will throw if no value is present:
$value = Env::mustGetValue("SOME_OTHER_VALUE");
```

### JSON Utilities

Utilities for working with associative arrays as JSON. AWS SDK primarily uses
associative arrays.

```php
use Awardit\Aws\Json;

$string = Json::encode([ "test" => "value" ]);
$value = Json::decode('{"some":"json"}');
```
