<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use Psr\Http\Message\ResponseInterface;
use Awardit\Aws\Exception;
use Awardit\Aws\Json;

/**
 * @template T of Array<string, mixed>
 */
class Request
{
    /**
     * Creates an incoming AWS request from a response obtained from the
     * lambda runtime api endpoint.
     */
    public static function fromLambdaPsrResponse(ResponseInterface $response): self
    {
        $invocationId = $response->getHeader("Lambda-Runtime-Aws-Request-Id");

        if (count($invocationId) === 0) {
            throw new Exception(sprintf(
                "%s: Lambda-Runtime-Aws-Request-Id header was not present in AWS Lambda API response",
                __METHOD__
            ));
        }

        $payload = Json::decode((string)$response->getBody());

        return new self(
            /* invocationId: */            $invocationId[0],
            /* traceId: */ $response->getHeader("Lambda-Runtime-Trace-Id")[0] ?? null,
            // deadline: intval($response->getHeader("Lambda-Runtime-Deadline-Ms")[0] ?? "0"),
            // functionArn: $response->getHeader("Lambda-Runtime-Invoked-Function-Arn")[0] ?? null,
            /* payload: */ $payload,
        );
    }

    private string $invocationId;
    private ?string $traceId;
    // private int $deadline;
    // private string|null $functionArn;
    /**
     * @var T
     */
    private array $payload;

    /**
     * @param T $payload
     */
    public function __construct(
        string $invocationId,
        ?string $traceId,
        // private int $deadline,
        // private string|null $functionArn,
        array $payload
    ) {
        $this->invocationId = $invocationId;
        $this->traceId = $traceId;
        $this->payload = $payload;
    }

    /**
     * Lambda invocation id, mainly used to be able to respond to the lambda
     * environment.
     *
     * @internal
     */
    public function getInvocationId(): string
    {
        return $this->invocationId;
    }

    /**
     * The Lambda request payload, contents depend on the lambda configuration.
     *
     * @api
     * @return T
     */
    public function getPayload(): array
    {
        return $this->payload;
    }

    /**
     * Returns the AWS Lambda Runtime Trace id, if any.
     *
     * @api
     */
    public function getTraceId(): ?string
    {
        return $this->traceId;
    }
}
