<?php

declare(strict_types=1);

namespace Awardit\Aws;

use Dotenv\Dotenv;

/**
 * Helper class used to obtain environment variables.
 */
class Env
{
    /**
     * Attempts to load a .env file from the supplied directory, will silently
     * skip if the file does not exist (usually the case in production).
     *
     * @api
     */
    public static function loadDotenv(string $directoryPath): void
    {
        // We have to load the environment variables using the unsafe variant
        // due to AWS SDK and libraries using the getenv function
        $dotenv = Dotenv::createUnsafeImmutable($directoryPath);

        // Safe load does not throw if the file is missing
        $dotenv->safeLoad();
    }

    /**
     * Obtains the given environment value, respecting local scope.
     *
     * @api
     * @return non-empty-string|null
     */
    public static function getValue(string $name): ?string
    {
        $localValue = getenv($name, true);

        if ($localValue !== false && !empty($localValue)) {
            return $localValue;
        }

        $globalValue = getenv($name);

        return $globalValue !== false && !empty($globalValue) ? $globalValue : null;
    }

    /**
     * Obtains the given environment value, respecting local scope, throwing if
     * the value is not set or is empty.
     *
     * @api
     * @throws Exception if the environment variable is not set or empty
     * @return non-empty-string
     */
    public static function mustGetValue(string $name): string
    {
        $value = self::getValue($name);

        if ($value === null) {
            throw new Exception(sprintf("Missing %s environment variable", $name));
        }

        return $value;
    }
}
