<?php

declare(strict_types=1);

namespace Awardit\Aws;

use Throwable;
use JsonException;

/**
 * Utility class for AWS.
 */
class Util
{
    /**
     * Encodes an associative array as JSON, preserving floats and makes sure
     * slashes are not escaped.
     *
     * @api
     * @deprecated Use Json::encode
     * @param Array<string, mixed> $value
     */
    public static function jsonEncode(array $value): string
    {
        return Json::encode($value);
    }

    /**
     * Decodes JSON into an associative array.
     *
     * @api
     * @deprecated Use Json::decode
     * @throws JsonException if the JSON decode fails or if the result is not
     *                       an associative array.
     * @return Array<string, mixed>
     */
    public static function jsonDecode(string $value): array
    {
        return Json::decode($value);
    }

    /**
     * Obtains the given environment value, respecting local scope.
     *
     * @api
     * @deprecated Use Env::getValue instead
     * @return non-empty-string|null
     */
    public static function getEnvValue(string $name): ?string
    {
        return Env::getValue($name);
    }

    /**
     * Obtains the given environment value, respecting local scope, throwing if
     * the value is not set or is empty.
     *
     * @api
     * @deprecated Use Env::mustGetValue instead
     * @throws Exception if the environment variable is not set or empty
     * @return non-empty-string
     */
    public static function mustGetEnvValue(string $name): string
    {
        return Env::mustGetValue($name);
    }

    /**
     * Constructs the common configuration for the AWS SDK instance from
     * environment, following AWS defaults.
     *
     * NOTE: The environment variables used are set by AWS.
     *
     * @api
     * @return array{region:string, endpoint?:string}
     */
    public static function getAwsSdkCommonConfig(): array
    {
        $commonConfig = [
            "region" => Env::getValue("AWS_REGION") ?? Env::getValue("AWS_DEFAULT_REGION") ?? "us-east-1",
        ];
        $endpointUrl = Env::getValue("AWS_ENDPOINT_URL") ?? "";

        // The AWS PHP SDK is *supposed* to support AWS_ENDPOINT_URL, but it somehow
        // ignores it, add it manually here:
        if (!empty($endpointUrl)) {
            $commonConfig["endpoint"] = $endpointUrl;
        }

        return $commonConfig;
    }

    /**
     * @return list<string>
     */
    public static function getExceptionTraceAsList(Throwable $t): array
    {
        $trace = [];

        /**
         * @var array{line:int, file?:string} $frame
         */
        foreach ($t->getTrace() as $frame) {
            if (isset($frame["file"])) {
                $trace[] = $frame["file"] . ":" . $frame["line"];
            }
        }

        return $trace;
    }
}
