<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use Awardit\Aws\Exception;

/**
 * Lambda handler for a Cloud Formation Custom Resource.
 *
 * @api
 * @psalm-type ResourceEvent array{
 *   StackId: string,
 *   RequestId: string,
 *   RequestType: string,
 *   LogicalResourceId: string,
 *   PhysicalResourceId?: string|null,
 *   ResourceProperties: Array<string, mixed>
 * }
 * @psalm-type ResourceResponse array{
 *   Status: CustomResourceResponse::STATUS_SUCCESS|CustomResourceResponse::STATUS_FAILURE,
 *   Reason?: string,
 *   Data?: Array<string, mixed>,
 *   StackId: string,
 *   RequestId: string,
 *   LogicalResourceId: string,
 *   PhysicalResourceId: string,
 * }
 * @implements HandlerInterface<ResourceEvent, ResourceResponse>
 */
class CustomResourceHandler implements HandlerInterface
{
    /**
     * The custom resource implementation.
     */
    private CustomResourceInterface $customResourceHandler;

    public function __construct(
        CustomResourceInterface $customResourceHandler
    ) {
        $this->customResourceHandler = $customResourceHandler;
    }

    public function handle(Request $request): array
    {
        $event = $request->getPayload();

        switch ($event["RequestType"]) {
            case "Create":
                return $this->customResourceHandler->create($event["ResourceProperties"])->toLambdaResponse($event);
            case "Update":
                return $this->customResourceHandler->update($event["ResourceProperties"])->toLambdaResponse($event);
            case "Delete":
                return $this->customResourceHandler->delete($event["ResourceProperties"])->toLambdaResponse($event);
            default:
                throw new Exception(sprintf("Unknown Custom Resource request type '%s'", $event["RequestType"]));
        }

        // We do not catch anything here. The Custom Resource Provider
        // framework will assume any response means it is OK. Instead we have
        // to let errors bubble to the Lambda Runtime to handle exceptions
        //
        // See: https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/custom-resources#provider-framework
    }
}
