<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use stdClass;
use PHPUnit\Framework\TestCase;

class CustomResourceHandlerTest extends TestCase
{
    public function testUnknown(): void
    {
        $this->expectExceptionMessageMatches("/Unknown Custom Resource request type 'foobar'/");

        $mock = $this->createMock(CustomResourceInterface::class);

        $mock->expects($this->never())->method($this->anything());

        $resource = new CustomResourceHandler($mock);
        $request = new Request(
            /* invocationId: */            "abc123",
            /* traceId: */ null,
            /* payload: */ [
                "StackId" => "stackid",
                "RequestId" => "reqid",
                "LogicalResourceId" => "logicalid",
                "ResourceProperties" => [],
                "RequestType" => "foobar",
            ],
        );

        $resource->handle($request);
    }

    /**
     * @dataProvider provideRequestsData
     */
    public function testRequests(string $type, string $method): void
    {
        $event = [
            "StackId" => "stackid",
            "RequestId" => "reqid",
            "LogicalResourceId" => "logicalid",
            "ResourceProperties" => ["custom" => "property"],
            "RequestType" => $type,
        ];

        $mock = $this->createMock(CustomResourceInterface::class);
        $response = $this->createMock(CustomResourceResponse::class);

        $response->expects($this->once())
            ->method("toLambdaResponse")
            ->with($event)
            ->willReturn(["lambda" => "response"]);
        $mock->expects($this->once())
            ->method($method)
            ->with(["custom" => "property"])
            ->willReturn($response);

        $resource = new CustomResourceHandler($mock);
        $request = new Request(
            /* invocationId: */            "abc123",
            /* traceId: */ null,
            /* payload: */ $event,
        );

        $this->assertEquals(["lambda" => "response"], $resource->handle($request));
    }

    public static function provideRequestsData(): array
    {
        return [
            ["Create", "create"],
            ["Update", "update"],
            ["Delete", "delete"],
        ];
    }
}
