<?php

declare(strict_types=1);

namespace Awardit\Aws;

use JsonException;
use PHPUnit\Framework\TestCase;

use function putenv;

class UtilTest extends TestCase
{
    public function testJsonDecode(): void
    {
        $this->expectException(JsonException::class);

        Util::jsonDecode("abc");
    }

    public function testJsonDecode2(): void
    {
        $this->expectException(JsonException::class);
        $this->expectExceptionMessage("Invalid JSON body: Expected associative array, got int");

        Util::jsonDecode("123");
    }

    public function testJsonDecode3(): void
    {
        $value = Util::jsonDecode('{"test":"value"}');

        $this->assertEquals(["test" => "value"], $value);
    }

    public function testJsonEncode(): void
    {
        $this->assertEquals('{"foo":"test"}', Util::jsonEncode(["foo" => "test"]));
        $this->assertEquals('{"value":0.0}', Util::jsonEncode(["value" => 0.0]));
        $this->assertEquals('{"value":"some/slashes"}', Util::jsonEncode(["value" => "some/slashes"]));
    }

    public function testGetEnvValue(): void
    {
        $value = Util::getEnvValue("PATH");
        $this->assertIsString($value);
        $this->assertStringContainsString("bin", $value);

        putenv("AWARDIT_AWS_TEST_VARIABLE");

        $value = Util::getEnvValue("AWARDIT_AWS_TEST_VARIABLE");
        $this->assertNull($value);

        putenv("AWARDIT_AWS_TEST_VARIABLE=foo");

        $value = Util::getEnvValue("AWARDIT_AWS_TEST_VARIABLE");
        $this->assertEquals("foo", $value);

        putenv("AWARDIT_AWS_TEST_VARIABLE");

        $value = Util::getEnvValue("AWARDIT_AWS_TEST_VARIABLE");
        $this->assertNull($value);
    }

    public function testMustGetEnvValue(): void
    {
        $this->expectException(Exception::class);
        $this->expectExceptionMessage("Missing AWARDIT_AWS_TEST_VARIABLE environment variable");

        putenv("AWARDIT_AWS_TEST_VARIABLE");

        Util::mustGetEnvValue("AWARDIT_AWS_TEST_VARIABLE");
    }

    public function testMustGetEnvValue2(): void
    {
        putenv("AWARDIT_AWS_TEST_VARIABLE=bar");

        $value = Util::mustGetEnvValue("AWARDIT_AWS_TEST_VARIABLE");
        $this->assertEquals("bar", $value);
    }

    public function testGetAwsSdkCommonConfig(): void
    {
        putenv("AWS_ENDPOINT_URL");
        putenv("AWS_REGION");
        putenv("AWS_DEFAULT_REGION");

        $value = Util::getAwsSdkCommonConfig();
        $this->assertEquals(["region" => "us-east-1"], $value);

        putenv("AWS_REGION=eu-north-1");

        $value = Util::getAwsSdkCommonConfig();
        $this->assertEquals(["region" => "eu-north-1"], $value);

        putenv("AWS_DEFAULT_REGION=eu-west-1");

        $value = Util::getAwsSdkCommonConfig();
        $this->assertEquals(["region" => "eu-north-1"], $value);

        putenv("AWS_REGION");

        $value = Util::getAwsSdkCommonConfig();
        $this->assertEquals(["region" => "eu-west-1"], $value);

        putenv("AWS_ENDPOINT_URL=http://localhost:1234");

        $value = Util::getAwsSdkCommonConfig();
        $this->assertEquals(["region" => "eu-west-1", "endpoint" => "http://localhost:1234"], $value);
    }

    public function testGetExceptionTraceAsList(): void
    {
        $list = Util::getExceptionTraceAsList(new Exception());

        $this->assertIsList($list);

        foreach ($list as $item) {
            $this->assertMatchesRegularExpression("/^[^:]+:\d+$/", $item);
        }
    }
}
