<?php

declare(strict_types=1);

namespace Awardit\Aws;

use Throwable;
use ReflectionClass;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use Monolog\Handler\TestHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger as MonologLogger;

class LoggerTest extends TestCase
{
    private TestHandler|null $handler = null;

    // TODO: Make into helper methods
    protected function setUp(): void
    {
        $this->handler = Logger::getTestStreamHandler();
    }

    public function tearDown(): void
    {
        Logger::resetStreamHandler();

        $this->handler = null;
    }

    public function testGetLogger(): void
    {
        $logger = Logger::getLogger("test-logger");

        $this->assertInstanceOf(LoggerInterface::class, $logger);

        $logger->debug("Debug");

        $this->assertTrue($this->handler?->hasDebugThatContains("Debug"));

        $records = $this->handler->getRecords();

        /** @var string */
        $formatted = $records[0]->formatted;
        $parsed = json_decode($formatted, true);

        $this->assertIsArray($parsed);

        $parsedNoTimestamp = array_diff_key($parsed, ["datetime" => true]);

        $this->assertEquals([
            "message" => "Debug",
            "context" => [],
            "level" => 100,
            "level_name" => "DEBUG",
            "channel" => "test-logger",
            "extra" => [],
        ], $parsedNoTimestamp);

        $this->assertIsString($parsed["datetime"]);
        // 2024-04-13T23:13:07.152494+0000
        $this->assertMatchesRegularExpression(
            "/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{6}\+\d{4}$/",
            $parsed["datetime"]
        );
    }

    public function testGetDefaultStreamInstance(): void
    {
        // Remove the setup stuff to trigger the singleton construction
        Logger::resetStreamHandler();

        $logger = Logger::getLogger("test-stream");
        $this->assertInstanceOf(MonologLogger::class, $logger);

        $handlers = $logger->getHandlers();
        $this->assertCount(1, $handlers);

        $stream = $handlers[0];
        $this->assertInstanceof(StreamHandler::class, $stream);
        $this->assertEquals("php://stderr", $stream->getUrl());
    }
}
