<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use Awardit\Aws\Exception;
use Awardit\Aws\Logger;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Request as PsrRequest;
use GuzzleHttp\Psr7\Response as PsrResponse;
use GuzzleHttp\Psr7\Utils as PsrUtils;
use GuzzleHttp\RequestOptions;
use JsonException;
use OutOfBoundsException;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;
use Monolog\LogRecord;

use function putenv;

class RuntimeTest extends TestCase
{
    public function setUp(): void
    {
        // Make sure we do not have the variable set
        putenv("AWS_LAMBDA_RUNTIME_API");
    }

    public function tearDown(): void
    {
        // Make sure we do not have the variable set
        putenv("AWS_LAMBDA_RUNTIME_API");
    }

    public function testGetLambdaRuntimeApiUrl(): void
    {
        $this->expectException(Exception::class);
        $this->expectExceptionMessageMatches("/Missing AWS_LAMBDA_RUNTIME_API/");

        Runtime::getLambdaRuntimeApiUrl();
    }

    public function testFromEnv(): void
    {
        putenv("AWS_LAMBDA_RUNTIME_API=http://example.com");

        $runtime = Runtime::fromEnv();

        $this->assertInstanceOf(Runtime::class, $runtime);
    }

    public function testRun(): void
    {
        ["log" => $log, "handler" => $handler] = Logger::createTestLoggerAndStream();
        // We're not making a mocked handler stack because it does not show all
        // requests if one of them is extra
        $client = $this->getMockBuilder(Client::class)->getMock();
        $n = 0;

        // We have to cast the type to avoid issues with Guzzle having a
        // deprecated __call:
        /** @var MockObject $client */
        $client->expects($this->exactly(2))
            ->method("get")
            ->with(
                "http://example.com/2018-06-01/runtime/invocation/next",
                [
                    RequestOptions::TIMEOUT => 0,
                    RequestOptions::READ_TIMEOUT => -1,
                    RequestOptions::CONNECT_TIMEOUT => 15,
                ]
            )
            ->willReturnCallback(function () use (&$n) {
                if ($n++ > 0) {
                    throw new OutOfBoundsException("Mock queue is empty");
                }
                return new PsrResponse(
                    200,
                    [
                        "Lambda-Runtime-Aws-Request-Id" => "test-id",
                    ],
                    '{"some":"data"}'
                );
            });

        /** @var Client $client */
        $runtime = new Runtime($log, $client, "example.com");

        $mock = $this->getMockBuilder(HandlerInterface::class)->getMock();
        $mock->expects($this->once())
            ->method("handle")
            ->with(new Request(
                invocationId: "test-id",
                traceId: null,
                payload: ["some" => "data"],
            ))
            ->willReturn(["a" => "response"]);

        try {
            // The MockHanlder will throw when we have no more responses,
            // kinda similar to how AWS triggers SIGTERM during the request.
            $runtime->run($mock);
        } catch (OutOfBoundsException $e) {
            $this->assertEquals("Mock queue is empty", $e->getMessage());
        }

        $messages = array_map(fn(LogRecord $m): string => $m->message, $handler->getRecords());

        $this->assertEquals([
            "Querying for next request",
            "Got request 'test-id'",
            "Responding to request 'test-id'",
            "Querying for next request",
        ], $messages);
    }

    public function testRunError(): void
    {
        ["log" => $log, "handler" => $handler] = Logger::createTestLoggerAndStream();
        // We're not making a mocked handler stack because it does not show all
        // requests if one of them is extra
        $client = $this->getMockBuilder(Client::class)->getMock();
        $n = 0;

        // We have to cast the type to avoid issues with Guzzle having a
        // deprecated __call:
        /** @var MockObject $client */
        $client->expects($this->exactly(2))
            ->method("get")
            ->with(
                "http://example.com/2018-06-01/runtime/invocation/next",
                [
                    RequestOptions::TIMEOUT => 0,
                    RequestOptions::READ_TIMEOUT => -1,
                    RequestOptions::CONNECT_TIMEOUT => 15,
                ]
            )
            ->willReturnCallback(function () use (&$n) {
                if ($n++ > 0) {
                    throw new OutOfBoundsException("Mock queue is empty");
                }
                return new PsrResponse(
                    200,
                    [
                        "Lambda-Runtime-Aws-Request-Id" => "test-id",
                    ],
                    '{"some":"data"}'
                );
            });

        /** @var Client $client */
        $runtime = new Runtime($log, $client, "example.com");

        $mock = $this->getMockBuilder(HandlerInterface::class)->getMock();
        $mock->expects($this->once())
            ->method("handle")
            ->with(new Request(
                invocationId: "test-id",
                traceId: null,
                payload: ["some" => "data"],
            ))
            ->will($this->throwException(new Exception("Test error here")));

        try {
            // The MockHanlder will throw when we have no more responses,
            // kinda similar to how AWS triggers SIGTERM during the request.
            $runtime->run($mock);
        } catch (OutOfBoundsException $e) {
            $this->assertEquals("Mock queue is empty", $e->getMessage());
        }

        $messages = array_map(fn(LogRecord $m): string => $m->message, $handler->getRecords());

        $this->assertCount(5, $messages);

        $this->assertEquals("Querying for next request", $messages[0]);
        $this->assertEquals("Got request 'test-id'", $messages[1]);
        $this->assertStringContainsString("Test error here", $messages[2]);
        $this->assertEquals("Error response to request 'test-id': Test error here", $messages[3]);
        $this->assertEquals("Querying for next request", $messages[4]);
    }
}
