<?php

declare(strict_types=1);

namespace Awardit\Aws;

use Throwable;
use JsonException;

/**
 * Utility class for AWS.
 */
class Util
{
    /**
     * Encodes an associative array as JSON, preserving floats and makes sure
     * slashes are not escaped.
     *
     * @param Array<string, mixed> $value
     */
    public static function jsonEncode(array $value): string
    {
        return json_encode(
            $value,
            JSON_PRESERVE_ZERO_FRACTION | JSON_UNESCAPED_SLASHES | JSON_THROW_ON_ERROR
        );
    }

    /**
     * Decodes JSON into an associative array.
     *
     * @throws JsonException if the JSON decode fails or if the result is not
     *                       an associative array.
     * @return Array<string, mixed>
     */
    public static function jsonDecode(string $value): array
    {
        $data = json_decode($value, true, 512, JSON_OBJECT_AS_ARRAY | JSON_THROW_ON_ERROR);

        if (!is_array($data)) {
            throw new JsonException(sprintf(
                "Invalid JSON body: Expected associative array, got %s",
                get_debug_type($data),
            ));
        }

        /** @var Array<string, mixed> */
        return $data;
    }

    /**
     * Obtains the given environment value, respecting local scope.
     *
     * @return non-empty-string|null
     */
    public static function getEnvValue(string $name): ?string
    {
        $localValue = getenv($name, true);

        if ($localValue !== false && !empty($localValue)) {
            return $localValue;
        }

        $globalValue = getenv($name);

        return $globalValue !== false && !empty($globalValue) ? $globalValue : null;
    }

    /**
     * Obtains the given environment value, respecting local scope, throwing if
     * the value is not set or is empty.
     *
     * @throws Exception if the environment variable is not set or empty
     * @return non-empty-string
     */
    public static function mustGetEnvValue(string $name): string
    {
        $value = self::getEnvValue($name);

        if ($value === null) {
            throw new Exception(sprintf("Missing %s environment variable", $name));
        }

        return $value;
    }

    /**
     * Constructs the common configuration for the AWS SDK instance from
     * environment, following AWS defaults.
     *
     * NOTE: The environment variables used are set by AWS.
     *
     * @return array{region:string, endpoint?:string}
     */
    public static function getAwsSdkCommonConfig(): array
    {
        $commonConfig = [
            "region" => self::getEnvValue("AWS_REGION") ?? self::getEnvValue("AWS_DEFAULT_REGION") ?? "us-east-1",
        ];
        $endpointUrl = self::getEnvValue("AWS_ENDPOINT_URL") ?? "";

        // The AWS PHP SDK is *supposed* to support AWS_ENDPOINT_URL, but it somehow
        // ignores it, add it manually here:
        if (!empty($endpointUrl)) {
            $commonConfig["endpoint"] = $endpointUrl;
        }

        return $commonConfig;
    }

    /**
     * @return list<string>
     */
    public static function getExceptionTraceAsList(Throwable $t): array
    {
        $trace = [];

        /**
         * @var array{line:int, file?:string} $frame
         */
        foreach ($t->getTrace() as $frame) {
            if (isset($frame["file"])) {
                $trace[] = $frame["file"] . ":" . $frame["line"];
            }
        }

        return $trace;
    }
}
