<?php

namespace Awardit\Aws;

use Monolog\ErrorHandler;
use Monolog\Formatter\JsonFormatter;
use Monolog\Handler\AbstractProcessingHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Handler\TestHandler;
use Monolog\Logger as MonologLogger;
use Monolog\Processor\PsrLogMessageProcessor;
use Psr\Log\LoggerInterface;

// TODO: Implement logger which will keep track of the current lambda
// invocation trace id (and traces in general).
/**
 * AWS adapted logger factory.
 *
 * Usage:
 *
 * ```
 * // Make sure to register for erorrs, exceptions, and fatal errors
 * Logger::registerErrorHandler();
 *
 * $log = Logger::getLogger("my-app");
 *
 * $log->debug("something"); // Will output a debug message on my-app
 * ```
 *
 * @api
 */
class Logger
{
    /**
     * Timestamp format for logging.
     *
     * We have to make sure to use standard timezone offset (O) without colon,
     * instead of Monolog default which is with a colon between hours and
     * minutes.
     */
    public const DATETIME_FORMAT = "Y-m-d\TH:i:s.uO";

    /**
     * Reusable stderr log stream.
     */
    private static AbstractProcessingHandler|null $stream = null;

    /**
     * Registers an error logger for all errors, exceptions, and fatal errors.
     *
     * @api
     * @codeCoverageIgnore
     */
    public static function registerErrorHandler(): void
    {
        ErrorHandler::register(self::getLogger("error"));
    }

    /**
     * Returns a logger for the given channel name.
     *
     * @api
     */
    public static function getLogger(string $name): LoggerInterface
    {
        return self::createLogger($name, self::getStreamInstance());
    }

    /**
     * For testing, replaces the standard stream used by newly created loggers
     * from getLogger and registerErrorHandler with a test stream.
     *
     * Use restoreStreamHandler() to reset the stream in test teardown.
     *
     * @api
     */
    public static function getTestStreamHandler(): TestHandler
    {
        return self::$stream = self::createTestStreamHandler();
    }

    /**
     * For testing, creates a logger and an associated test stream handler.
     *
     * This logger and stream are not tied to the singleton in this class
     * and no reset is required.
     *
     * @api
     * @return array{log:MonologLogger, handler:TestHandler}
     */
    public static function createTestLoggerAndStream(): array
    {
        $handler = self::createTestStreamHandler();
        $log = self::createLogger("test-logger", $handler);

        return ["log" => $log, "handler" => $handler];
    }

    /**
     * For Testing, resets the stream used to create new LoggerInterface
     * instances from getLogger() and registerErrorHandler().
     *
     * @api
     */
    public static function resetStreamHandler(): void
    {
        self::$stream = null;
    }

    private static function createLogger(
        string $name,
        AbstractProcessingHandler $stream
    ): MonologLogger {
        return new MonologLogger(
            $name,
            [$stream],
            // TODO: Add introspection processor?
            [new PsrLogMessageProcessor(self::DATETIME_FORMAT)],
        );
    }

    /**
     * The main log stream instance.
     */
    private static function getStreamInstance(): AbstractProcessingHandler
    {
        if (!self::$stream) {
            self::$stream = self::configureStream(new StreamHandler("php://stderr"));
        }

        return self::$stream;
    }

    /**
     * @template T of AbstractProcessingHandler
     * @param T $stream
     * @return T
     */
    private static function configureStream(
        AbstractProcessingHandler $stream
    ): AbstractProcessingHandler {
        $formatter = new JsonFormatter(includeStacktraces: true);

        $formatter->setDateFormat(self::DATETIME_FORMAT);
        $stream->setFormatter($formatter);

        // TODO: Maybe also change output structure to better match AWS?

        return $stream;
    }

    /**
     * Creates a configured test stream handler.
     */
    private static function createTestStreamHandler(): TestHandler
    {
        return self::configureStream(new TestHandler());
    }
}
