<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use Awardit\Aws\Exception;
use Awardit\Aws\Logger;
use Awardit\Aws\Util;
use GuzzleHttp\Client as GuzzleClient;
use GuzzleHttp\RequestOptions;
use Psr\Log\LoggerInterface;
use Throwable;

// TODO: Graceful shutdown: Lambda sends SIGTERM to the process while it is
// waiting for the next invocation (ie. in getNextRequest).
/**
 * Lambda runtime loop implementation.
 *
 * If the initialization of the lambda fails before calling Runtime::run() it
 * is recommended to call Runtime::sendInitializationException() or
 * Runtime::sendInitializationError().
 *
 * Example:
 *
 *   $runtime = Runtime::fromEnv();
 *
 *   $runtime->run(new MyHandler());
 *
 * @api
 * @psalm-type LambdaErrorResponse array{
 *   errorMessage:string,
 *   errorType:string,
 *   stackTrace:list<string>,
 * }
 * @psalm-type LambdaInitializationError LambdaErrorResponse
 * @psalm-type LambdaErrorType self::ERROR_UNKNOWN_REASON|
 *   self::ERROR_CONFIG_INVALID|
 *   self::ERROR_NO_SUCH_HANDLER|
 *   self::ERROR_API_KEY_NOT_FOUND
 */
class Runtime
{
    /**
     * Lambda user agent to track which lambdas are PHP.
     */
    public const USER_AGENT = "Awardit PHP Lambda Runtime";

    /**
     * Lambda error type/category indicating that no handler with the given
     * name, indicated by `_HANDLER` environment variable, could be found.
     */
    public const ERROR_NO_SUCH_HANDLER = "Runtime.NoSuchHandler";
    /**
     * Lambda error type/category indicating a failure to obtain the required API keys.
     */
    public const ERROR_API_KEY_NOT_FOUND = "Runtime.APIKeyNotFound";
    /**
     * Lambda error type/category indicating a configuration error.
     */
    public const ERROR_CONFIG_INVALID = "Runtime.ConfigInvalid";
    /**
     * Generic lambda error type/category.
     */
    public const ERROR_UNKNOWN_REASON = "Runtime.UnknownReason";

    /**
     * Request options for the request to read the next lambda event.
     *
     * This can wait indefinitely until the lambda is finally terminated by
     * the AWS runtime if no events are available.
     */
    public const NEXT_REQUEST_OPTIONS = [
        RequestOptions::TIMEOUT => 0,
        // Negative should be infinite according to PHP socket documentation
        RequestOptions::READ_TIMEOUT => -1,
        // We still need to connect though
        RequestOptions::CONNECT_TIMEOUT => 15,
    ];

    /**
     * Options for other requests.
     */
    public const REQUEST_OPTIONS = [
        RequestOptions::CONNECT_TIMEOUT => 15,
        RequestOptions::TIMEOUT => 60,
    ];

    /**
     * Creates a Lambda Runtime from the AWS environment.
     */
    public static function fromEnv(): self
    {
        return new self(
            log: Logger::getLogger("lambda-runtime"),
            // TODO: https://openswoole.com/docs/modules/swoole-curl
            guzzle: new GuzzleClient([
                "headers" => [
                    "User-Agent" => self::USER_AGENT,
                ],
            ]),
            lambdaUrl: self::getLambdaRuntimeApiUrl(),
        );
    }

    /**
     * Reads the AWS Lambda Runtime API URL from environment, this is ALWAYS
     * set by AWS as an environment variable in lambdas.
     *
     * @throws Exception if not in a lambda environment/AWS_LAMBDA_RUNTIME_API
     *         is missing
     */
    public static function getLambdaRuntimeApiUrl(): string
    {
        $runtimeApi = Util::getEnvValue("AWS_LAMBDA_RUNTIME_API");

        if ($runtimeApi === null) {
            throw new Exception("Missing AWS_LAMBDA_RUNTIME_API environment variable");
        }

        return $runtimeApi;
    }

    /**
     * @internal Use fromEnv()
     */
    public function __construct(
        /**
         * Logger used for runtime or handler exceptions.
         */
        private LoggerInterface $log,
        /**
         * HTTP client instance to use when communicating with the Lambda environment.
         */
        private GuzzleClient $guzzle,
        /**
         * URL provided to the lambda through AWS_LAMBDA_RUNTIME_API environment variable.
         */
        private string $lambdaUrl,
    ) {
    }

    /**
     * Blocking function which will continously fetch new events to process
     * from the lambda API, calling the handler with the events.
     *
     * @api
     * @template T of Array<string, mixed>
     * @template U of Array<string, mixed>
     * @param HandlerInterface<T, U> $handler Handler to call for each incoming
     *                                        Lambda request.
     */
    public function run(handlerInterface $handler): void
    {
        // TODO: Loop-max setting
        // We continue consuming data until we are no longer able to
        while (true) {
            // We do not catch anything here on purpose, since any request
            // error should not happen, and if they do we should abort anyway
            $this->runRequest($handler, $this->getNextRequest());
        }
    }

    /**
     * Runs a single request with the supplied handler and responds to the
     * lambda runtime.
     */
    protected function runRequest(HandlerInterface $handler, Request $request): void
    {
        try {
            // TODO: Set _X_AMZN_TRACE_ID env variable or scoped context based
            // on request trace id
            $this->sendResponse($request, $handler->handle($request));
        } catch (Throwable $t) {
            $this->log->error("Failed to process lambda: {exception}", [
                "method" => __METHOD__,
                "exception" => $t,
            ]);

            $this->sendErrorResponse($request, [
                "errorMessage" => $t->getMessage(),
                "errorType" => $t::class,
                "stackTrace" => Util::getExceptionTraceAsList($t),
            ]);

            // TODO: Should we abort here if we do actually fail hard?
            // Maybe a counter or something?
        }
    }

    /**
     * Sends an error to the Lambda Runtime Environment to indicate that the
     * initialization of this lambda failed.
     *
     * @api
     * @param LambdaErrorType $lambdaErrorType
     */
    public function sendInitializationException(
        Throwable $exception,
        string $lambdaErrorType = self::ERROR_UNKNOWN_REASON
    ): void {
        $this->sendInitializationError([
            "errorMessage" => $exception->getMessage(),
            "errorType" => $exception::class,
            "stackTrace" => Util::getExceptionTraceAsList($exception),
        ], $lambdaErrorType);
    }

    /**
     * Sends an error to the Lambda Runtime Environment to indicate that the
     * initialization of this lambda failed.
     *
     * @param LambdaInitializationError $error
     * @param LambdaErrorType $lambdaErrorType
     */
    public function sendInitializationError(
        array $error,
        string $lambdaErrorType = self::ERROR_UNKNOWN_REASON
    ): void {
        $this->log->notice("Initialization error: {errorMessage}", [
            "method" => __METHOD__,
            "lambda.errorType" => $lambdaErrorType,
            ...$error,
        ]);

        $this->guzzle->post(sprintf(
            "http://%s/2018-06-01/runtime/init/error",
            $this->lambdaUrl,
        ), [
            ...self::REQUEST_OPTIONS,
            "headers" => [
                "Lambda-Runtime-Function-Error-Type" => $lambdaErrorType,
            ],
            RequestOptions::JSON => $error,
        ]);
    }

    /**
     * @internal
     */
    public function getNextRequest(): Request
    {
        $this->log->debug("Querying for next request", [
            "method" => __METHOD__,
        ]);

        $lambdaResponse = $this->guzzle->get(sprintf(
            "http://%s/2018-06-01/runtime/invocation/next",
            $this->lambdaUrl
        ), self::NEXT_REQUEST_OPTIONS);
        $request = Request::fromLambdaPsrResponse($lambdaResponse);

        $this->log->debug("Got request '{invocationId}'", [
            "method" => __METHOD__,
            "invocationId" => $request->getInvocationId(),
        ]);

        return $request;
    }

    /**
     * Sends the lambda-response associated with the given request, the
     * contents will depend on the type of lambda.
     *
     * @internal
     */
    public function sendResponse(Request $request, array $response): void
    {
        $this->log->debug("Responding to request '{invocationId}'", [
            "method" => __METHOD__,
            "invocationId" => $request->getInvocationId(),
        ]);

        $this->guzzle->post(sprintf(
            "http://%s/2018-06-01/runtime/invocation/%s/response",
            $this->lambdaUrl,
            $request->getInvocationId(),
        ), [
            ...self::REQUEST_OPTIONS,
            RequestOptions::JSON => $response,
        ]);
    }

    /**
     * @internal
     *
     * @param LambdaErrorResponse $response
     * @param LambdaErrorType $lambdaErrorType
     */
    public function sendErrorResponse(
        Request $request,
        array $response,
        string $lambdaErrorType = self::ERROR_UNKNOWN_REASON
    ): void {
        $this->log->notice("Error response to request '{invocationId}': {errorMessage}", [
            "method" => __METHOD__,
            "invocationId" => $request->getInvocationId(),
            "lambda.errorType" => $lambdaErrorType,
            ...$response
        ]);

        $this->guzzle->post(sprintf(
            "http://%s/2018-06-01/runtime/invocation/%s/error",
            $this->lambdaUrl,
            $request->getInvocationId(),
        ), [
            ...self::REQUEST_OPTIONS,
            "headers" => [
                "Lambda-Runtime-Function-Error-Type" => $lambdaErrorType,
            ],
            RequestOptions::JSON => $response,
        ]);
    }
}
