<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

/**
 * @psalm-import-type ResourceEvent from CustomResourceHandler
 * @psalm-import-type ResourceResponse from CustomResourceHandler
 */
class CustomResourceResponse
{
    /**
     * Indicates a successful creation/update/deletion of the resource.
     */
    public const STATUS_SUCCESS = "SUCCESS";
    /**
     * Indicates a failed creation/update/deletion of the resource.
     */
    public const STATUS_FAILURE = "FAILURE";

    /**
     * @api
     * @param Array<string, mixed>|null $data Additional data provided from the custom resource
     */
    public static function success(array|null $data = null): self
    {
        return new self(
            status: self::STATUS_SUCCESS,
            data: $data,
        );
    }

    /**
     * @api
     * @param Array<string, mixed>|null $data Additional data provided from the custom resource
     */
    public static function failure(string $reason, array|null $data = null): self
    {
        return new self(
            status: self::STATUS_FAILURE,
            reason: $reason,
            data: $data,
        );
    }

    /**
     * @internal Use success and failure static methods to construct this object.
     */
    public function __construct(
        /**
         * If the reason is self::STATUS_FAILURE, reason MUST be specified.
         *
         * @var self::STATUS_FAILURE|self::STATUS_SUCCESS
         */
        private string $status,
        private ?string $reason = null,
        /**
         * @var Array<string, mixed>|null
         */
        private ?array $data = null
    ) {
        if ($this->status === self::STATUS_FAILURE) {
            assert($this->reason !== null);
        }
    }

    /**
     * @param ResourceEvent $event
     * @return ResourceResponse
     */
    public function toLambdaResponse(array $event): array
    {
        $response = [
            "Status" => $this->status,
            "StackId" => $event["StackId"],
            "RequestId" => $event["RequestId"],
            "LogicalResourceId" => $event["LogicalResourceId"],
            "PhysicalResourceId" => $event["PhysicalResourceId"] ??
                substr(md5($event["StackId"] . "-" . $event["LogicalResourceId"]), 0, 7),
        ];

        if ($this->reason !== null) {
            $response["Reason"] = $this->reason;
        }

        if ($this->data !== null) {
            $response["Data"] = $this->data;
        }

        return $response;
    }
}
