<?php

declare(strict_types=1);

namespace Awardit\Aws\Lambda;

use Awardit\Aws\Exception;

/**
 * Lambda handler for a Cloud Formation Custom Resource.
 *
 * @api
 * @psalm-type ResourceEvent array{
 *   StackId: string,
 *   RequestId: string,
 *   RequestType: string,
 *   LogicalResourceId: string,
 *   PhysicalResourceId: string|null,
 *   ResourceProperties: Array<string, mixed>
 * }
 * @psalm-type ResourceResponse array{
 *   Status: CustomResourceResponse::STATUS_SUCCESS|CustomResourceResponse::STATUS_FAILURE,
 *   Reason?: string,
 *   Data?: Array<string, mixed>,
 *   StackId: string,
 *   RequestId: string,
 *   LogicalResourceId: string,
 *   PhysicalResourceId: string,
 * }
 * @implements HandlerInterface<ResourceEvent, ResourceResponse>
 */
class CustomResourceHandler implements HandlerInterface
{
    /**
     * Indicates a successful creation/update/deletion of the resource.
     */
    public const STATUS_SUCCESS = "SUCCESS";
    /**
     * Indicates a failed creation/update/deletion of the resource.
     */
    public const STATUS_FAILURE = "FAILURE";

    public function __construct(
        /**
         * The custom resource implementation.
         */
        private CustomResourceInterface $customResourceHandler
    ) {
    }

    public function handle(Request $request): array
    {
        $event = $request->getPayload();

        switch ($event["RequestType"]) {
            case "Create":
                return $this->customResourceHandler->create($event["ResourceProperties"])->toLambdaResponse($event);
            case "Update":
                return $this->customResourceHandler->update($event["ResourceProperties"])->toLambdaResponse($event);
            case "Delete":
                return $this->customResourceHandler->delete($event["ResourceProperties"])->toLambdaResponse($event);
            default:
                throw new Exception(sprintf("Unknown Custom Resource request type '%s'", $event["RequestType"]));
        }
    }
}
