<?php

use Aws\Exception\AwsException;
use Aws\PersonalizeRuntime\PersonalizeRuntimeClient;

class Awardit_AwsPersonalize_Helper_Data extends Mage_Core_Helper_Abstract
{
    private ?PersonalizeRuntimeClient $client = null;
    private bool $enabled = false;
    private array $recommenders = [];

    public function __construct()
    {
        $region = Mage::getStoreConfig('awardit_awspersonalize/credentials/region');
        $key = Mage::getStoreConfig('awardit_awspersonalize/credentials/key');
        $secret = Mage::getStoreConfig('awardit_awspersonalize/credentials/secret');
        $this->enabled = $region && $key && $secret;

        if (!$this->enabled) {
            return; // No credentials
        }
        $this->recommenders = array_filter([
            Awardit_AwsPersonalize_Recommender::RECOMMEND => Mage::getStoreConfig(
                Awardit_AwsPersonalize_Recommender::RECOMMEND,
                Mage::app()->getStore()
            ),
            Awardit_AwsPersonalize_Recommender::BEST_SELLER => Mage::getStoreConfig(
                Awardit_AwsPersonalize_Recommender::BEST_SELLER,
                Mage::app()->getStore()
            ),
            Awardit_AwsPersonalize_Recommender::BOUGHT_TOGETHER => Mage::getStoreConfig(
                Awardit_AwsPersonalize_Recommender::BOUGHT_TOGETHER,
                Mage::app()->getStore()
            ),
        ]);
        if (empty($this->recommenders)) {
            return; // No recommenders to use
        }

        // Create PersonalizeRuntime client
        $this->client = new PersonalizeRuntimeClient([
            'version' => 'latest',
            'region' => $region,
            'credentials' => [
                'key' => $key,
                'secret' => $secret,
            ],
            // PHP <8.1 is deprecated
            'suppress_php_deprecation_warning' => true,
        ]);
        $this->enabled = true;
    }

    // If any or requested recommnder is enabled
    public function isEnabled(?string $recommender = null): bool
    {
        if (!$this->enabled) {
            return false;
        }
        return is_null($recommender) || array_key_exists($recommender, $this->recommenders);
    }

    public function getProducts(
        string $recommender,
        ?string $userId = null,
        ?string $productId = null,
        int $numberOfResults = MageQL_Catalog_Model_Product_CollectionInterface::DEFAULT_PAGE_SIZE
    ): array {
        if (!$this->isEnabled($recommender)) {
            return [];
        }

        try {
            // Prepare request parameters
            $recommenderArn = $this->recommenders[$recommender];

            $params = [
                'recommenderArn' => $recommenderArn,  // For ECOMMERCE domain
                // 'campaignArn' => $recommenderArn,  // Use this line instead for custom campaigns
                'numResults' => $numberOfResults
            ];

            // Add either userId or itemId depending on what's provided
            if ($userId) {
                $params['userId'] = $userId;
            } elseif ($productId) {
                $params['itemId'] = $productId;
            } else {
                throw new InvalidArgumentException("Either userId or productId must be provided");
            }

            // Get recommendations
            $result = $this->client->getRecommendations($params);

            // Extract recommended items
            $recommendedProducts = [];
            if (isset($result['itemList'])) {
                foreach ($result['itemList'] as $item) {
                    $recommendedProducts[] = $item['itemId'];
                }
            }
            return $recommendedProducts;
        } catch (AwsException $e) {
            // Handle errors
            Mage::log("Aws Error: {$e->getMessage()}");
            return [];
        } catch (Throwable $e) {
            // Handle errors
            Mage::log("General Error: {$e->getMessage()}");
            return [];
        }
    }
}
