<?php

/**
 * Controller for Sales > Fraud management > Fraud rules page.
 */
class Awardit_Antifraud_Adminhtml_Antifraud_RuleController extends Mage_Adminhtml_Controller_Action
{
    /** @var Awardit_Antifraud_Helper_Data */
    private $helper;

    public function __construct(
        Zend_Controller_Request_Abstract $request,
        Zend_Controller_Response_Abstract $response,
        array $invokeArgs = []
    ) {
        $this->helper = Mage::helper('awardit_antifraud');
        parent::__construct($request, $response, $invokeArgs);
    }

    /**
     * List of fraud rule entries.
     */
    public function indexAction(): void
    {
        /** @var Awardit_Antifraud_Block_Rule */
        $block = $this->getLayout()->createBlock('awardit_antifraud/rule');
        $this->_initAction()
            ->_addContent($block)
            ->renderLayout();
    }

    /**
     * Editor for an existing fraud rule entry.
     */
    public function editAction(): void
    {
        $uid = $this->getRequest()->getParam('id');
        $rule = Mage::getModel('awardit_antifraud/rule')->load($uid);
        $this->_initAction();
        /** @var Awardit_Antifraud_Block_Rule_Edit */
        $block = $this->getLayout()->createBlock('awardit_antifraud/rule_edit');
        if ($rule->exists()) {
            Mage::register('rule', $rule);
            $this->_addContent(
                $block
                    ->assign('header',$this->helper->__('Edit fraud rule'))
                    ->assign('form', $this->getLayout()->createBlock('awardit_antifraud/rule_edit_form'))
            )->renderLayout();
        } else {
            Mage::getSingleton('adminhtml/session')->addError($this->helper->__('Item does not exist'));
            $this->_redirect('*/*/');
        }
    }

    /**
     * Editor for a new fraud rule entry.
     */
    public function newAction(): void
    {
        $rule = Mage::getModel('awardit_antifraud/rule');
        Mage::register('rule', $rule);
        $this->_initAction();
        /** @var Awardit_Antifraud_Block_Rule_Edit */
        $block = $this->getLayout()->createBlock('awardit_antifraud/rule_edit');
        $this->_addContent(
            $block
                ->assign('header',$this->helper->__('New fraud rule'))
                ->assign('form', $this->getLayout()->createBlock('awardit_antifraud/rule_edit_form'))
        )->renderLayout();
    }

    /**
     * Saving new or existing fraud rule entry.
     */
    public function saveAction(): void
    {
        $request = $this->getRequest()->getPost();
        $model = Mage::getModel('awardit_antifraud/rule');
        $uid = $this->getRequest()->getParam('id');

        if ($this->getRequest()->getPost()) {
            try {
                $model->load($uid);

                if (!is_numeric($request['rule_value']) || $request['rule_value'] < 1) {
                    // @todo: Should be ValueError, but not available in php7.
                    throw new TypeError($this->helper->__('Rule value must be positive numeric.'));
                }

                if (!$model->exists()) {
                    $model->setCreatedAt(date('c'));
                }
                $model->setStoreId($request['store_id']);
                $model->setRuleType($request['rule_type']);
                $model->setRuleValue((int)$request['rule_value']);
                $model->setGroupType($request['group_type']);
                $model->setGroupInterval($request['group_interval']);
                $model->setMode($request['mode']);
                $model->setComment($request['comment']);
                if ($model->isSkuFilterable()) {
                    $model->setSkuFilter($request['sku_filter']);
                }
                $model->save();

                Mage::getSingleton('adminhtml/session')->addSuccess($this->helper->__('Successfully saved.'));
                Mage::getSingleton('adminhtml/session')->setData('antifraud.rule', false);

            } catch (TypeError $e) {
                Mage::getSingleton('adminhtml/session')->addError($e->getMessage());
                Mage::getSingleton('adminhtml/session')->setData('antifraud.rule', $request);
                if ($model->exists()) {
                    $this->_redirect('*/*/edit', ['id' => $uid]);
                } else {
                    $this->_redirect('*/*/new');
                }
                return;

            } catch (Exception $e) {
                Mage::getSingleton('adminhtml/session')->addError($this->helper->__('Invalid data provided.'));
                Mage::getSingleton('adminhtml/session')->setData('antifraud.rule', $request);
                if ($model->exists()) {
                    $this->_redirect('*/*/edit', ['id' => $uid]);
                } else {
                    $this->_redirect('*/*/new');
                }
                return;
            }
        }
        $this->_redirect('*/*/');
    }

    /**
     * Delete an existing fraud rule entry.
     */
    public function deleteAction(): void
    {
        $uid = $this->getRequest()->getParam('id');
        $rule = Mage::getModel('awardit_antifraud/rule')->load($uid);
        $this->_initAction();

        if ($rule->exists()) {
            try {
                $rule->delete();
                Mage::getSingleton('adminhtml/session')->addSuccess($this->helper->__('Item was successfully deleted'));
                $this->_redirect('*/*/');
            } catch (Exception $e) {
                Mage::getSingleton('adminhtml/session')->addError($e->getMessage());
                $this->_redirect('*/*/edit', ['id' => $uid]);
            }
        }
        $this->_redirect('*/*/');
    }

    /**
     * Init helper method.
     * @return $this
     */
    protected function _initAction(): self
    {
        $this->_title($this->__('Sales'))
             ->_title($this->__('Fraud management'))
             ->_title($this->__('Fraud rules'));

        $this->loadLayout()
            ->_setActiveMenu('sales/antifraud/rule')
            ->_addBreadcrumb($this->helper->__('Fraud management'), $this->helper->__('Fraud management'))
            ->_addBreadcrumb($this->helper->__('Fraud rules'), $this->helper->__('Fraud rules'))
        ;
        return $this;
    }
}
