<?php

/**
 * @psalm-type OrderOrQuote = Mage_Sales_Model_Order|Mage_Sales_Model_Quote|Mage_Core_Model_Abstract
 * @psalm-type OrderOrQuoteOrItem = Mage_Sales_Model_Order_Item|Mage_Sales_Model_Quote_Item|OrderOrQuote
 */
class Awardit_Antifraud_Helper_Data extends Mage_Core_Helper_Abstract
{
    /** @const string */
    private const LOG_CHANNEL = 'antifraud.log';


    // ----- Notifications ----------------------------------------------------------- //

    /**
     * Send email to admins when antifraud has evaluated Order or Quote.
     * @param OrderOrQuote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertEmail(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        if (Mage::getStoreConfigFlag('system/smtp/disable')) {
            return; // Mail disabled
        }
        preg_match_all(
            '/([\w.-_]{1,}@[\w.-_]{1,}\.[\w]{2,})/',
            Mage::getStoreConfig('awardit_antifraud/alerts/alert_email', $model->getStore()),
            $matches
        );
        $receivers = array_shift($matches);
        if (empty($receivers)) {
            return; // No-one to send to
        }
        $sender = Mage::getStoreConfig('awardit_antifraud/alerts/alert_context', $model->getStore())
            ?: $this->__('Magento Antifraud module');

        $modes = $this->getModeOptions();
        $body = array_merge([
            $this->__('An order has been evaluated for fraud with result: %s.', $modes[$result->getMode()]),
            '',
            $this->__('Check result:'),
        ], $result->getDescriptionList('• '), [
            '',
            $this->__('Summary:'),
        ], $this->getSaleInfo($model));

        $mail = new Zend_Mail('UTF-8');
        $mail->setBodyText(implode("\r\n", $body))
            ->setFrom('noreply@awardit.com', $sender)
            ->setSubject($this->__('Antifraud alert'));
        foreach ($receivers as $receiver) {
            $mail->addTo($receiver);
        }
        try {
            $mail->send();
            $this->log('Sent email to ' . implode(', ', $receivers) . '.');
        } catch (Exception $e) {
            Mage::logException($e); // Log but don't break
        }
    }

    /**
     * Send to external message service when antifraud has evaluated Order or Quote.
     * @param OrderOrQuote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertMessageService(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        // Use Slack
        // $this->alertSlack($model, $result);

        // Use Teams
        $this->alertTeams($model, $result);
    }

    /**
     * Send to Teams channel.
     * @param OrderOrQuote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertTeams(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        $url = Mage::getStoreConfig('awardit_antifraud/alerts/alert_teams', $model->getStore());

        // If no url set, skip
        if (empty($url)) {
            return;
        }

        $ch = curl_init();

        // Create adaptive card structure
        $card = [
            "type" => "message",
            "attachments" => [
                [
                    "contentType" => "application/vnd.microsoft.card.adaptive",
                    "contentUrl"  => null,
                    "content"     => [
                        "type"    => "AdaptiveCard",
                        '$schema' => "http://adaptivecards.io/schemas/adaptive-card.json",
                        "version" => "1.5",
                        "body"    => [
                            [
                                "type"  => "Container",
                                "style" => "attention",
                                "bleed" => true,
                                "items" => [
                                    [
                                      "type"   => "TextBlock",
                                      "size"   => "Medium",
                                      "text"   => "Anti Fraud System: Warning",
                                      "wrap"   => true,
                                      "weight" => "Bolder"
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ]
        ];

        // Create message body (copied from Slack alert)
        $modes = $this->getModeOptions();
        $card["attachments"][0]["content"]["body"][] = [
            "type" => "TextBlock",
            "text" =>
                $this->__(
                    'An order has been evaluated for fraud with result: **%s**.',
                    $modes[$result->getMode()]
                ),
            "size" => "Medium",
            "wrap" => true
        ];

        $descriptionList = $result->getDescriptionList("", true);
        if (!empty($descriptionList)) {
            $factList1 = [
                "type" => "FactSet",
                "separator" => true,
                "facts" => []
            ];
            foreach ($descriptionList as $value) {
                $factList1["facts"][] = [
                    "title" => "•",
                    "value" => $value,
                ];
            }
            $card["attachments"][0]["content"]["body"][] = $factList1;
        }

        $saleInfo = $this->getSaleInfo($model);
        if (!empty($saleInfo)) {
            $factList2 = [
                "type" => "FactSet",
                "separator" => true,
                "facts" => []
            ];

            $sender = Mage::getStoreConfig("awardit_antifraud/alerts/alert_context", $model->getStore())
            ?: $this->__('Magento Antifraud module');
            $factList2["facts"][] = [
                "title" => "Sender:",
                "value" => $sender,
            ];

            foreach ($saleInfo as $factRow) {
                $values = explode(": ", $factRow);
                $factList2["facts"][] = [
                    "title" => !empty($values[0]) ? "{$values[0]}:" : "&ndash;",
                    "value" => !empty($values[1]) ? $values[1] : "&ndash;",
                ];
            }
            $card["attachments"][0]["content"]["body"][] = $factList2;
        }

        // Add order link.
        // We get increment id from order and sometimes as reserved order id from quote.
        $incrementId = (string) $model->getData("increment_id") ?: (string) $model->getData("reserved_order_id");
        if (!empty($incrementId)) {
            $orderURL = null;
            $prefixes = [
                "M6-"   => "magento6.crossroads.se",
                "C-"    => "magento7.crossroads.se",
                "M8-"   => "magento8.crossroads.se",
                "J-"    => "magento8.crossroads.se",
                "M9-"   => "magento9.crossroads.se",
                "A-"    => "magento10.crossroads.se",
                "M10A-" => "magento10a.crossroads.se",
                "M11-"  => "magento11.crossroads.se",
                "S-"    => "sas-next-admin.crossroads.se",
            ];
            foreach ($prefixes as $key => $val) {
                if (stripos($incrementId, $key) === 0) {
                    $orderURL = "https://{$val}/magento-index.php/administration/sales_order/view/order_id/{$model->getId()}/"; // phpcs:ignore
                    break;
                }
            }
            if (!empty($orderURL)) {
                $card["attachments"][0]["content"]["actions"] = [
                    [
                        "type"  => "Action.OpenUrl",
                        "title" => "View order",
                        "url"   => $orderURL
                    ]
                ];
            }
        }

        $header = [ "Content-type: application/json" ];

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($card));
        curl_setopt($ch, CURLOPT_HEADER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);

        try {
            curl_exec($ch);
        } catch (Exception $e) {
            Mage::logException($e); // Log but don't break
        }

        curl_close($ch);
    }

    /**
     * Send to Slack channel.
     * @param OrderOrQuote $model Instance to alert for.
     * @param Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function alertSlack(Mage_Core_Model_Abstract $model, Awardit_Antifraud_Model_Result $result): void
    {
        $conversation = Mage::getStoreConfig('awardit_antifraud/alerts/alert_slack', $model->getStore());
        if (empty($conversation)) {
            return; // No-one to send to
        }
        $sender = Mage::getStoreConfig('awardit_antifraud/alerts/alert_context', $model->getStore())
            ?: $this->__('Magento Antifraud module');

        $modes = $this->getModeOptions();
        $body = array_merge([
            $this->__('An order has been evaluated for fraud with result: *%s*.', $modes[$result->getMode()]),
            '',
            $this->__('*Check result:*'),
        ], $result->getDescriptionList('• '), [
            '',
            $this->__('*Summary:*'),
        ], $this->getSaleInfo($model));

        $slack = Mage::helper('awardit_slack');
        try {
            $slack->postInChannel($conversation, $this->__('Antifraud alert'), [
                [
                    'type' => 'header',
                    'text' => [
                        'type' => 'plain_text',
                        'text' => $this->__('Antifraud alert'),
                    ]
                ],
                [
                    'type' => 'section',
                    'text' => [
                        'type' => 'mrkdwn',
                        'text' => implode("\n", $body),
                    ]
                ],
                [
                    'type' => 'context',
                    'elements' => [
                        [
                            'type' => 'mrkdwn',
                            'text' => $sender,
                        ],
                    ]
                ],
            ]);
            $this->log("Sent Slack to {$conversation}.");
        } catch (Exception $e) {
            Mage::logException($e); // Log but don't break
        }
    }

    /**
     * Write to log.
     * @param string $message Message to log.
     * @param 0|1|2|3|4|5|6|7|null $level Any Zend_Log level constant.
     */
    public function log(string $message, ?int $level = Zend_Log::DEBUG): void
    {
        Mage::log($message, $level, self::LOG_CHANNEL);
    }

    /**
     * Message to use when checkout is denied.
     * @param Mage_Core_Model_Store|null $store Store to use.
     * @return string Deny message.
     */
    public function getDenyMessage(Mage_Core_Model_Store $store = null): string
    {
        $message = Mage::getStoreConfig('awardit_antifraud/general/deny_message', $store);
        return $message ?: $this->__('Order can not be placed.');
    }

    /**
     * Get sale summary for Quote/Order.
     * @param OrderOrQuote $model Instance to get info on.
     * @return array<string> Sale info.
     */
    private function getSaleInfo(Mage_Core_Model_Abstract $model): array
    {
        if ($address = $model->getShippingAddress()) {
            $name = $address->getName();
        } elseif ($address = $model->getBillingAddress()) {
            $name = $address->getName();
        } elseif ($model instanceof Mage_Sales_Model_Order) {
            $name = $model->getCustomerName();
        } else {
            $name = '-';
        }
        $payment = $model->getPayment();
        $method = $payment ? $payment->getMethodInstance()->getTitle() : '-';
        return [
            $this->__('Store: %s', $model->getStore()->getName()),
            $model instanceof Mage_Sales_Model_Order
                ? $this->__('Order: %s (%s)', $model->getIncrementId(), $model->getStatusLabel())
                : $this->__('Quote: %s', $model->getId()),
            $this->__('IP-address: %s', $this->resolveIpAddr($model)),
            $this->__('Customer: %s %s', $name, $model->getCustomerEmail()),
            $this->__('Payment: %s', $method),
        ];
    }


    // ----- Verifiers --------------------------------------------------------------- //

    /**
     * If antifraud evaluation should be applied.
     * @param Mage_Core_Model_Store|null $store Store to use.
     * @return bool If enabled.
     */
    public function isEnabled(Mage_Core_Model_Store $store = null): bool
    {
        return (bool)Mage::getStoreConfig('awardit_antifraud/general/enabled', $store);
    }

    /**
     * Verifies that used currencies have defined currency rate to base currency.
     * @return bool True if rates are correctly defined.
     */
    public function hasCurrencyRates(): bool
    {
        $currency = Mage::getModel('directory/currency');
        $base_currency = Mage::app()->getBaseCurrencyCode();
        $used_currencies = $currency->getConfigAllowCurrencies();
        $currency_rates = $currency->getCurrencyRates($base_currency, $used_currencies);
        return count($currency_rates) >= count($used_currencies);
    }


    // ----- Status handlers --------------------------------------------------------- //

    /**
     * Get status to set for held orders.
     * @param Mage_Core_Model_Store|null $store Store to use.
     * @return string Status.
     */
    public function getHoldStatus(Mage_Core_Model_Store $store = null): string
    {
        $key = Mage::getStoreConfig('awardit_antifraud/general/order_status', $store);
        $status = Mage::getModel('sales/order_status')->load($key);
        if ($status->getId()) {
            return $status->getStatus();
        }
        return Mage::getSingleton('sales/order_config')
            ->getStateDefaultStatus(Mage_Sales_Model_Order::STATE_HOLDED);
    }

    /**
     * Set hold status on order.
     * @param Mage_Sales_Model_Order $order Order to hold.
     */
    public function setHoldStatus(Mage_Sales_Model_Order $order): void
    {
        $order->setData('hold_before_state', $order->getState());
        $order->setData('hold_before_status', $order->getStatus());
        $order->setData('state', Mage_Sales_Model_Order::STATE_HOLDED);
        $order->setData('status', $this->getHoldStatus($order->getStore()));
    }

    /**
     * If order can be held.
     * @param Mage_Sales_Model_Order $order Order to check.
     * @return bool If holdable.
     */
    public function isHoldable(Mage_Sales_Model_Order $order): bool
    {
        $payment = $order->getPayment();
        $method = $payment ? $payment->getMethod() : 'none';
        return in_array($order->getState(), [
            Mage_Sales_Model_Order::STATE_COMPLETE,
            Mage_Sales_Model_Order::STATE_NEW,
            Mage_Sales_Model_Order::STATE_PROCESSING,
        ]) && !in_array($method, [
            // Not all payment methods allow hold, list unsupported methods here
            'Crossroads_CollectorCheckout', // Walley
        ]);
    }

    /**
     * If order hold should be evaluated.
     * @param Mage_Sales_Model_Order $order Order to check.
     * @return bool If it should evaluate hold.
     */
    public function changedToHoldable(Mage_Sales_Model_Order $order): bool
    {
        if ($order->getState() == $order->getOrigData('state')) {
            return false; // No change
        }
        if (!$this->isHoldable($order)) {
            return false; // Current state not holdable
        }
        if (
            in_array($order->getOrigData('state'), [
                Mage_Sales_Model_Order::STATE_CANCELED,
                Mage_Sales_Model_Order::STATE_CLOSED,
                Mage_Sales_Model_Order::STATE_COMPLETE,
                Mage_Sales_Model_Order::STATE_HOLDED,
            ])
        ) {
            return false; // If previously in this state, don't change
        }

        $hold_status = $this->getHoldStatus($order->getStore());
        $history = array_filter($order->getAllStatusHistory(), function ($historial) use ($hold_status) {
            return $historial->getStatus() == $hold_status;
        });

        return empty($history); // Has been held before
    }


    // ----- Resolvers --------------------------------------------------------------- //

    /**
     * Resolve IP-adddress.
     * @param OrderOrQuote $model Resolve source.
     * @return string IP-adddress.
     */
    public function resolveIpAddr(Mage_Core_Model_Abstract $model): string
    {
        $ip = $model->getData('x_forwarded_for') ?? $model->getData('remote_ip') ?? '127.0.0.1';
        if (empty($ip)) {
            return '';
        }

        // Can contain comma separated list of IP-addresses.
        $ips = array_filter(array_map('trim', explode(',', (string)$ip)));

        return array_pop($ips);
    }

    /**
     * Resolve Email user as hash.
     * @param OrderOrQuote $model Resolve source.
     * @return string Email user hash.
     */
    public function resolveEmailUserHash(Mage_Core_Model_Abstract $model): string
    {
        $email = trim($model->getCustomerEmail());
        if (empty($email)) {
            return hash('sha256', 'empty');
        }
        $parts = explode('@', trim($email));
        if (count($parts) < 2) {
            return hash('sha256', trim($email));
        }
        return hash('sha256', trim($parts[0]));
    }

    /**
     * Resolve Email domain.
     * @param OrderOrQuote $model Resolve source.
     * @return string Email domain.
     */
    public function resolveEmailDomain(Mage_Core_Model_Abstract $model): string
    {
        $email = trim($model->getCustomerEmail());
        if (empty($email)) {
            return 'faulty';
        }
        $parts = explode('@', trim($email));
        if (count($parts) < 2) {
            return 'faulty';
        }
        return trim($parts[1]);
    }

    /**
     * Resolve quantity.
     * @param Mage_Sales_Model_Order_Item|Mage_Sales_Model_Quote_Item|OrderOrQuote $model Resolve source.
     * @return float Quantity.
     */
    public function resolveQty(Mage_Core_Model_Abstract $model): float
    {
        switch (get_class($model)) {
            case Mage_Sales_Model_Order_Item::class:
                return (float)$model->getQtyOrdered();
            case Mage_Sales_Model_Quote_Item::class:
                return (float)$model->getQty();
            case Mage_Sales_Model_Order::class:
                return (float)$model->getTotalQtyOrdered();
            case Mage_Sales_Model_Quote::class:
                return (float)$model->getItemsSummaryQty();
            default:
                throw new TypeError(get_class($model) . ' is not a valid source model.');
        }
    }

    /**
     * Resolve value.
     * @param Mage_Sales_Model_Order_Item|Mage_Sales_Model_Quote_Item|OrderOrQuote $model Resolve source.
     * @return float Value.
     */
    public function resolveValue(Mage_Core_Model_Abstract $model, float $rate = 1): float
    {
        switch (get_class($model)) {
            case Mage_Sales_Model_Order_Item::class:
            case Mage_Sales_Model_Quote_Item::class:
                return round($model->getBaseRowTotalInclTax() * $rate, 4);
            case Mage_Sales_Model_Order::class:
                return round($model->getBaseSubtotalInclTax() * $rate, 4);
            case Mage_Sales_Model_Quote::class:
                return round($model->getBaseSubtotal() * $rate, 4);
            default:
                throw new TypeError(get_class($model) . ' is not a valid source model.');
        }
    }

    /**
     * Resolve money paid.
     * @param OrderOrQuoteOrItem $model Resolve source.
     * @return float Value.
     */
    public function resolveMoneyPaid(Mage_Core_Model_Abstract $model, float $rate = 1): float
    {
        switch (get_class($model)) {
            case Mage_Sales_Model_Order_Item::class:
                $order = $model->getOrder();
                $value = $this->resolveValue($model, $rate);
                $order_value = $this->resolveValue($order, $rate);
                $order_money = $this->resolveMoneyPaid($order, $rate);
                $paid = $order_value ? round($order_money / $order_value * $value, 4) : 0;
                return round($paid, 4);
            case Mage_Sales_Model_Order::class:
                $payment = $model->getPayment();
                return $payment ? round($payment->getBaseAmountOrdered() * $rate, 4) : 0;
            case Mage_Sales_Model_Quote::class:
            case Mage_Sales_Model_Quote_Item::class:
                return 0; // We don't know how customer will pay for this
            default:
                throw new TypeError(get_class($model) . ' is not a valid source model.');
        }
    }

    /**
     * Resolve risk recommendation.
     * @param OrderOrQuote $model Resolve source.
     * @return string Risk recommendation.
     */
    public function resolveRiskRecommendation(Mage_Core_Model_Abstract $model): string
    {
        $payment = $model->getPayment();
        if (!$payment) {
            return '';
        }
        return $payment->getAdditionalInformation('riskAnalysis.recommendation') ?? '';
    }

    /**
     * Resolve risk score.
     * @param OrderOrQuote $model Resolve source.
     * @return int Risk score.
     */
    public function resolveRiskScore(Mage_Core_Model_Abstract $model): int
    {
        $payment = $model->getPayment();
        if (!$payment) {
            return 0;
        }
        return intval($payment->getAdditionalInformation('riskAnalysis.riskScore'));
    }

    // ----- Options ----------------------------------------------------------------- //

    /**
     * Get Blacklist types.
     * @return array<array-key, string> Types.
     */
    public function getTypeOptions(): array
    {
        return [
            'email' => $this->__('Email'),
            'ip' => $this->__('IP-address'),
            'risk_recommendation' => $this->__('Risk recommendation'),
        ];
    }

    /**
     * Get Fraud rule group types.
     * @return array<array-key, string> Group types.
     */
    public function getGroupTypeOptions(): array
    {
        return [
            'email' => $this->__('Email'),
            'email_user_hash' => $this->__('Email username'),
            'email_domain' => $this->__('Email domain'),
            'ip' => $this->__('IP-address'),
        ];
    }

    /**
     * Get Fraud rule rule types.
     * @return array<array-key, string> Rule types.
     */
    public function getRuleTypeOptions(): array
    {
        $currency = Mage::app()->getBaseCurrencyCode();
        return [
            'order_count' => $this->__('Order count'),
            'order_value' => $this->__('Order value (%s)', $currency),
            'order_money' => $this->__('Order money paid (%s)', $currency),
            'item_count' => $this->__('Order item count'),
            'product_count' => $this->__('Product count'),
            'product_value' => $this->__('Product value (%s)', $currency),
            'product_money' => $this->__('Product money paid (%s)', $currency),
            'risk_score' => $this->__('Risk score'),
        ];
    }

    /**
     * Get Fraud rule intervals.
     * @return array<array-key, string> Intervals.
     */
    public function getGroupIntervalOptions(): array
    {
        return [
            0 => $this->__('Now'),
            1 => $this->__('1 hour'),
            2 => $this->__('2 hours'),
            4 => $this->__('4 hours'),
            12 => $this->__('12 hours'),
            24 => $this->__('1 day'),
            48 => $this->__('2 days'),
            168 => $this->__('1 week'),
            720 => $this->__('1 month'),
            8760 => $this->__('1 year'),
        ];
    }

    /**
     * Get modes.
     * @return array<array-key, string> Modes.
     */
    public function getModeOptions(): array
    {
        return [
            Awardit_Antifraud_ResolverInterface::ALLOW => $this->__('Allow'),
            Awardit_Antifraud_ResolverInterface::ALERT => $this->__('Alert'),
            Awardit_Antifraud_ResolverInterface::HOLD => $this->__('Hold'),
            Awardit_Antifraud_ResolverInterface::DENY => $this->__('Deny'),
        ];
    }

    /**
     * Get stores.
     * @return array<array-key, string> Stores.
     */
    public function getStoreOptions(): array
    {
        $stores = [0 => "{$this->__('Global')} (0)"];
        foreach (Mage::app()->getStores(false, true) as $store) {
            $stores[$store->getId()] = "{$store->getName()} ({$store->getId()})";
        }
        return $stores;
    }
}
