<?php

class Awardit_Antifraud_Model_Rule extends Mage_Core_Model_Abstract implements Awardit_Antifraud_ResolverInterface
{
    /** @var Awardit_Antifraud_Helper_Data */
    private $helper;

    /**
     * Initialize object.
     */
    protected function _construct(): void
    {
        $this->helper = Mage::helper('awardit_antifraud');
        $this->_init('awardit_antifraud/rule');
    }

    /**
     * If entry exists in database.
     * @return bool If entry exists in database.
     */
    public function exists(): bool
    {
        return !is_null($this->getId());
    }

    /**
     * Get mode.
     * @return string Mode of this instance.
     */
    public function getMode(): string
    {
        return (string)$this->getData('mode');
    }

    /**
     * If rule can use SKU filter.
     * @return bool
     */
    public function isSkuFilterable(): bool
    {
        return in_array($this->getRuleType(), ['product_count', 'product_value', 'product_money']);
    }

    /**
     * Get SKU filter if applicable.
     * @return array|null
     */
    private function getSkuFilter(): ?array
    {
        if (!$this->isSkuFilterable()) {
            return null;
        }
        return array_filter(array_map('trim', explode(',', $this->getData('sku_filter')))) ?: null;
    }

    /**
     * Resolve rules for Quote or Order instance.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @return Awardit_Antifraud_Model_Result Result accumulator.
     */
    public function getResolved(Mage_Core_Model_Abstract $model): Awardit_Antifraud_Model_Result
    {
        $collection = $this->getCollection();
        $collection->addFieldToFilter('store_id', ['in' => [(int)$model->getStoreId(), 0]]);
        $connection = $collection->getResource()->getReadConnection();
        $result = new Awardit_Antifraud_Model_Result();

        foreach ($collection as $rule) {
            $resolved = false;
            $select = new Zend_Db_Select($connection);
            $sku_filter = $rule->getSkuFilter();

            if ($store = $rule->getStoreId()) {
                $select->where("ao.store_id = {$store}");
            }
            $select->where("ao.created_at >= DATE_SUB(NOW(), INTERVAL {$rule->getGroupInterval()} HOUR)");
            if ($states = Mage::getStoreConfig('awardit_antifraud/general/order_states', $model->getStoreId())) {
                $select->where(sprintf("ao.order_state IN (%s)", $this->sqlIn(explode(',', $states))));
            }
            if ($model instanceof Mage_Sales_Model_Order) {
                $select->where("ao.order_id != {$model->getId()}"); // Exclude self
            }

            // Selects by group
            switch ($rule->getGroupType()) {
                case 'ip':
                    $ip = $this->helper->resolveIpAddr($model);
                    $select->where("ao.ip = '{$ip}'");
                    break;
                case 'email_user_hash':
                    $select->where("ao.email_user_hash = '{$this->helper->resolveEmailUserHash($model)}'");
                    break;
                case 'email_domain':
                    $select->where("ao.email_domain = '{$this->helper->resolveEmailDomain($model)}'");
                    break;
            }

            // Selects by rule type
            switch ($rule->getRuleType()) {
                case 'order_count':
                    $resolved = $this->resolveOrderCount($model, $select);
                    break;
                case 'order_value':
                    $resolved = $this->resolveOrderValue($model, $select);
                    break;
                case 'order_money':
                    $resolved = $this->resolveOrderMoney($model, $select);
                    break;
                case 'item_count':
                    $resolved = $this->resolveItemCount($model, $select);
                    break;
                case 'product_count':
                    $resolved = $this->resolveProductCount($model, $select, $sku_filter);
                    break;
                case 'product_value':
                    $resolved = $this->resolveProductValue($model, $select, $sku_filter);
                    break;
                case 'product_money':
                    $resolved = $this->resolveProductMoney($model, $select, $sku_filter);
                    break;
            }
            if ($resolved) {
                $result->append($rule);
            }
        }

        // Return result accumulation
        return $result;
    }

    /**
     * Resolve order count rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @return bool True if filter rule applies.
     */
    private function resolveOrderCount(Mage_Core_Model_Abstract $model, Zend_Db_Select $select): bool
    {
        if (1 >= $this->getRuleValue()) {
            return true; // Direct hit
        }

        $select->from(['ao' => 'awardit_antifraud_order'], [
            new Zend_Db_Expr('COUNT(ao.order_id) AS order_count'),
        ]);
        foreach ($select->query() as $row) {
            if ($row['order_count'] + 1 >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve order value rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @return bool True if filter rule applies.
     */
    private function resolveOrderValue(Mage_Core_Model_Abstract $model, Zend_Db_Select $select): bool
    {
        $rate = $model->getBaseToGlobalRate();
        $value = $this->helper->resolveValue($model, $rate);
        if ($value >= $this->getRuleValue()) {
            return true; // Direct hit
        }

        $select->from(['ao' => 'awardit_antifraud_order'], [
            new Zend_Db_Expr('SUM(ao.order_value) AS order_value'),
        ]);
        foreach ($select->query() as $row) {
            if ($row['order_value'] + $value >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve order money paid rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @return bool True if filter rule applies.
     */
    private function resolveOrderMoney(Mage_Core_Model_Abstract $model, Zend_Db_Select $select): bool
    {
        $rate = $model->getBaseToGlobalRate();
        $value = $this->helper->resolveMoneyPaid($model, $rate);
        if ($value >= $this->getRuleValue()) {
            return true; // Direct hit
        }

        $select->from(['ao' => 'awardit_antifraud_order'], [
            new Zend_Db_Expr('SUM(ao.money_paid) AS money_paid'),
        ]);
        foreach ($select->query() as $row) {
            if ($row['money_paid'] + $value >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve order item count rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @return bool True if filter rule applies.
     */
    private function resolveItemCount(Mage_Core_Model_Abstract $model, Zend_Db_Select $select): bool
    {
        $qty = $this->helper->resolveQty($model);
        if ($qty >= $this->getRuleValue()) {
            return true; // Direct hit
        }

        $select->from(['aoi' => 'awardit_antifraud_order_item'], [
            new Zend_Db_Expr('SUM(aoi.qty) AS qty'),
        ]);
        $select->join(['ao' => 'awardit_antifraud_order'], 'ao.order_id=aoi.order_id', '');
        foreach ($select->query() as $row) {
            if ($row['qty'] + $qty >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve product count rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @param array|null $sku_filter SKU filter
     * @return bool True if filter rule applies.
     */
    private function resolveProductCount(Mage_Core_Model_Abstract $model, Zend_Db_Select $select, ?array $sku_filter = null): bool
    {
        $skus = $this->filterSkus($model, $sku_filter, function ($item) {
            return $this->helper->resolveQty($item);
        });
        $eval = $this->evalDirect($skus, $this);
        if (!is_null($eval)) {
            return $eval; // Direct result, no furher evaluation needed
        }

        $select->from(['aoi' => 'awardit_antifraud_order_item'], [
            new Zend_Db_Expr('aoi.sku AS sku'),
            new Zend_Db_Expr('SUM(aoi.qty) AS qty'),
        ]);
        $select->join(['ao' => 'awardit_antifraud_order'], 'ao.order_id=aoi.order_id', '');
        $select->where(sprintf('aoi.sku IN (%s)', $this->sqlIn(array_keys($skus))));
        $select->group('aoi.sku');
        foreach ($select->query() as $item_row) {
            if ($item_row['qty'] + $skus[$item_row['sku']] >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve product value rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @param array|null $sku_filter SKU filter
     * @return bool True if filter rule applies.
     */
    private function resolveProductValue(Mage_Core_Model_Abstract $model, Zend_Db_Select $select, ?array $sku_filter = null): bool
    {
        $rate = $model->getBaseToGlobalRate();
        $skus = $this->filterSkus($model, $sku_filter, function ($item) use ($rate) {
            return $this->helper->resolveValue($item, $rate);
        });
        $eval = $this->evalDirect($skus);
        if (!is_null($eval)) {
            return $eval; // Direct result, no furher evaluation needed
        }

        $select->from(['aoi' => 'awardit_antifraud_order_item'], [
            new Zend_Db_Expr('aoi.sku AS sku'),
            new Zend_Db_Expr('SUM(aoi.item_value) AS item_value'),
        ]);
        $select->join(['ao' => 'awardit_antifraud_order'], 'ao.order_id=aoi.order_id', '');
        $select->where(sprintf('aoi.sku IN (%s)', $this->sqlIn(array_keys($skus))));
        $select->group('aoi.sku');
        foreach ($select->query() as $item_row) {
            if ($item_row['item_value'] + $skus[$item_row['sku']] >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Resolve product money paid rule.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param Zend_Db_Select $select Select to use.
     * @param array|null $sku_filter SKU filter
     * @return bool True if filter rule applies.
     */
    private function resolveProductMoney(Mage_Core_Model_Abstract $model, Zend_Db_Select $select, ?array $sku_filter = null): bool
    {
        $rate = $model->getBaseToGlobalRate();
        $skus = $this->filterSkus($model, $sku_filter, function ($item) use ($rate) {
            return $this->helper->resolveMoneyPaid($item, $rate);
        });
        $eval = $this->evalDirect($skus);
        if (!is_null($eval)) {
            return $eval; // Direct result, no furher evaluation needed
        }

        $select->from(['aoi' => 'awardit_antifraud_order_item'], [
            new Zend_Db_Expr('aoi.sku AS sku'),
            new Zend_Db_Expr('SUM(aoi.money_paid) AS money_paid'),
        ]);
        $select->join(['ao' => 'awardit_antifraud_order'], 'ao.order_id=aoi.order_id', '');
        $select->where(sprintf('aoi.sku IN (%s)', $this->sqlIn(array_keys($skus))));
        $select->group('aoi.sku');

        foreach ($select->query() as $item_row) {
            if ($item_row['money_paid'] + $skus[$item_row['sku']] >= $this->getRuleValue()) {
                return true; // History hit, continue with Rule collection loop
            }
        }
        return false;
    }

    /**
     * Evaluate current quote/order.
     * @param array SKU list.
     * @return ?bool True if filter rule applies.
     */
    private function evalDirect(array $sku): ?bool
    {
        if (empty($sku)) {
            return false; // Nothing to evaluate
        }
        foreach ($sku as $value) {
            if ($value >= $this->getRuleValue()) {
                return true; // Direct hit
            }
        }
        return null;
    }

    /**
     * Evaluate which SKU to use, optionally using a filter.
     * @param Mage_Sales_Model_Order|Mage_Sales_Model_Quote $model Instance to resolve.
     * @param array|null Optional SKU filter.
     * @param callable $resolver Callback function to resolve value.
     * @return array<string,mixed> SKU:s to use.
     */
    private function filterSkus(
        Mage_Core_Model_Abstract $model,
        ?array $sku_filter,
        callable $resolver
    ): array
    {
        $skus = [];
        foreach ($model->getAllVisibleItems() as $item) {
            $sku = $item->getProduct()->getSku();
            if (is_array($sku_filter) && !in_array($sku, $sku_filter)) {
                continue; // Not interested
            }
            $skus[$sku] = $resolver($item);
        }
        return $skus;
    }

    /**
     * Format SQL list
     * @param array $list
     * @return string SQL code
     */
    private function sqlIn(array $list): string
    {
        return implode(',', array_map('json_encode', $list));
    }

    /**
     * Print friendly description.
     * @return string Rule description
     */
    public function describe(): string
    {
        $modes = $this->helper->getModeOptions();
        $rtypes = $this->helper->getRuleTypeOptions();
        $gtypes = $this->helper->getGroupTypeOptions();
        $gintervals = $this->helper->getGroupIntervalOptions();
        return "Fraud {$modes[$this->getMode()]}: {$rtypes[$this->getRuleType()]} ≥ {$this->getRuleValue()} for {$gtypes[$this->getGroupType()]} in {$gintervals[$this->getGroupInterval()]}";
    }
}
